/*
 ** Copyright (C) 2003-2006 Teus Benschop.
 **  
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **  
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **  
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **  
 */


#include "libraries.h"
#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "notes_utils.h"
#include "directories.h"
#include <sqlite3.h>
#include "sqlite_reader.h"
#include "bible.h"
#include "date_time_utils.h"
#include "gwrappers.h"
#include "shell.h"


#define TABLE_VERSION "version"
#define NOTES_DATABASE_VERSION 5


void notes_database_verify_update ()
// Updates the notes database to the current version.
{
  try
  {
    // Get the version number of the database.
    int version = 0;
    {
      // Connect to the database.
      sqlite3 *db;
      int rc;
      char *error = NULL;
      rc = sqlite3_open(notes_database_filename ().c_str (), &db);
      SqliteReader reader (0);
      // How many tables are there?
      int count = 0;
      char * sql;
      sql = g_strdup_printf ("select count(*) from sqlite_master;");
      rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
      g_free (sql);
      count = convert_to_int (reader.ustring0[0]);
      if (count == 1) {
        // If there is 1 table, then the version number is 1, because that
        // was the structure of the notes database at the time of version 1.
        version = 1;
      } else {
        // If there are more than one table, get the version number from the
        // database.
        SqliteReader reader2 (0);
        char * sql;
        sql = g_strdup_printf ("select version from '%s';", TABLE_VERSION);
        rc = sqlite3_exec(db, sql, reader2.callback, &reader2, &error);
        g_free (sql);
        if (reader2.ustring0.size() > 0) {
          version = convert_to_int (reader2.ustring0[0]);
        }
      }
      sqlite3_close (db);
      // Log database version.
      gw_message ("Notes database version " + convert_to_string (version));
    }
    
    // Update the database to the newest version. Using the fall-through
    // mechanism of the switch statement, the database is updated automatically
    // to the newest version, step by step.
    // Note that the ALTER TABLE command is not fully implemented in sqlite3,
    // and the command to add a row was implemented somewhere in version 3,
    // so we cannot rely on it.
    ustring original_database_name = notes_database_filename ();
    ustring temporal_database_name = original_database_name + "_temporal";
    switch (version) {
      case 0 : 
      {
      }
      case 1 :
      {
        // Upgrade to version 2.
        gw_message ("Upgrading notes database to version 2");
        
        GtkWidget *updatedatabasewindow;
        GtkWidget *vbox1;
        GtkWidget *label1;
        GtkWidget *progressbar1;
        updatedatabasewindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        gtk_window_set_position (GTK_WINDOW (updatedatabasewindow), GTK_WIN_POS_CENTER);
        gtk_window_set_default_size (GTK_WINDOW (updatedatabasewindow), 400, -1);
        gtk_window_set_type_hint (GTK_WINDOW (updatedatabasewindow), GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);
        vbox1 = gtk_vbox_new (FALSE, 0);
        gtk_widget_show (vbox1);
        gtk_container_add (GTK_CONTAINER (updatedatabasewindow), vbox1);
        label1 = gtk_label_new ("Bibledit is updating the notes to version 2");
        gtk_widget_show (label1);
        gtk_box_pack_start (GTK_BOX (vbox1), label1, FALSE, FALSE, 0);
        progressbar1 = gtk_progress_bar_new ();
        gtk_widget_show (progressbar1);
        gtk_box_pack_start (GTK_BOX (vbox1), progressbar1, FALSE, FALSE, 0);
        gtk_widget_show (updatedatabasewindow);
        int updated_records = 0;        
        // Read values from old notes database, store in memory.
        vector<int> id;
        vector<ustring> reference;
        vector<ustring> project;
        vector<int> referred;
        vector<ustring> note;
        vector<ustring> casefolded;
        vector<int> created;
        vector<int> modified;
        vector<ustring> user;
        {
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(original_database_name.c_str (), &db);
          SqliteReader reader (0);
          char * sql;
          sql = g_strdup_printf  ("select id, reference, project, referred, note, casefolded, created, modified, user from %s;", TABLE_NOTES);
          rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
          g_free (sql);
          for (unsigned int rc = 0; rc < reader.ustring0.size(); rc++) {
            updated_records++;
            id.push_back (convert_to_int (reader.ustring0[rc]));
            reference.push_back (reader.ustring1[rc]);
            project.push_back (double_apostrophy (reader.ustring2[rc]));
            referred.push_back (convert_to_int (reader.ustring3[rc]));
            note.push_back (double_apostrophy (reader.ustring4[rc]));
            casefolded.push_back (double_apostrophy (reader.ustring5[rc]));
            created.push_back (convert_to_int (reader.ustring6[rc]));
            modified.push_back (convert_to_int (reader.ustring7[rc]));
            user.push_back (reader.ustring8[rc]);
          }
          sqlite3_close (db);
        }
        {
          // Create temporal new database.
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(temporal_database_name.c_str (), &db);
          char * sql;
          sql = g_strdup_printf("create table '%s' (id integer, reference text, project text, referred integer, note text, casefolded text, created integer, modified integer, user text);", TABLE_NOTES);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          sql = g_strdup_printf("create table '%s' (version integer);", TABLE_VERSION);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          sql = g_strdup_printf("insert into '%s' values (%d);", TABLE_VERSION, 2);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          // Store all values in it.
          for (unsigned int i = 0; i < id.size(); i++) {
            double fraction;
            fraction = double (i) / double (updated_records);
            gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progressbar1), fraction);
            while (gtk_events_pending ()) gtk_main_iteration ();
            sql = g_strdup_printf("insert into %s values (%d, '%s', '%s', %d, '%s', '%s', %d, %d, '%s');", TABLE_NOTES, id[i], reference[i].c_str(), project[i].c_str(), referred[i], note[i].c_str(), casefolded[i].c_str(), created[i], modified[i], user[i].c_str());
            rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          }
          g_free (sql);
          sqlite3_close (db);
        }

        // Copy temporal database back to the database.
        unlink (original_database_name.c_str());
        ustring command;
        command = "cp" + shell_quote_space (temporal_database_name) + shell_quote_space (original_database_name);
        system (command.c_str());
        unlink (temporal_database_name.c_str());
        
        gtk_widget_destroy (updatedatabasewindow);

        // Log.
        gw_message ("Records updated: " + convert_to_string(updated_records));
      }
      case 2 :
      {
        // Upgrade to version 3.
        gw_message ("Upgrading notes database to version 3");
        
        GtkWidget *updatedatabasewindow;
        GtkWidget *vbox1;
        GtkWidget *label1;
        GtkWidget *progressbar1;
        updatedatabasewindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        gtk_window_set_position (GTK_WINDOW (updatedatabasewindow), GTK_WIN_POS_CENTER);
        gtk_window_set_default_size (GTK_WINDOW (updatedatabasewindow), 400, -1);
        gtk_window_set_type_hint (GTK_WINDOW (updatedatabasewindow), GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);
        vbox1 = gtk_vbox_new (FALSE, 0);
        gtk_widget_show (vbox1);
        gtk_container_add (GTK_CONTAINER (updatedatabasewindow), vbox1);
        label1 = gtk_label_new ("Bibledit is updating the notes to version 3");
        gtk_widget_show (label1);
        gtk_box_pack_start (GTK_BOX (vbox1), label1, FALSE, FALSE, 0);
        progressbar1 = gtk_progress_bar_new ();
        gtk_widget_show (progressbar1);
        gtk_box_pack_start (GTK_BOX (vbox1), progressbar1, FALSE, FALSE, 0);
        gtk_widget_show (updatedatabasewindow);
        int updated_records = 0;        
        // Read values from old notes database, store in memory.
        vector<int> id;
        vector<ustring> reference;
        vector<ustring> project;
        vector<int> referred;
        vector<ustring> note;
        vector<ustring> casefolded;
        vector<int> created;
        vector<int> modified;
        vector<ustring> user;
        {
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(original_database_name.c_str (), &db);
          SqliteReader reader (0);
          char * sql;
          sql = g_strdup_printf("select id, reference, project, referred, note, casefolded, created, modified, user from %s;", TABLE_NOTES);
          rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
          g_free (sql);
          for (unsigned int rc = 0; rc < reader.ustring0.size(); rc++) {
            updated_records++;
            id.push_back (convert_to_int (reader.ustring0[rc]));
            reference.push_back (reader.ustring1[rc]);
            project.push_back (double_apostrophy (reader.ustring2[rc]));
            referred.push_back (convert_to_int (reader.ustring3[rc]));
            note.push_back (double_apostrophy (reader.ustring4[rc]));
            casefolded.push_back (double_apostrophy (reader.ustring5[rc]));
            created.push_back (convert_to_int (reader.ustring6[rc]));
            modified.push_back (convert_to_int (reader.ustring7[rc]));
            user.push_back (reader.ustring8[rc]);
          }
          sqlite3_close (db);
        }
        {
          // Create temporal new database.
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(temporal_database_name.c_str (), &db);
          char * sql;
          sql = g_strdup_printf("create table '%s' (id integer, reference text, project text, category text, note text, casefolded text, created integer, modified integer, user text);", TABLE_NOTES);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          sql = g_strdup_printf("create table '%s' (version integer);", TABLE_VERSION);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          sql = g_strdup_printf("insert into '%s' values (%d);", TABLE_VERSION, 3);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          // Store all values in it.
          for (unsigned int i = 0; i < id.size(); i++) {
            double fraction;
            fraction = double (i) / double (updated_records);
            gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progressbar1), fraction);
            while (gtk_events_pending ()) gtk_main_iteration ();
            ustring category;
            switch (referred[i]) {
              case 1:  { category = "For myself"; break; }
              case 2:  { category = "For subteam"; break; }
              case 3:  { category = "For team"; break; }
              case 4:  { category = "For panel"; break; }
              case 5:  { category = "For church"; break; }
              case 6:  { category = "For consultant"; break; }
              case 7:  { category = "For Bible society"; break; }
              default: { category = "No issue"; break; }
            }
            sql = g_strdup_printf("insert into %s values (%d, '%s', '%s', '%s', '%s', '%s', %d, %d, '%s');", TABLE_NOTES, id[i], reference[i].c_str(), project[i].c_str(), category.c_str(), note[i].c_str(), casefolded[i].c_str(), created[i], modified[i], user[i].c_str());
            rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          }
          g_free (sql);
          sqlite3_close (db);
        }

        // Copy temporal database back to the database.
        unlink (original_database_name.c_str());
        ustring command;
        command = "cp" + shell_quote_space (temporal_database_name) + shell_quote_space (original_database_name);
        system (command.c_str());
        unlink (temporal_database_name.c_str());
        
        gtk_widget_destroy (updatedatabasewindow);

        // Log.
        gw_message ("Records updated: " + convert_to_string(updated_records));
      }
      case 3 :
      {
        // Upgrade to version 4.
        gw_message ("Upgrading notes database to version 4");
        
        GtkWidget *updatedatabasewindow;
        GtkWidget *vbox1;
        GtkWidget *label1;
        GtkWidget *progressbar1;
        updatedatabasewindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        gtk_window_set_position (GTK_WINDOW (updatedatabasewindow), GTK_WIN_POS_CENTER);
        gtk_window_set_default_size (GTK_WINDOW (updatedatabasewindow), 400, -1);
        gtk_window_set_type_hint (GTK_WINDOW (updatedatabasewindow), GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);
        vbox1 = gtk_vbox_new (FALSE, 0);
        gtk_widget_show (vbox1);
        gtk_container_add (GTK_CONTAINER (updatedatabasewindow), vbox1);
        label1 = gtk_label_new ("Bibledit is updating the notes to version 4");
        gtk_widget_show (label1);
        gtk_box_pack_start (GTK_BOX (vbox1), label1, FALSE, FALSE, 0);
        progressbar1 = gtk_progress_bar_new ();
        gtk_widget_show (progressbar1);
        gtk_box_pack_start (GTK_BOX (vbox1), progressbar1, FALSE, FALSE, 0);
        gtk_widget_show (updatedatabasewindow);
        int updated_records = 0;        
        // Read values from old notes database, store in memory.
        vector<int> id;
        vector<ustring> reference;
        vector<ustring> project;
        vector<ustring> category;
        vector<ustring> note;
        vector<ustring> casefolded;
        vector<int> created;
        vector<int> modified;
        vector<ustring> user;
        {
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(original_database_name.c_str (), &db);
          SqliteReader reader (0);
          char * sql;
          sql = g_strdup_printf("select id, reference, project, category, note, casefolded, created, modified, user from %s;", TABLE_NOTES);
          rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
          g_free (sql);
          for (unsigned int rc = 0; rc < reader.ustring0.size(); rc++) {
            updated_records++;
            id.push_back (convert_to_int (reader.ustring0[rc]));
            reference.push_back (reader.ustring1[rc]);
            project.push_back (double_apostrophy (reader.ustring2[rc]));
            category.push_back (reader.ustring3[rc]);
            note.push_back (double_apostrophy (reader.ustring4[rc]));
            casefolded.push_back (double_apostrophy (reader.ustring5[rc]));
            created.push_back (convert_to_int (reader.ustring6[rc]));
            modified.push_back (convert_to_int (reader.ustring7[rc]));
            user.push_back (reader.ustring8[rc]);
          }
          sqlite3_close (db);
        }
        {
          // Create temporal new database.
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(temporal_database_name.c_str (), &db);
          char * sql;
          sql = g_strdup_printf("create table '%s' (id integer, reference text, ref_osis text, project text, category text, note text, casefolded text, created integer, modified integer, user text);", TABLE_NOTES);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          sql = g_strdup_printf("create table '%s' (version integer);", TABLE_VERSION);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          // Store new version number.
          sql = g_strdup_printf("insert into '%s' values (%d);", TABLE_VERSION, 4);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          // Store all values in it.
          for (unsigned int i = 0; i < id.size(); i++) {
            if ((i % 50) == 0) {
              double fraction;
              fraction = double (i) / double (updated_records);
              gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progressbar1), fraction);
              while (gtk_events_pending ()) gtk_main_iteration ();
            }
            // As the updating this time is about references, work on it here.
            // We are going to create one string that contains all references
            // this note refers to, and it is in OSIS format because this format
            // does not contain spaces, which enables us later to retrieve these
            // references again using word parsing routines.
            ustring osis_reference;
            // Also create a string with the encoded references, in the half-verse
            // format, e.g. verse 0 yields "0 1" and verse 1a yields "2", and
            // verse 2 yields "4 5". This includes numerical book- and chapter-
            // information too.
            ustring numerical_equivalent;
            // Parse current reference and deal with each one separately.
            Parse parse (reference[i]);
            for (unsigned int i2 = 0; i2 < parse.words.size(); i2++) {
              // Get book / chapter / verse.
              int book, chapter, verse;
              book = convert_to_int (parse.words[i2].substr(0, 3));
              chapter = convert_to_int (parse.words[i2].substr(3, 3));
              verse = convert_to_int (parse.words[i2].substr (6, 3));
              // Create OSIS reference.
              ustring ubook = index_to_osis_id (book);
              ustring reference = ubook + "." + convert_to_string (chapter) + "." + convert_to_string (verse);              
              if (!osis_reference.empty())
                osis_reference.append (" ");
              osis_reference.append (reference);
              // Create numerical equivalent.
              int int_num_equivalent;
              int_num_equivalent = book * 1000000;
              int_num_equivalent = int_num_equivalent + (chapter * 1000);
              vector<int> verses = verses_encode (convert_to_string (verse));
              numerical_equivalent.append (" ");
              numerical_equivalent.append (convert_to_string (int_num_equivalent + verses[0]));
              numerical_equivalent.append (" ");
              numerical_equivalent.append (convert_to_string (int_num_equivalent + verses[1]));              
            }
            // For matching functionality in notes selection, surround the reference
            // with a space.
            numerical_equivalent.append (" ");
            sql = g_strdup_printf("insert into %s values (%d, '%s', '%s', '%s', '%s', '%s', '%s', %d, %d, '%s');", TABLE_NOTES, id[i], numerical_equivalent.c_str(), osis_reference.c_str(), project[i].c_str(), category[i].c_str(), note[i].c_str(), casefolded[i].c_str(), created[i], modified[i], user[i].c_str());
            rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          }
          g_free (sql);
          sqlite3_close (db);
        }

        // Copy temporal database back to the database.
        unlink (original_database_name.c_str());
        ustring command;
        command = "cp" + shell_quote_space (temporal_database_name) + shell_quote_space (original_database_name);
        system (command.c_str());
        unlink (temporal_database_name.c_str());
        
        gtk_widget_destroy (updatedatabasewindow);

        // Log.
        gw_message ("Records updated: " + convert_to_string(updated_records));
      }
      case 4 :
      {
        // Upgrade to version 5.
        gw_message ("Upgrading notes database to version 5");
        
        GtkWidget *updatedatabasewindow;
        GtkWidget *vbox1;
        GtkWidget *label1;
        GtkWidget *progressbar1;
        updatedatabasewindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        gtk_window_set_position (GTK_WINDOW (updatedatabasewindow), GTK_WIN_POS_CENTER);
        gtk_window_set_default_size (GTK_WINDOW (updatedatabasewindow), 400, -1);
        gtk_window_set_type_hint (GTK_WINDOW (updatedatabasewindow), GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);
        vbox1 = gtk_vbox_new (FALSE, 0);
        gtk_widget_show (vbox1);
        gtk_container_add (GTK_CONTAINER (updatedatabasewindow), vbox1);
        label1 = gtk_label_new ("Bibledit is updating the notes to version 5");
        gtk_widget_show (label1);
        gtk_box_pack_start (GTK_BOX (vbox1), label1, FALSE, FALSE, 0);
        progressbar1 = gtk_progress_bar_new ();
        gtk_widget_show (progressbar1);
        gtk_box_pack_start (GTK_BOX (vbox1), progressbar1, FALSE, FALSE, 0);
        gtk_widget_show (updatedatabasewindow);
        int updated_records = 0;        
        // Read values from old notes database, store in memory. 
        vector<int> id;
        vector<ustring> reference;
        vector<ustring> ref_osis;
        vector<ustring> project;
        vector<ustring> category;
        vector<ustring> note;
        vector<ustring> casefolded;
        vector<int> created;
        vector<int> modified;
        vector<ustring> user;
        {
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(original_database_name.c_str (), &db);
          SqliteReader reader (0);
          char * sql;
          sql = g_strdup_printf("select id, reference, ref_osis, project, category, note, casefolded, created, modified, user from %s;", TABLE_NOTES);
          rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
          g_free (sql);
          for (unsigned int rc = 0; rc < reader.ustring0.size(); rc++) {
            updated_records++;
            id.push_back (convert_to_int (reader.ustring0[rc]));
            reference.push_back (reader.ustring1[rc]);
            ref_osis.push_back (reader.ustring2[rc]);
            project.push_back (double_apostrophy (reader.ustring3[rc]));
            category.push_back (reader.ustring4[rc]);
            note.push_back (double_apostrophy (reader.ustring5[rc]));
            casefolded.push_back (double_apostrophy (reader.ustring6[rc]));
            created.push_back (convert_to_int (reader.ustring7[rc]));
            modified.push_back (convert_to_int (reader.ustring8[rc]));
            user.push_back (reader.ustring9[rc]);
          }
          sqlite3_close (db);
        }
        {
          // Create temporal new database.
          sqlite3 *db;
          int rc;
          char *error = NULL;
          rc = sqlite3_open(temporal_database_name.c_str (), &db);
          char * sql;
          sql = g_strdup_printf("create table '%s' (id integer, reference text, ref_osis text, project text, category text, note text, casefolded text, created integer, modified integer, user text, logbook text);", TABLE_NOTES);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          sql = g_strdup_printf("create table '%s' (version integer);", TABLE_VERSION);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          // Store new version number.
          sql = g_strdup_printf("insert into '%s' values (%d);", TABLE_VERSION, 5);
          rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          // Store all values in it.
          for (unsigned int i = 0; i < id.size(); i++) {
            if ((i % 50) == 0) {
              double fraction;
              fraction = double (i) / double (updated_records);
              gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progressbar1), fraction);
              while (gtk_events_pending ()) gtk_main_iteration ();
            }
            sql = g_strdup_printf("insert into %s values (%d, '%s', '%s', '%s', '%s', '%s', '%s', %d, %d, '%s', '%s');", TABLE_NOTES, id[i], reference[i].c_str(), ref_osis[i].c_str(), project[i].c_str(), category[i].c_str(), note[i].c_str(), casefolded[i].c_str(), created[i], modified[i], user[i].c_str(), "");
            rc = sqlite3_exec (db, sql, NULL, NULL, &error);
          }
          g_free (sql);
          sqlite3_close (db);
        }

        // Copy temporal database back to the database.
        unlink (original_database_name.c_str());
        ustring command;
        command = "cp" + shell_quote_space (temporal_database_name) + shell_quote_space (original_database_name);
        system (command.c_str());
        unlink (temporal_database_name.c_str());
        
        gtk_widget_destroy (updatedatabasewindow);

        // Log.
        gw_message ("Records updated: " + convert_to_string(updated_records));
      }
      case 5 :
      {
        // Up-to-date.
        gw_message ("Notes database is up to date");
      }
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


void notes_database_verify_create ()
/*
This verifies the notes database is fine.
If it's not there, it creates one.
*/
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database.
    rc = sqlite3_open(notes_database_filename ().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    int count;
    // Check for the table.
    SqliteReader reader (0);
    char * sql;
    sql = g_strdup_printf ("select count(*) from sqlite_master where name='%s';", TABLE_NOTES);
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    count = convert_to_int (reader.ustring0[0]);
    if (count == 0) {
      // Create the notes table.
      char * sql;
      sql = g_strdup_printf("create table '%s' (id integer, reference text, ref_osis text, project text, category text, note text, casefolded text, created integer, modified integer, user text, logbook text);", TABLE_NOTES);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      // Create the version table and store version number.
      sql = g_strdup_printf("create table '%s' (version integer);", TABLE_VERSION);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      sql = g_strdup_printf("insert into '%s' values (%d);", TABLE_VERSION, NOTES_DATABASE_VERSION);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.  
  sqlite3_close (db);
  notes_database_verify_update ();
}


ustring notes_database_filename ()
// Returns the filename of the notes database.
{
  return gw_build_filename (directories_get_notes (), "database");
}


void insert_link (GtkTextBuffer *buffer, ustring text, gint id)
/* Inserts a piece of text into the buffer, giving it the usual
 * appearance of a hyperlink in a web browser: blue and underlined.
 * Additionally, attaches some data on the tag, to make it recognizable
 * as a link. 
 */
{
  GtkTextTag *tag;
  tag = gtk_text_buffer_create_tag (buffer, NULL, 
				    "foreground", "blue", 
				    "underline", PANGO_UNDERLINE_SINGLE, 
				    NULL);
  g_object_set_data (G_OBJECT (tag), "id", GINT_TO_POINTER (id));
  GtkTextIter iter;
  gtk_text_buffer_get_end_iter (buffer, &iter);
  gtk_text_buffer_insert_with_tags (buffer, &iter, text.c_str(), -1, tag, NULL);
}
  

gint notes_database_get_unique_id ()
// This generates a unique id, one that is not yet used in the notes database.
{
  gint32 result = 0;
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(notes_database_filename().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    bool found = false;
    while (!found) {
      result = g_random_int_range (1, 100000000);
      SqliteReader reader (0);
      char * sql;
      sql = g_strdup_printf ("select count(*) from '%s' where id=%i;", TABLE_NOTES, result);
      rc = sqlite3_exec (db, sql, reader.callback, &reader, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
      gint count = convert_to_int (reader.ustring0[0]);
      if (count == 0) {
        found = true;
      }
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
  return result;
}


void notes_delete_one (int id)
// Deletes the note with id.
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to database and start transaction
    rc = sqlite3_open(notes_database_filename ().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    rc = sqlite3_exec (db, "begin;", NULL, NULL, &error);
    // Delete data with "id".
    char * sql;
    sql = g_strdup_printf("delete from %s where id = %d;", TABLE_NOTES, id);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    // Commit the transaction and close connection.
    rc = sqlite3_exec (db, "commit;", NULL, NULL, &error);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
}


void notes_sort (vector<gint32>& ids, const vector<ustring>& refs, const vector<int>& dates)
/*
This sorts notes.
ids         - ID's of the notes.
              The ids will be modified and sorted, so that these can be used 
              to fetch data from the database in the right order.
refs        - References to sort on first.
dates       - Modification dates to sort on next.
*/
{
  // No sorting when less than two entries.
  if (ids.size() < 2)
    return;

  // The sorting is done by putting everything in one string, and then
  // sorting the vector of strings, and then taking out the ids again.

  // Storage for strings to sort
  vector<ustring> strings_to_sort;
  
  // Go through all data to sort.
  for (unsigned int i = 0; i < ids.size(); i++) {
  
    // Storage for string to build.
    ustring string_to_build;
    
    // First part of the string is the reference.
    // It does not always has the same length, so make it 9.
    string_to_build = "0000" + refs[i];
    string_to_build = string_to_build.substr (string_to_build.length() - 9, 9);

    // Next part to sort on is the date. This Julian day has a length of 6
    // characters at the time of programming, and will continue so until
    // Deo Volente the year 2739 A.D. Therefore there is no need to modify the
    // length of it.
    {
      ustring date;
      date = convert_to_string (dates[i]);
      string_to_build.append(date);
    }
    
    // Last part of the string to sort is the id.
    // Maximum id = 100000000, so maximum length of 9 characters.
    {
      ustring id;
      id = "000000000" + convert_to_string (ids[i]);
      id = id.substr(id.length() - 9, 9);
      string_to_build.append (id);
    }

    // Store built string
    strings_to_sort.push_back(string_to_build);    
  }

  // Sort the numerical equivalents.
  sort (strings_to_sort.begin(), strings_to_sort.end());
  
  // Extract the sorted ids from the sorted data.
  ids.clear();
  for (unsigned int i = 0; i < strings_to_sort.size(); i++) {
    ustring id = strings_to_sort[i].substr(15, 9);
    gint32 numerical_id = convert_to_int (id);
    ids.push_back(numerical_id);
  }
}


void notes_select (Configuration * configuration, vector<gint32>& ids, const ustring& currentreference)
/*
This selects notes for display.
It does this based on the criteria passed.
The resulting selection will be stored in ids.
*/
{
  // Clear ids.
  ids.clear();
  // Variables.
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to database.
    rc = sqlite3_open(notes_database_filename().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    // Read from database, and sort the results.
    SqliteReader sqlitereader (0);
    // See which notes to select.
    switch (configuration->show_notes_selector) {
      case snsCurrentVerse:
      {
        // This selects any notes which refer to the current verse.
        ustring book, chapter, verse;
        decode_reference (currentreference, book, chapter, verse);
        unsigned int verse_zero;
        verse_zero = reference_to_numerical_equivalent (book, chapter, "0");
        vector<int> verses = verses_encode (verse);
        for (unsigned int i = 0; i < verses.size(); i++) {
          ustring this_verse = convert_to_string (int (verse_zero + verses[i]));
          char * sql;
          sql = g_strdup_printf ("select id, reference, modified, project from '%s' where reference glob ('* %s *');", TABLE_NOTES, this_verse.c_str());
          rc = sqlite3_exec (db, sql, sqlitereader.callback, &sqlitereader, &error);
          g_free (sql);
          if (rc != SQLITE_OK) {
            throw runtime_error (error);
          }
        }
        break;
      }
      case snsToday:
      {
        int currentdate = date_time_julian_day_get_current ();
        char * sql;
        sql = g_strdup_printf ("select id, reference, modified, project from '%s' where modified = %d;", TABLE_NOTES, currentdate);
        rc = sqlite3_exec (db, sql, sqlitereader.callback, &sqlitereader, &error);
        g_free (sql);
        if (rc != SQLITE_OK) {
          throw runtime_error (error);
        }
        break;
      }
      case snsAll:
      {
        char * sql;
        sql = g_strdup_printf ("select id, reference, modified, project from '%s';", TABLE_NOTES);
        rc = sqlite3_exec (db, sql, sqlitereader.callback, &sqlitereader, &error);
        g_free (sql);
        if (rc != SQLITE_OK) {
          throw runtime_error (error);
        }
        break;
      }
      case snsCategory:
      {
        char * sql;
        sql = g_strdup_printf ("select id, reference, modified, project from '%s' where category = '%s';", TABLE_NOTES, configuration->show_notes_category.c_str());
        rc = sqlite3_exec (db, sql, sqlitereader.callback, &sqlitereader, &error);
        g_free (sql);
        if (rc != SQLITE_OK) {
          throw runtime_error (error);
        }
        break;
      }
      case snsDateRange:
      {
        char * sql;
        sql = g_strdup_printf ("select id, reference, modified, project from '%s' where modified  >= %d and modified <= %d;", TABLE_NOTES, configuration->show_notes_date_range_from, configuration->show_notes_date_range_to);
        rc = sqlite3_exec (db, sql, sqlitereader.callback, &sqlitereader, &error);
        g_free (sql);
        if (rc != SQLITE_OK) {
          throw runtime_error (error);
        }
        break;
      }
    }
    // Storage for sorting purposes.
    vector<ustring> references;
    vector<int> dates;
    set<gint32> already_stored_ids;
    // Read all resulting data from the db.
    for (unsigned int rc = 0; rc < sqlitereader.ustring0.size(); rc++) {
      // See whether we show notes for current project only.
      if (configuration->show_notes_for_current_project_only) {
        bool project_ok = false;
        ustring project_in_db = sqlitereader.ustring3[rc];
        if (project_in_db == configuration->project)
          project_ok = true;
        // Some old notes used "None". Handle that too.
        if (project_in_db == "None")
          project_ok = true;
        // Current notes can use "All".
        if (project_in_db == "All")
          project_ok = true;
        // Only show notes when the project is fine.
        if (!project_ok)
          continue;
      }
      // Get id.
      gint32 id = convert_to_int (sqlitereader.ustring0[rc]);
      // Get the numerical equivalent of the reference.
      ustring reference;
      reference = sqlitereader.ustring1[rc];
      {
        // Parse the string into its possible several references.
        Parse parse (reference);
        // Take the first refence available.
        if (parse.words.size() > 0)
          reference = parse.words[0];
      }
      // Get date modified.
      int date = convert_to_int (sqlitereader.ustring2[rc]);
      // Store data. 
      // As we now work with half-verses (10a, 10b), because of the way we select 
      // notes we might have repeating ids. Filter these out.
      if (already_stored_ids.find (id) == already_stored_ids.end()) {
        ids.push_back(id);
        already_stored_ids.insert (id);
        references.push_back(reference);
        dates.push_back(date);
      }
    }
    // Sort the notes.
    notes_sort (ids, references, dates);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close database
  sqlite3_close (db);
}


void notes_display_one (GtkWidget *textview, GtkTextBuffer *textbuffer, gint32 id, Configuration * configuration)
/*
This cares for the actual displaying of the notes.
It displays the note with an ID of id.
*/
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to database.
    rc = sqlite3_open(notes_database_filename().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }

    // Get from the database.
    SqliteReader reader (0);
    char * sql;
    sql = g_strdup_printf ("select id, ref_osis, note, project from '%s' where id = %d;", TABLE_NOTES, id);
    rc = sqlite3_exec (db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    
    // Go through the results (there should be only one anyway).
    for (unsigned int r = 0; r < reader.ustring0.size(); r++) {

      // Get the numerical equivalent of the reference.
      ustring reference;
      reference = reader.ustring1[r];

      // Parse the string into its possible several references.
      Parse parse (reference, false);
      reference.clear();

      // Go through each reference.
      for (unsigned int i2 = 0; i2 < parse.words.size(); i2++) {
        // Make it human readable.
        ustring book, chapter, verse;
        reference_discover ("", "", "", parse.words[i2], book, chapter, verse);
        if (!reference.empty())
          reference.append (", ");
        reference.append (book + " " + chapter + ":" + verse);
      }

      // Get project.
      ustring project = reader.ustring3[r];
      
      // Store position of cursor.
      gint linenumber, offset;
      {
        GtkTextIter cursorposition;
        GtkTextMark * textmark;
        textmark = gtk_text_buffer_get_insert (textbuffer);
        gtk_text_buffer_get_iter_at_mark (textbuffer, &cursorposition, textmark);
        linenumber = gtk_text_iter_get_line (&cursorposition);
        offset = gtk_text_iter_get_line_offset (&cursorposition);
      }

      // Move cursor to end of buffer.
      GtkTextIter endposition;
      gtk_text_buffer_get_end_iter (textbuffer, &endposition);
      gtk_text_buffer_place_cursor (textbuffer, &endposition);
      
      // Insert a link with this heading.
      ustring linkheading (reference);
      if (configuration->notes_show_project)
        linkheading.append (" " + project);
      insert_link (textbuffer, linkheading, id); 
      gtk_text_buffer_insert_at_cursor (textbuffer, "\n", -1);
        
      // Get the text of the note.
      ustring note;
      note = reader.ustring2[r];
      gtk_text_buffer_insert_at_cursor (textbuffer, note.c_str(), -1);
      gtk_text_buffer_insert_at_cursor (textbuffer, "\n", -1);
      
      // Move cursor to where it was before.
      GtkTextIter formerposition;
      gtk_text_buffer_get_iter_at_line_offset (textbuffer, &formerposition, linenumber, offset);
      gtk_text_buffer_place_cursor (textbuffer, &formerposition);
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.  
  sqlite3_close (db);
}


void notes_get_references_from_editor (GtkTextBuffer *textbuffer, vector<ustring>& references, vector<ustring>& messages)
/*
Gets all references from the notes editor.
Normalizes them.
Produces messages on trouble.
*/
{
  // Get all lines from the textbuffer.
  vector<ustring> lines;
  textbuffer_get_lines (textbuffer, lines);
  // When discovering a reference from a user's entry, use previous values,
  // so that it becomes quicker for a user to enter new references.
  // If Leviticus 10:11 is already there, and the user wishes to add verse
  // 12 also, he just enters 12 on a line, and that' it.
  ustring previousbook, previouschapter, previousverse;
  for (unsigned int i = 0; i < lines.size(); i++) {
    if (!lines[i].empty ()) {
      // Normalize reference.
      ustring book;
      ustring chapter;
      ustring verse;
      if (reference_discover (previousbook, previouschapter, previousverse, lines[i], book, chapter, verse)) {
        // Store reference.
        ustring ref = book + " " + chapter + ":" + verse;
        references.push_back (ref);
        // Store values to discover next reference.
        previousbook = book;
        previouschapter = chapter;
        previousverse = verse;
      } else {
        messages.push_back ("Reference " + lines[i] + " is not valid and has been removed.");
      }
    }
  }
}


ustring notes_categories_filename ()
// Returns the filename of the notes database.
{
  return gw_build_filename (directories_get_notes (), "categories");
}


void notes_categories_check ()
// Check categories are there - if not, create default set.
{
  if (!g_file_test (notes_categories_filename ().c_str(), G_FILE_TEST_IS_REGULAR)) {
    vector<ustring> categories;
    categories.push_back ("No issue");
    categories.push_back ("For myself");
    categories.push_back ("For subteam");
    categories.push_back ("For team");
    categories.push_back ("For scholar");
    categories.push_back ("For panel");
    categories.push_back ("For church");
    categories.push_back ("For consultant");
    categories.push_back ("For Bible society");
    write_lines (notes_categories_filename (), categories);
  }
}


void notes_categories_add_from_database (vector<ustring> & categories)
// Takes the existing notes categories, if there are any, and adds any
// extra categories found in the database.
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Get the unique categories.
    set<ustring> database_categories;
    rc = sqlite3_open(notes_database_filename ().c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    SqliteReader reader (0);
    char * sql;
    sql = g_strdup_printf("select category from '%s';", TABLE_NOTES);
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    for (unsigned int i = 0; i < reader.ustring0.size(); i++) {
      database_categories.insert (reader.ustring0[i]);
    }
    // Add any new categories to the container.
    set<ustring> localcategories (categories.begin(), categories.end());
    vector<ustring> db_categories (database_categories.begin(), database_categories.end());
    for (unsigned int i = 0; i < db_categories.size(); i++) {
      if (localcategories.find (db_categories[i]) == localcategories.end()) {
        categories.push_back (db_categories[i]);
      }
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
}


void notes_database_vacuum ()
// Run external program to vacuum the database.
{
  string filename = notes_database_filename ();
  ustring command = BIBLEDIT_VACUUM;
  command.append (shell_quote_space (filename) + "&");
  system (command.c_str());
}
