/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "mapping.h"
#include <config.h>
#include "utilities.h"
#include "bible.h"
#include "shell.h"


Mapping::Mapping (const ustring& name, const ustring& book)
{
  ustring command;
  FILE *stream;
  vector<int> raw_values;
  command = "sc-mapping --name" + shell_quote_space (name) + "--book" + shell_quote_space (book);
  stream = popen (command.c_str (), "r");
  char buf[1024];
  while (fgets (buf, sizeof (buf), stream)) {
    raw_values.push_back (convert_to_int (trim(buf)));
  }
  pclose (stream);
  for (unsigned int i = 0; i < raw_values.size(); i++) {
    mychapters.push_back (raw_values[i++]);
    myverses.push_back (raw_values[i++]);
    originalchapters.push_back (raw_values[i++]);
    originalverses.push_back (raw_values[i]);
  }
}


Mapping::~Mapping ()
{
}


void Mapping::me_to_original (int mychapter, const ustring& verse, 
                              vector<int>& original_chapter, vector<int>& original_verse)
/*
Input:
- a chapter (int).
- verse (ustring) : can be a sequence or range too.
Output:
- containers of mapped chapters/verses.
*/
{
  // Storage for the containers with chapters and verses to map.
  vector<int> chapters;
  vector<int> verses;
  // Convert the verse to a container with half verses.
  vector<int> half_verses = verses_encode (verse);
  set<int> encoded;
  // Divide the half-verses by 2 and add them to the containers with values to map.
  for (unsigned int i = 0; i < half_verses.size(); i++) {
    int verse = half_verses[i] / 2;
    if (encoded.find (verse) == encoded.end()) {
      chapters.push_back (mychapter);
      verses.push_back (verse);
      encoded.insert (verse);
    }
  }
  // Do the mapping.
  me_to_original (chapters, verses, original_chapter, original_verse);
}


void Mapping::me_to_original (vector<int> mychapter, vector<int> myverse, 
                              vector<int>& original_chapter, vector<int>& original_verse)
/*
Input:
- containers of chapters/verses.
Output:
- containers of mapped chapters/verses.
Containers as input and as output are needed as there may be a verse that 
maps to one verse in a chapter, and one verse in the next chapter.
*/
{
  bool match = false;
  set <int> chapter_set (mychapter.begin(), mychapter.end());
  set <int> verse_set (myverse.begin(), myverse.end());
  for (unsigned int i = 0; i < mychapters.size(); i++) {
    if (chapter_set.find (mychapters[i]) != chapter_set.end()) {
      if (verse_set.find (myverses[i]) != verse_set.end()) {
        original_chapter.push_back (originalchapters[i]);
        original_verse.push_back (originalverses[i]);
        match = true;
      }
    }
  }
  if (!match) {
    original_chapter.assign (mychapter.begin(), mychapter.end());
    original_verse.assign (myverse.begin(), myverse.end());
  }
}


void Mapping::original_to_me (vector<int> original_chapter, vector<int> original_verse,
                              vector<int>& mychapter, vector<int>& myverse)
{
  bool match = false;
  set <int> chapter_set (original_chapter.begin(), original_chapter.end());
  set <int> verse_set (original_verse.begin(), original_verse.end());
  for (unsigned int i = 0; i < originalchapters.size(); i++) {
    if (chapter_set.find (originalchapters[i]) != chapter_set.end()) {
      if (verse_set.find (originalverses[i]) != verse_set.end()) {
        mychapter.push_back (mychapters[i]);
        myverse.push_back (myverses[i]);
        match = true;
      }
    }
  }
  if (!match) {
    mychapter.assign (original_chapter.begin(), original_chapter.end());
    myverse.assign (original_verse.begin(), original_verse.end());
  }
}
