/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "directories.h"
#include "gwrappers.h"
#include <config.h>
#include "stylesheetutils.h"


int main (int argc, char *argv[])
// Upgrades older stylesheets to newer ones.
{
  // Deal with all available stylesheets as they are stored in the way the first
  // generation of Bibledit did it: A directory, with separate style files in it.
  vector<ustring> available_stylesheets;
  {
    ReadDirectories rd (directories_get_stylesheets (), "", "");
    available_stylesheets.assign (rd.directories.begin(), rd.directories.end());
  }
  for (unsigned int ss = 0; ss < available_stylesheets.size(); ss++) {

    ustring directory;
    directory = gw_build_filename (directories_get_stylesheets (), available_stylesheets[ss]);
    
    // Check whether first version needs upgrade. The upgrade affects styles 
    // that start paragraphs. The substyle should be set properly.
    ReadFiles rf (directory, "", ".xml");
    if (!rf.files.empty())
      gw_message ("Upgrading stylesheet " + available_stylesheets[ss]);
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      // Get the three filenames: old name of stylesheet, template, and new name.
      ustring stylesheet_oldname = gw_build_filename (directory, rf.files[i]);
      ustring templatefile = gw_build_filename (PACKAGE_DATA_DIR, rf.files[i].substr (0, rf.files[i].length() - 4) + ".style");
      ustring stylesheet_newname = gw_build_filename (directory, rf.files[i].substr (0, rf.files[i].length() - 4) + ".style2");
      // Read the text, and see whether this is a paragraph style, and get the line number to upgrade.
      ReadText rt (stylesheet_oldname, true, false);
      int substyle_line_number = -1;
      bool paragraph_style = false;
      for (unsigned int ln = 0; ln < rt.lines.size(); ln++) {
        if (rt.lines[ln].find ("<subtype>") != string::npos) {
          substyle_line_number = ln;
        }
        if (rt.lines[ln].find ("starts-paragraph") != string::npos) {
          paragraph_style = true;
        }
      }
      // If paragraph style, upgrade it.
      if (paragraph_style && (substyle_line_number > 0)) {
        ReadText rt_template (templatefile, true, false);
        for (unsigned int ln = 0; ln < rt_template.lines.size(); ln++) {
          if (rt_template.lines[ln].find ("<subtype>") != string::npos) {
            rt.lines[substyle_line_number] = rt_template.lines[ln];
          }
        }
      }
      // Save under new name and delete old filename.
      write_lines (stylesheet_newname, rt.lines);
      unlink (stylesheet_oldname.c_str());
    }
  }

  // As Bibledit switched to stylesheet stored in SQLite, upgrade to that format.
  for (unsigned int ss = 0; ss < available_stylesheets.size(); ss++) {
    
    ustring stylesheet = available_stylesheets[ss];

    gw_message ("Upgrading stylesheet " + stylesheet);
    
    stylesheet_create_new_empty (stylesheet);

    // Get the available stylesheets.
    ustring directory = gw_build_filename (directories_get_stylesheets (), stylesheet);
    ReadFiles rf (directory, "", STYLE_SUFFIX);

    // Upgrade each of them.
    for (unsigned int i = 0; i < rf.files.size(); i++) {
      ustring marker = rf.files[i].substr (0, rf.files[i].length() - strlen (STYLE_SUFFIX));
      Style style (stylesheet, marker, true);
      style.import_it (gw_build_filename (directory, rf.files[i]));
    }

    // Delete the stylesheet in old formaat.
    remove_directory (directory);      
    
    // Get all recently used styles.
    {
      // Read all the styles (format: style <space> name).
      ustring filename = gw_build_filename (directories_get_stylesheets (), stylesheet, "recently-used");      
      ReadText rt (filename, true);
      vector<ustring> recent_markers;
      // Keep the marker only, and throw away the name.
      for (unsigned int i = 0; i < rt.lines.size(); i++) {
        Parse parse (rt.lines[i]);
        recent_markers.push_back (parse.words[0]);
      } 
      stylesheet_set_recently_used (stylesheet, recent_markers);      
    }
  }  

  return 0;
}
