/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ContentFilterGroup.h"
#include "RegexFilterDescriptor.h"
#include "FilterTag.h"
#include "ConfigErrorHandler.h"
#include "Log.h"
#include "OperationLog.h"
#include "FileOps.h"
#include "Application.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_unistd.h> // for ACE_OS::access
#include <boost/tokenizer.hpp>
#include <string>
#include <sstream>
#include <algorithm>
#include <iterator>
#include <set>
#include <errno.h>

namespace wxGUI
{

namespace {

class FilterCloner
{
public:
	FilterCloner(FilterGroupTag const& group_tag)
	: m_groupTag(group_tag) {}
	
	RegexFilterDescriptor operator()(RegexFilterDescriptor const& filter) const {
		return RegexFilterDescriptor(FilterTag(), m_groupTag, filter);
	}
private:
	FilterGroupTag m_groupTag;
};

} // anonymous namespace

ContentFilterGroup::ContentFilterGroup(wxString const& fname)
:	m_fileName(fname),
	m_tag(m_fileName)
{
}

ContentFilterGroup::ContentFilterGroup(ContentFilterGroup const& other)
:	m_fileName(other.m_fileName),
	m_tag(other.m_tag),
	m_fileStructure(other.m_fileStructure),
	m_filters(other.m_filters)
{
}

ContentFilterGroup::ContentFilterGroup(
	wxString const& fname,
	ContentFilterGroup const& prototype,
	FilterTagPolicy filter_tag_policy)
:	m_fileName(fname),
	m_tag(m_fileName),
	m_fileStructure(prototype.m_fileStructure),
	m_filters()
{
	if (filter_tag_policy == PRESERVE_FILTER_TAGS) {
		ContentFilterList(prototype.m_filters).swap(m_filters);
	} else {
		std::transform(
			prototype.m_filters.begin(),
			prototype.m_filters.end(),
			std::back_inserter(m_filters),
			FilterCloner(m_tag)
		);
	}
}

ContentFilterGroup::~ContentFilterGroup()
{
}

ContentFilterGroup&
ContentFilterGroup::operator=(ContentFilterGroup const& rhs)
{
	ContentFilterGroup(rhs).swap(*this);
	return *this;
}

void
ContentFilterGroup::swap(ContentFilterGroup& other)
{
	m_fileName.swap(other.m_fileName);
	m_tag.swap(other.m_tag);
	m_fileStructure.swap(other.m_fileStructure);
	m_filters.swap(other.m_filters);
}

bool
ContentFilterGroup::load()
{
	std::string text;
	
	if (!FileOps::readFile(getFilterFilePath(), text)) {
		return false;
	}
	
	FilterFileStructure new_structure;
	ContentFilterList new_filters;
	ConfigErrorHandler eh(m_fileName);
	new_structure.load(text, new_filters, eh, m_tag);
	loadEnabledFilterList(new_filters);
	
	m_fileStructure.swap(new_structure);
	m_filters.swap(new_filters);
	
	return true;
}

bool
ContentFilterGroup::save() const
{
	return saveEnabledFileOnly() && saveFilterFileOnly();
}

bool
ContentFilterGroup::saveFilterFileOnly() const
{
	std::string text;
	
	{
		std::ostringstream strm;
		m_fileStructure.toStream(strm);
		text = strm.str();
	}
	
	return FileOps::writeFile(getFilterFilePath(), text);
}

bool
ContentFilterGroup::saveEnabledFileOnly() const
{
	std::string text;
	bool all_enabled = true;
	bool all_disabled = true;
	
	{
		std::ostringstream strm;
		ContentFilterList::const_iterator it(m_filters.begin());
		ContentFilterList::const_iterator const end(m_filters.end());
		for (; it != end; ++it) {
			if (it->isEnabled()) {
				all_disabled = false;
				strm << it->name() << '\n';
			} else {
				all_enabled = false;
			}
		}
		if (all_enabled) {
			text = "*";
		} else {
			text = strm.str();
		}
	}
	
	wxFileName const path(getEnabledFilePath());
	
	if (all_disabled) {
		return FileOps::deleteFile(path);
	} else {
		return FileOps::writeFile(path, text);
	}
}

bool
ContentFilterGroup::deleteFromDisk() const
{
	Log* log = OperationLog::instance();
	wxFileName const filter_file_path(getFilterFilePath());
	wxFileName const enabled_file_path(getEnabledFilePath());
	
	if (ACE_OS::access(getFilterFileDir().GetFullPath().c_str(), W_OK|X_OK) == -1) {
		if (errno != ENOENT) {
			log->appendRecord(
				_T("Error deleting ")+filter_file_path.GetFullPath(),
				log->getErrorStyle()
			);
			return false;
		}
	}
	
	return FileOps::deleteFile(enabled_file_path)
		&& FileOps::deleteFile(filter_file_path);
}

bool
ContentFilterGroup::renameOnDisk(wxString const& new_fname) const
{
	if (new_fname == m_fileName) {
		return true;
	}
	wxFileName const old_filter_path(getFilterFilePath());
	wxFileName const new_filter_path(getFilterFilePath(new_fname));
	wxFileName const old_enabled_path(getEnabledFilePath());
	wxFileName const new_enabled_path(getEnabledFilePath(new_fname));
	
	bool const enabled_file_exists = old_enabled_path.FileExists();
	if (enabled_file_exists) {
		if (!FileOps::renameFile(old_enabled_path, new_enabled_path)) {
			return false;
		}
	}
	if (!FileOps::renameFile(old_filter_path, new_filter_path)) {
		if (enabled_file_exists) {
			FileOps::renameFile(new_enabled_path, old_enabled_path);
		}
		return false;
	}
	
	return true;
}

RegexFilterDescriptor const*
ContentFilterGroup::findFilter(FilterTag const& id) const
{
	ContentFilterList::const_iterator const it(m_filters.find(id));
	if (it == m_filters.end()) {
		return 0;
	}
	return &*it;
}

bool
ContentFilterGroup::appendFilter(RegexFilterDescriptor const& filter)
{
	if (m_filters.find(filter.getTag()) != m_filters.end()) {
		return false;
	}
	
	m_filters.push_back(filter);
	m_fileStructure.updateWith(m_filters);
	return true;
}

bool
ContentFilterGroup::insertFilter(
	RegexFilterDescriptor const& filter, FilterTag const& before)
{
	ContentFilterList::iterator const it(m_filters.find(before));
	if (it == m_filters.end()) {
		return false;
	}
	
	m_filters.insert(it, filter);
	m_fileStructure.updateWith(m_filters);
	return true;
}

bool
ContentFilterGroup::updateFilter(RegexFilterDescriptor const& filter)
{
	ContentFilterList::iterator const it(m_filters.find(filter.getTag()));
	if (it == m_filters.end()) {
		return false;
	}
	
	*it = filter;
	m_fileStructure.updateWith(m_filters);
	return true;
}

bool
ContentFilterGroup::renameFilter(FilterTag const& id, std::string const& new_name)
{
	ContentFilterList::iterator const it(m_filters.find(id));
	if (it == m_filters.end()) {
		return false;
	}
	
	it->name() = new_name;
	m_fileStructure.renameFilter(std::distance(m_filters.begin(), it), new_name);
	return true;
}

bool
ContentFilterGroup::deleteFilter(FilterTag const& id)
{
	ContentFilterList::iterator const it(m_filters.find(id));
	if (it == m_filters.end()) {
		return false;
	}
	
	m_filters.erase(it);
	m_fileStructure.updateWith(m_filters);
	return true;
}

bool
ContentFilterGroup::swapFilters(FilterTag const& id1, FilterTag const& id2)
{
	ContentFilterList::iterator const it1(m_filters.find(id1));
	if (it1 == m_filters.end()) {
		return false;
	}
	ContentFilterList::iterator const it2(m_filters.find(id2));
	if (it2 == m_filters.end()) {
		return false;
	}
	
	if (it1 != it2) {
		it1->swap(*it2);
		m_fileStructure.updateWith(m_filters);
	}
	return true;
}

bool
ContentFilterGroup::filterSetEnabled(FilterTag const& id, bool enabled)
{
	ContentFilterList::iterator const it(m_filters.find(id));
	if (it == m_filters.end()) {
		return false;
	}
	
	it->setEnabled(enabled);
	return true;
}

void
ContentFilterGroup::allFiltersSetEnabled(bool enabled)
{
	ContentFilterList::iterator it(m_filters.begin());
	ContentFilterList::iterator const end(m_filters.end());
	for (; it != end; ++it) {
		it->setEnabled(enabled);
	}
}

wxFileName
ContentFilterGroup::getFilterFileDir() const
{
	return Application::instance()->getFiltersDir();
}

wxFileName
ContentFilterGroup::getEnabledFileDir() const
{
	return Application::instance()->getFiltersDir();
}

wxFileName
ContentFilterGroup::getFilterFilePath(wxString const& filter_fname) const
{
	wxFileName path(getFilterFileDir());
	path.SetName(filter_fname);
	return path;
}

wxFileName
ContentFilterGroup::getFilterFilePath() const
{
	return getFilterFilePath(m_fileName);
}

wxFileName
ContentFilterGroup::getEnabledFilePath(wxString const& filter_fname) const
{
	wxFileName path(getEnabledFileDir());
	path.SetName(filter_fname+_T(".enabled"));
	return path;
}

wxFileName
ContentFilterGroup::getEnabledFilePath() const
{
	return getEnabledFilePath(m_fileName);
}

void
ContentFilterGroup::loadEnabledFilterList(ContentFilterList& filters)
{
	std::string text;
	
	if (!FileOps::readFile(getEnabledFilePath(), text, 0, false)) {
		return;
	}
	
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("\r\n");
	Tokenizer tokens(text, sep);
	std::set<std::string> enabled_set(tokens.begin(), tokens.end());
	bool const all_enabled = (enabled_set.find("*") != enabled_set.end());

	ContentFilterList::iterator it(filters.begin());
	ContentFilterList::iterator const end(filters.end());
	for (; it != end; ++it) {
		RegexFilterDescriptor& filter = *it;
		bool const enabled = all_enabled ||
			(enabled_set.find(filter.name()) != enabled_set.end());
		filter.setEnabled(enabled);
	}
}

} // namespace wxGUI
