/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef WORKERTHREADPOOLEX_H_
#define WORKERTHREADPOOLEX_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "WorkerThreadPool.h"
#include <sigc++/sigc++.h>

/**
 * \brief Accept and process client connections.
 *
 * This class is a wrapper around WorkerThreadPool. Besides the functionality
 * provided by WorkerThreadPool, it tracks "is accepting connections" status
 * and provides notifications on status change.
 * \note Unlike WorkerThreadPool, this class is not thread-safe.
 */
class WorkerThreadPoolEx
{
	DECLARE_NON_COPYABLE(WorkerThreadPoolEx)
public:
	/**
	 * \brief Constructs an initially inactive object.
	 */
	WorkerThreadPoolEx();
	
	/**
	 * \brief Destroys the object, deactivating it first.
	 */
	~WorkerThreadPoolEx();
	
	/**
	 * \see WorkerThreadPool::activate()
	 */
	void activate();
	
	/**
	 * \see WorkerThreadPool::deactivate()
	 */
	void deactivate();
	
	/**
	 * \see WorkerThreadPool::addAcceptor()
	 */
	void addAcceptor(ConnAcceptor::AcceptorPtr const& acceptor);
	
	/**
	 * \see WorkerThreadPool::removeAcceptor()
	 * \note Unlike WorkerThreadPool::removeAcceptor(), passing
	 *       an acceptor not registered with addAcceptor() results
	 *       in undefined behavior.
	 */
	void removeAcceptor(ConnAcceptor::AcceptorPtr const& acceptor);
	
	/**
	 * \see WorkerThreadPool::removeAllAcceptors()
	 */
	void removeAllAcceptors();
	
	/**
	 * \return true if there is at least one registered acceptor
	 *         and activate() was called.
	 */
	bool isAcceptingConnections() { return m_isAcceptingConnections; }
	
	/**
	 * \brief Allows receiving notifications on "is accepting connections"
	 *        status change.
	 */
	sigc::signal<void, bool>& acceptingConnectionsSignal() {
		return m_acceptingConnectionsSignal;
	}
private:
	void updateStatus();
	
	WorkerThreadPool m_rep;
	sigc::signal<void, bool> m_acceptingConnectionsSignal;
	int m_numAcceptors;
	bool m_isActive;
	bool m_isAcceptingConnections;
};

#endif
