/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include "Client.h"
#include "AbstractServer.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestLine.h"
#include "RequestPtr.h"
#include "RequestTag.h"
#include "ImmediateResponse.h"
#include "ErrorDescriptor.h"
#include "ErrorFactory.h"
#include "ErrorCodes.h"
#include "ErrorWrapper.h"
#include "SplittableBuffer.h"
#include "WeakPtr.h"
#include "DataChunk.h"
#include "ScopedIncDec.h"
#include "AbstractDataConsumer.h"
#include "AbstractRequestHandler.h"
#include "AbstractResponseHandler.h"
#include "HttpStreamWriter.h"
#include "HttpMessageShaper.h"
#include "HttpVersion.h"
#include "TimeDelta.h"
#include "RequestLog.h"
#include "URI.h"
#include "NetworkActivityReporter.h"
#include "Debug.h"
#include <stddef.h>
#include <cassert>

class ResponseErrorInitiator;
class DownloadProgress;

using namespace std;

class Client::ResponseHandler :
	public AbstractResponseHandler,
	public AbstractDataConsumer
{
	DECLARE_NON_COPYABLE(ResponseHandler)
public:
	ResponseHandler(
		ClientWeakPtr const& client_ptr,
		ConstRequestPtr const& request,
		RequestTag const& request_tag,
		HttpStreamWriter::Connection conn_persistence);
	
	virtual ~ResponseHandler();
private:
	virtual void processProvisionalResponse(
		RequestStatus& status,
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processResponseMetadata(
		RequestStatus& status,
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processBodyData(
		RequestStatus& status, SplittableBuffer& data, bool eof);
	
	virtual void processError(
		RequestStatus& status, std::auto_ptr<ErrorDescriptor> edesc);
	
	virtual void processNewData(SplittableBuffer& data, bool eof);
	
	bool tryDiscardOutput();
	
	ClientWeakPtr m_ptrClient;
	ConstRequestPtr const m_ptrRequest;
	RequestTag m_requestTag;
	bool m_isEOF;
	uintmax_t m_producedDataSize;
	HttpStreamWriter m_streamWriter;
	HttpStreamWriter::Connection m_connPersistence;
};


class Client::ScopedSuspender
{
public:
	ScopedSuspender(Client& owner) : m_rOwner(owner) {
		m_rOwner.suspend();
	}
	
	~ScopedSuspender() {
		m_rOwner.resume();
	}
private:
	Client& m_rOwner;
};


Client::Client(
	SocketPtr const& client_stream,
	AbstractServer& server, Reactor& reactor)
:	m_ptrClientStream(client_stream),
	m_rServer(server),
	m_rReactor(reactor),
	m_isEndOfOutput(false),
	m_readTimer(
		sigc::mem_fun(*this, &Client::onReadTimeout),
		TimeDelta::fromSec(READ_TIMEOUT), &reactor
	),
	m_writeTimer(
		sigc::mem_fun(*this, &Client::onWriteTimeout),
		TimeDelta::fromSec(WRITE_TIMEOUT), &reactor
	),
	m_idleTimer(
		sigc::mem_fun(*this, &Client::onIdleTimeout),
		TimeDelta::fromSec(IDLE_TIMEOUT), &reactor
	),
	m_eventMask(Reactor::READ),
	m_flags(0),
	m_isWaitingForWrite(0),
	m_ptrSelf(this)
{
	DBG_CLIENT_CONNECTION_BEGIN();
	m_ptrClientStream->enable(ACE_NONBLOCK);
	
	m_rReactor.beforeSleepSignal().connect(
		sigc::mem_fun(*this, &Client::updateState)
	);
	try {
		m_handlerId = m_rReactor.registerHandler(
			m_ptrClientStream->get_handle(),
			IntrusivePtr<EventHandlerBase>(this),
			m_eventMask
		);
	} catch (Reactor::Exception&) {
		DEBUGLOG("could not register client socket in the reactor");
		scheduleSessionTermination();
	}
}

Client::~Client()
{
	if (m_handlerId) {
		m_rReactor.unregisterHandler(m_handlerId);
	}
	DBG_CLIENT_CONNECTION_END();
}

void
Client::processOutputData()
{
	//DEBUGLOG("Client::processOutputData()");
	while (!m_outputData.empty()) {
		if (writeOutputData() == 0) {
			break;
		}
	}
	
}

size_t
Client::writeOutputData()
{
	iovec iov[ACE_IOV_MAX];
	int iov_size = 0;
	for (SplittableBuffer::SpanIterator iter(m_outputData.spanBegin());
		iov_size < int(sizeof(iov)/sizeof(*iov)) && !iter.isAtRightBorder();
		++iov_size, ++iter) {
		iov[iov_size].iov_base = (char*)iter->begin();
		iov[iov_size].iov_len = iter->size();
	}
	ssize_t sent = m_ptrClientStream->sendv(
		iov, iov_size
		// the timeout is infinite, but the socket is in nonblocking mode
	);
	if (sent < 0 && errno == EWOULDBLOCK) {
		m_flags |= WRITE_BUFFER_FULL;
	} else if (sent > 0) {
		m_flags &= ~WRITE_BUFFER_FULL;
		m_writeTimer.postpone();
		NetworkActivityReporter::instance()->reportActivity();
#ifdef DEBUG
		SplittableBuffer traf;
		m_outputData.splitFrontAppendBack(sent, traf);
		DBG_TRAFFIC_TO_CLIENT(traf);
#else
		m_outputData.trimFront(sent);
#endif
		if (m_outputData.empty()) {
			m_ptrLastOutputWriter.reset(0);
		}
		return sent;
	} else {
		DEBUGLOG("ERROR writing to client");
		scheduleSessionTermination();
	}
	return 0;
}

void
Client::processInputData(bool eof)
{
	//DEBUGLOG("Client::processInputData()");
	while (true) {
		if (m_inputData.empty() && !eof) {
			break;
		}
		if (isInactiveState()) {
			switchToRequestStart();
		}
		// at this point the parser is guaranteed to be in the RequestStart state
		SplittableBuffer body_data;
		bool body_eof = false;
		bool const empty_input = m_inputData.empty();
		Status const status = processNewData(
			m_inputData, eof, body_data, body_eof
		);
		
		if (status == MESSAGE_ENDS) {
			DBG_INCOMING_REQUEST_END();
		}

		if (!body_data.empty() || body_eof) {
			ScopedSuspender suspender(*this);
			
			// This may result in a recursive invocation
			// of reactor event loop.
			m_ptrRequestHandler->appendBodyData(body_data, body_eof);
		}
		
		if (status == MESSAGE_ENDS) {
			m_ptrRequestHandler.reset(0);
			m_ptrRequestURI.reset(0);
		} else if (status == ERROR_STATE) {
			//DEBUGLOG("ERROR parsing request");
			std::auto_ptr<ErrorDescriptor> edesc(retrieveErrorDescriptor());
			if (!edesc.get()) {
				edesc = ErrorFactory::errParsingRequest(
					ErrorCodes::ERROR_PARSING_RESPONSE,
					"error parsing request: "+errorMessage(),
					m_ptrRequestURI.get() ? *m_ptrRequestURI : URI(BString()),
					errorMessage()
				);
			}
			
			if (m_ptrRequestHandler.get()) {
				// the error happened after we've received the metadata
				m_ptrRequestHandler->cancel(edesc);
				m_ptrRequestHandler.reset(0);
				m_ptrRequestURI.reset(0);
			} else {
				if (empty_input) {
					// an eof while in RequestStart state generates an error
					break;
				}
				// We don't have request metadata, as the error happened
				// too early. We synthesize a dummy request and submit it
				// with an error response attached.
				HttpRequestLine const rline(
					BString("GET"),
					URI(BString("<broken request>")),
					HttpVersion::HTTP_1_0
				);
				RequestPtr request(new HttpRequestMetadata(rline));
				request->setBodyStatus(HttpRequestMetadata::BODY_FORBIDDEN);
				
				RequestTag request_tag;
				
				DBG_REGISTER_CLIENT_REQUEST(*request, request_tag);
				DBG_INCOMING_REQUEST_BEGIN(*request, request_tag);
				
				IntrusivePtr<ResponseHandler> response_handler(
					new ResponseHandler(
						m_ptrSelf, request, request_tag,
						HttpStreamWriter::CONN_CLOSE
					)
				);
				
				ScopedSuspender suspender(*this);
				
				IntrusivePtr<AbstractRequestHandler> request_handler(
					m_rServer.submitRequest(
						request, request_tag,
						response_handler,
						ErrorWrapper::wrap(edesc)
					)
				);
				SplittableBuffer body_data;
				
				// This may result in a recursive invocation
				// of reactor event loop.
				request_handler->appendBodyData(body_data, true);
			}
			
			DBG_INCOMING_REQUEST_ERROR();
			switchToInactive();
			m_flags |= READING_FORBIDDEN;
			m_inputData.clear();
			break;
		}
		
		if (status == NEED_MORE_DATA) {
			// Can happen for example if input ends in the middle of a header.
			break;
		}
	}
}

void
Client::processResponseData(
	SplittableBuffer& data, bool const eof,
	IntrusivePtr<ResponseHandler> const& response_handler)
{
	if (m_isEndOfOutput || isSessionTerminationScheduled()) {
		data.clear();
		return;
	}
	
	if (!data.empty()) {
		m_outputData.appendDestructive(data);
		m_ptrLastOutputWriter = response_handler;
	}
	m_isEndOfOutput = eof;
	
	if (m_outputData.size() > MAX_WRITE_BUF_SIZE) {
		ScopedIncDec<int> flag_manager(m_isWaitingForWrite);
		while (!m_outputData.empty()) {
			if (m_rReactor.handleEvents() != Reactor::SUCCESS) {
				break;
			}
		}
	}
}

bool
Client::tryDiscardResponseOutput(
	ResponseHandler const& response_handler,
	uintmax_t const discard_size)
{
	if (&response_handler !=  m_ptrLastOutputWriter.get()) {
		return false;
	}
	if (discard_size > m_outputData.size()) {
		return false;
	}
	
	m_outputData.trimBack(discard_size);
	if (m_outputData.empty()) {
		m_ptrLastOutputWriter.reset(0);
	}
	
	return true;
}

bool
Client::isRequestInProgress() const
{
	return !(isInactiveState() || isInitialState() || isErrorState());
}

bool
Client::isIdle() const
{
	return (m_outputData.empty() && !isResponseInProgress() &&
		m_rServer.isIdle() && !isRequestInProgress());
}

void
Client::onReadTimeout()
{
	DEBUGLOG("client read timeout");
	scheduleSessionTermination();
}

void
Client::onWriteTimeout()
{
	DEBUGLOG("client write timeout");
	scheduleSessionTermination();
}

void
Client::onIdleTimeout()
{
	DEBUGLOG("client idle timeout");
	scheduleSessionTermination();
}

void
Client::terminateSession()
{
	m_readTimer.deactivate();
	m_writeTimer.deactivate();
	m_idleTimer.deactivate();
	m_eventMask = Reactor::NO_EVENTS;
	if (m_handlerId) {
		m_rReactor.setEvents(m_handlerId, m_eventMask);
	}
	m_rReactor.stop();
}

void
Client::updateState()
{
	/*
	upadateState() MUST NOT generate any output!!!
	Server::updateState() and Client::updateState() are called just before
	going to sleep to update their event masks. If one of them generates
	output for the other, that other one would need to update its event
	mask, but it may not have a chance to do so, as it may have already run
	in the current reactor dispatch.
	*/

	if (!m_outputData.empty() && !(m_flags & WRITE_BUFFER_FULL) &&
	    !isSessionTerminationScheduled()) {
		// necessary on the edge-triggered WFMOReactor
		processOutputData(); // can schedule session termination
	}
	if (m_rServer.isIdle() && isReadingForbidden()) {
		// This takes care of a situation when parse error happened
		// after submitting the request to the server.
		m_isEndOfOutput = true;
	}
	if (m_outputData.empty() && m_isEndOfOutput) {
		scheduleSessionTermination();
	}
	if (isSessionTerminationScheduled()) {
		terminateSession();
		return;
	}
	
	Reactor::IOEvents mask = m_eventMask;
	
	if (m_flags & (READING_FORBIDDEN|READING_SUSPENDED) ||
	    m_rServer.isOverloaded()) {
		mask &= ~Reactor::READ;
	} else {
		mask |= Reactor::READ;
		
	}
	if (!m_outputData.empty()) {
		mask |= Reactor::WRITE;
	} else {
		mask &= ~Reactor::WRITE;
		if (m_isWaitingForWrite) {
			m_rReactor.wakeup();
		}
		
		m_flags &= ~WRITE_BUFFER_FULL;
		// It may actually be full, but we don't want to miss
		// an edge-triggered (in case of WFMOReactor) write event.
	}
	
	if ((mask & Reactor::READ) && isRequestInProgress()) {
		m_readTimer.ensureActive();
	} else {
		m_readTimer.deactivate();
	}
	if (mask & Reactor::WRITE) {
		m_writeTimer.ensureActive();
	} else {
		m_writeTimer.deactivate();
	}
	if (isIdle()) {
		m_idleTimer.ensureActive();
	} else {
		m_idleTimer.deactivate();
	}
	
	if (mask != m_eventMask) {
		m_rReactor.setEvents(m_handlerId, mask);
		m_eventMask = mask;
	}
}

void
Client::eventGotMetadata(auto_ptr<HttpRequestMetadata> metadata, bool is_persistent)
{
	m_ptrRequestURI.reset(new URI(metadata->requestLine().getURI()));
	
	RequestTag request_tag;
	
	DBG_REGISTER_CLIENT_REQUEST(*metadata, request_tag);
	DBG_INCOMING_REQUEST_BEGIN(*metadata, request_tag);
	
	HttpMessageShaper::prepareForForwarding(*metadata);
	
	ConstRequestPtr request(metadata.release());
	// metadata is null now !!!
	
	IntrusivePtr<ResponseHandler> response_handler(
		new ResponseHandler(
			m_ptrSelf, request, request_tag,
			is_persistent ? HttpStreamWriter::CONN_KEEP_ALIVE
			: HttpStreamWriter::CONN_CLOSE
		)
	);
	ScopedSuspender suspender(*this);
	m_ptrRequestHandler = m_rServer.submitRequest(
 		request, request_tag, response_handler
	);
}

void
Client::handleRead(ACE_HANDLE)
{
	if (m_flags & (READING_FORBIDDEN|READING_SUSPENDED)) {
		// this may happen because the event mask is updated after
		// all of the triggered events are processed
		return;
	}
	auto_ptr<DataChunk> chunk(DataChunk::create(READ_BUF_SIZE));
	ssize_t received = m_ptrClientStream->recv(
		chunk->getDataAddr(), chunk->getDataSize()
		// the timeout is infinite, but the socket is in nonblocking mode
	);
	if (received > 0) {
		NetworkActivityReporter::instance()->reportActivity();
		chunk = DataChunk::resize(chunk, received);
#ifdef DEBUG
		SplittableBuffer traf;
		traf.append(chunk);
		DBG_TRAFFIC_FROM_CLIENT(traf);
		m_inputData.appendDestructive(traf);
#else
		m_inputData.append(chunk);
#endif
	} else if (received < 0 && errno == EWOULDBLOCK) {
		// not a problem
		return;
	}
	m_readTimer.postpone();
	bool eof = (received <= 0);
	if (received < 0) {
		DEBUGLOG("ERROR reading from client");
	}
	processInputData(eof);
	if (received < 0 || (received == 0 && isIdle())) {
		scheduleSessionTermination();
	} else if (received == 0) {
		// an EOF from the client doesn't mean it's refusing to accept data
		m_flags |= READING_FORBIDDEN;
	}
}

void
Client::handleWrite(ACE_HANDLE)
{
	m_flags &= ~WRITE_BUFFER_FULL;
	processOutputData();
}


/*======================== Client::ResponseHandler ===========================*/

Client::ResponseHandler::ResponseHandler(
	ClientWeakPtr const& client_ptr, ConstRequestPtr const& request,
	RequestTag const& request_tag,
	HttpStreamWriter::Connection conn_persistence)
:	m_ptrClient(client_ptr),
	m_ptrRequest(request),
	m_requestTag(request_tag),
	m_isEOF(false),
	m_producedDataSize(0),
	m_streamWriter(*this),
	m_connPersistence(conn_persistence)
{
}

Client::ResponseHandler::~ResponseHandler()
{
}

void
Client::ResponseHandler::processProvisionalResponse(
	RequestStatus&, std::auto_ptr<HttpResponseMetadata> metadata)
{
#ifdef DEBUG
	metadata->headers().setHeader(
		BString("X-Request-URI"),
		m_ptrRequest->requestLine().getURI().toBString()
	);
#endif
	
	if (m_ptrRequest->requestLine().getHttpVersion() >= HttpVersion::HTTP_1_1) {
		m_streamWriter.startResponse(
			*metadata, m_ptrRequest->requestLine(),
			HttpStreamWriter::CONN_KEEP_ALIVE
		);
		SplittableBuffer body_data;
		m_streamWriter.appendBodyData(body_data, true);
	}
}

void
Client::ResponseHandler::processResponseMetadata(
	RequestStatus&, std::auto_ptr<HttpResponseMetadata> metadata)
{
#ifdef DEBUG
	metadata->headers().setHeader(
		BString("X-Request-URI"),
		m_ptrRequest->requestLine().getURI().toBString()
	);
#endif
	
	if (Client* client = m_ptrClient.get()) {
		client->setResponseInProgress(true);
	}
	m_streamWriter.startResponse(
		*metadata, m_ptrRequest->requestLine(),
		m_connPersistence
	);
	m_connPersistence = m_streamWriter.getConnPersistence();

#ifdef DEBUG
	typedef AbstractDebugAgent ADA;
	ADA::HttpMessageType msg_type = ADA::UNMODIFIED_OUTGOING_RESPONSE;
	int const scode = metadata->statusLine().getCode();
	if (scode >= 400 && scode < 600) {
		msg_type = ADA::ERROR_OUTGOING_RESPONSE;
	} else if (m_requestTag->flags().isSet(RequestTag::RESPONSE_CRAFTED)) {
		msg_type = ADA::CRAFTED_OUTGOING_RESPONSE;
	} else if (m_requestTag->flags().isSet(RequestTag::RESPONSE_MODIFIED)) {
		msg_type = ADA::FILTERED_OUTGOING_RESPONSE;
	}
	DBG_HTTP_MESSAGE_BEGIN(*metadata, m_requestTag, msg_type);
#endif
}

void
Client::ResponseHandler::processBodyData(
	RequestStatus&, SplittableBuffer& data, bool eof)
{
	if (m_isEOF) {
		data.clear();
		return;
	}
	m_isEOF = eof;
	if (m_isEOF) {
		DBG_OUTGOING_RESPONSE_END();
	}
	return m_streamWriter.appendBodyData(data, eof);
}

void
Client::ResponseHandler::processError(
	RequestStatus& status, std::auto_ptr<ErrorDescriptor> edesc)
{
	DEBUGLOG(
		"Client::ResponseHandler::processError(): "
		<< edesc->getDebugMessage() << "\r\nURL: "
		<< m_ptrRequest->requestLine().getURI()
	);
	
	if (edesc->getErrorCode() == ErrorCodes::ERROR_DECOMPRESSING_RESPONSE) {
		// Some sites actually produce broken compressed content,
		// but browsers display it correctly, because the error
		// is at the end of file.
		if (m_producedDataSize != 0) {
			if (!m_isEOF) {
				m_isEOF = true;
				SplittableBuffer data;
				m_streamWriter.appendBodyData(data, true);
			}
		}
	}
	if (m_isEOF) {
		// ignore errors that come after EOF
		// that could be a REQUEST_CANCEL_TO_SAVE_TRAFFIC error
	} else if (m_producedDataSize == 0 || tryDiscardOutput()) {
		edesc->retrieveResponse()->output(*this, m_ptrRequest, status, m_requestTag);
	} else {
		//DEBUGLOG("some part of the response has already made its way to the client");
		DBG_OUTGOING_RESPONSE_ERROR();
		if (Client* client = m_ptrClient.get()) {
			client->scheduleSessionTermination();
		}
	}
}

void
Client::ResponseHandler::processNewData(SplittableBuffer& data, bool eof)
{
	m_producedDataSize += data.size();
	
	if (Client* client = m_ptrClient.get()) {
		bool stream_eof = eof && (
			m_connPersistence == HttpStreamWriter::CONN_CLOSE
		);
		client->processResponseData(
			data, stream_eof,
			IntrusivePtr<ResponseHandler>(this)
		);
		if (eof) {
			client->setResponseInProgress(false);
		}
	}
}

bool
Client::ResponseHandler::tryDiscardOutput()
{
	if (Client* client = m_ptrClient.get()) {
		if (client->tryDiscardResponseOutput(*this, m_producedDataSize)) {
			m_producedDataSize = 0;
			return true;
		}
	}
	return false;
}
