/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "ProxyWatcher.h"
#include "NetworkPrefs.h"
#include "CFUtils.h"
#include "StringLess.h"
#include "GlobalState.h"
#include "Forwarding.h"
#include "ProxyDescriptor.h"
#include "SymbolicInetAddr.h"
#include "AssertException.h"
#include <string>
#include <list>
#include <vector>

using namespace std;

class ProxyWatcher::GenericError
{
public:
	GenericError(std::string const& msg);
	
	GenericError(char const* msg);
	
	~GenericError();
	
	std::string const& getMessage() const { return m_msg; }
private:
	std::string m_msg;
};
	

ProxyWatcher::ProxyWatcher(
	StringRef const& client_name, RefPtr<CFRunLoopRef> const& run_loop)
try
:	m_ptrClientName(client_name),
	m_ptrRunLoop(run_loop),
	m_ptrNetworkStateKey(
		SCDynamicStoreKeyCreateNetworkGlobalEntity(
			NULL, kSCDynamicStoreDomainState, kSCEntNetIPv4
		),
		NORETAIN
	),
	m_ptrProxiesStateKey(
		SCDynamicStoreKeyCreateNetworkGlobalEntity(
			NULL, kSCDynamicStoreDomainState, kSCEntNetProxies
		),
		NORETAIN
	),
	m_bfPrefix(CFSTR("(BF) "))
{
	SCDynamicStoreContext context;
	context.version = 0;
	context.info = this;
	context.retain = 0;
	context.release = 0;
	context.copyDescription = 0;
	
	m_ptrStore = RefPtr<SCDynamicStoreRef>(
		SCDynamicStoreCreate(NULL, m_ptrClientName, &storeModifiedCallback, &context),
		NORETAIN
	);
	if (!m_ptrStore) {
		throw GenericError("SCDynamicStoreCreate() failed");
	}
	
	m_ptrRunLoopSource = RefPtr<CFRunLoopSourceRef>(
		SCDynamicStoreCreateRunLoopSource(NULL, m_ptrStore, 0), NORETAIN
	);
	if (!m_ptrRunLoopSource) {
		throw GenericError("SCDynamicStoreCreateRunLoopSource() failed");
	}
	
	CFRunLoopAddSource(m_ptrRunLoop, m_ptrRunLoopSource, kCFRunLoopCommonModes);
	
	ArrayRef keys = CFUtils::createCFTypeArray(
		m_ptrNetworkStateKey.get(), m_ptrProxiesStateKey.get(), 0
	);
	if (!SCDynamicStoreSetNotificationKeys(m_ptrStore, keys, NULL)) {
		throw GenericError("SCDynamicStoreSetNotificationKeys() failed");
	} 
		
	updateProxies();
} catch (GenericError const& e) {
	setUnmaskableError(e.getMessage());
} catch (AssertException const& e) {
	setUnmaskableError(e.getMessage());
} catch (std::exception const& e) {
	setUnmaskableError(string("Exception: ")+e.what());
}

ProxyWatcher::~ProxyWatcher()
{
	if (m_ptrRunLoopSource) {
		CFRunLoopRemoveSource(m_ptrRunLoop, m_ptrRunLoopSource, kCFRunLoopCommonModes);
	}
}

void
ProxyWatcher::updateProxies()
try {
	DictRef network_state(
		(CFDictionaryRef)SCDynamicStoreCopyValue(m_ptrStore, m_ptrNetworkStateKey),
		NORETAIN
	);
	if (!network_state) {
		/*
		This means no network connection is available.  It can happen
		during location changes.  In this case, we'll receive another
		notification later.
		We don't reset proxy settings here, because it makes no difference
		when the network is down.
		*/
		return;
	}
	
	StringRef primary_service(
		(CFStringRef)CFDictionaryGetValue(
			network_state, kSCDynamicStorePropNetPrimaryService
		),
		RETAIN
	);
	if (!primary_service || CFGetTypeID(primary_service) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	StringRef service_path = CFUtils::composePath(kSCPrefNetworkServices, primary_service.get(), 0);
	
	// PrimaryInterface is of no use to us, as it's not the same thing
	// as DeviceName of the service.
	// Example (PPPoE): PrimaryInterface = ppp0, DeviceName = en0
	
	getProxiesFromPersistentStore(service_path);
} catch (GenericError const& e) {
	setUnmaskableError(e.getMessage());
} catch (AssertException const& e) {
	setUnmaskableError(e.getMessage());
} catch (std::exception const& e) {
	setUnmaskableError(string("Exception: ")+e.what());
}

void
ProxyWatcher::storeModifiedCallback(
	SCDynamicStoreRef store, CFArrayRef changedKeys, void* info)
{
	ProxyWatcher* obj = static_cast<ProxyWatcher*>(info);
	obj->updateProxies();
}

void
ProxyWatcher::getProxiesFromPersistentStore(StringRef const& primary_service)
{	
	NetworkPrefs prefs(m_ptrClientName);
	
	StringRef device_name = getServiceDeviceName(prefs, primary_service);
	if (!device_name) {
		throw GenericError("Could not get the network device name");
	}
	
	StringPair cur_set = findSetReferencingService(prefs, primary_service);
	if (!cur_set.first) {
		cur_set = getCurrentSet(prefs);
	}
	StringRef const& set_name = cur_set.second;
	
	StringRef service_path;
	if (!CFStringHasPrefix(set_name, m_bfPrefix)) {
		service_path = primary_service;
	} else { // has (BF) prefix
		CFIndex name_len = CFStringGetLength(set_name);
		CFIndex bf_prefix_len = CFStringGetLength(m_bfPrefix);
		CFRange range = CFRangeMake(bf_prefix_len, name_len - bf_prefix_len);
		StringRef counterpart_set_name(
			CFStringCreateWithSubstring(NULL, set_name, range),
			NORETAIN
		);
		StringRef counterpart_set_id = findSetByName(prefs, counterpart_set_name);
		if (!counterpart_set_id) {
			throw GenericError(
				"The current (BF) Location doesn't have a non-(BF) counterpart"
			);
		}
		
		service_path = findServiceBySetAndDeviceName(
			prefs, counterpart_set_id, device_name
		);
		if (!service_path) {
			throw GenericError(
				"(BF) Locations are out-of-sync with their counterparts.\n"
				"Reinstalling BFilter will fix that."
			);
		}
	}
	
	DictRef service(
		SCPreferencesPathGetValue(prefs.getRep(), service_path),
		RETAIN
	);
	if (!service || CFGetTypeID(service) != CFDictionaryGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	getProxiesFromService(service, set_name);
}

StringRef
ProxyWatcher::getServiceDeviceName(
	NetworkPrefs& prefs, StringRef const& service_path)
{
	StringRef path = CFUtils::composePath(service_path, kSCEntNetInterface, 0);
	
	DictRef iface_dict(
		SCPreferencesPathGetValue(prefs.getRep(), path),
		RETAIN
	);
	if (!iface_dict) {
		return StringRef();
	}
	
	StringRef device_name(
		(CFStringRef)CFDictionaryGetValue(iface_dict, kSCPropNetInterfaceDeviceName),
		RETAIN
	);
	if (!device_name || CFGetTypeID(device_name) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	return device_name;
}

ProxyWatcher::StringPair
ProxyWatcher::findSetReferencingService(
	NetworkPrefs& prefs, StringRef const& service_path)
{
	typedef vector<StringPair> Sets;
	Sets sets;
	DECLARE_INSERTER(SetInserter, StringPair, Sets, c.push_back(val));
	SetInserter set_inserter(sets);
	prefs.enumSetsReferencingService(service_path, set_inserter);
	
	if (sets.size() == 1) {
		return sets[0];
	} else {
		return std::make_pair(StringRef(), StringRef());
	}
}

ProxyWatcher::StringPair
ProxyWatcher::getCurrentSet(NetworkPrefs& prefs)
{
	StringRef cur_set_path(
		(CFStringRef)SCPreferencesGetValue(prefs.getRep(), kSCPrefCurrentSet),
		RETAIN
	);
	if (!cur_set_path || CFGetTypeID(cur_set_path) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	DictRef set(
		SCPreferencesPathGetValue(prefs.getRep(), cur_set_path),
		RETAIN
	);
	if (!set) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	StringRef name(
		(CFStringRef)CFDictionaryGetValue(set, kSCPropUserDefinedName),
		RETAIN
	);
	if (!name || CFGetTypeID(name) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	StringRef set_id = CFUtils::extractLastPathComponent(cur_set_path);
	return std::make_pair(set_id, name);
}

StringRef
ProxyWatcher::findSetByName(NetworkPrefs& prefs, StringRef const& name)
{
	class SetFinder : public OutputFunction<StringPair>
	{
	public:
		SetFinder(StringRef const& set_name)
		: m_setName(set_name), m_setId() {}
		
		virtual void operator()(StringPair const& set) {
			if (!m_setId && CFEqual(set.second, m_setName)) {
				m_setId = set.first;
			}
		}
		
		StringRef const& getSetId() const { return m_setId; }
	private:
		StringRef m_setName;
		StringRef m_setId;
	};
	
	SetFinder set_finder(name);
	prefs.enumSets(set_finder);
	return set_finder.getSetId();
}

StringRef
ProxyWatcher::findServiceBySetAndDeviceName(
	NetworkPrefs& prefs, StringRef const& set_id, StringRef const& device_name)
{
	typedef vector<StringRef> Services;
	Services services;
	DECLARE_INSERTER(ServiceInserter, StringPair, Services, c.push_back(val.second));
	ServiceInserter service_inserter(services);
	prefs.enumServicesReferencedBySet(set_id, service_inserter);
	
	Services::iterator it = services.begin();
	Services::iterator const end = services.end();
	for (; it != end; ++it) {
		StringRef path = CFUtils::composePath(*it, kSCEntNetInterface, 0);
		DictRef iface_dict(
			SCPreferencesPathGetValue(prefs.getRep(), path),
			RETAIN
		);
		if (!iface_dict) {
			continue;
		}
		if (CFGetTypeID(iface_dict) != CFDictionaryGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		CFStringRef dev_name = (CFStringRef)CFDictionaryGetValue(
			iface_dict, kSCPropNetInterfaceDeviceName
		);
		if (dev_name && CFEqual(dev_name, device_name)) {
			return *it;
		}
	}
	
	return StringRef();
}

void
ProxyWatcher::getProxiesFromService(DictRef const& service, StringRef const& set_name)
{
	DictRef proxies(
		(CFDictionaryRef)CFDictionaryGetValue(service, kSCEntNetProxies),
		RETAIN
	);
	if (!proxies || CFGetTypeID(service) != CFDictionaryGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	Forwarding::SystemOption sys_opt(Forwarding::Utf8String(CFUtils::stringToUtf8(set_name)));
	
	if (isFlagSet(proxies,  kSCPropNetProxiesHTTPEnable)) {
		processHttpProxy(proxies, sys_opt);
	} else if (isFlagSet(proxies, kSCPropNetProxiesSOCKSEnable)) {
		processSocksProxy(proxies, sys_opt);
	} else if (isFlagSet(proxies, CFSTR("ProxyAutoDiscoveryEnable"))) {
		processProxyAutoDiscovery(proxies, sys_opt);
	} else if (isFlagSet(proxies, CFSTR("ProxyAutoConfigEnable"))) {
		processProxyAutoConfig(proxies, sys_opt);
	} else {
		// No proxy. Nothing to be done with sys_opt.
	}
	
	GlobalState::WriteAccessor()->forwardingResolver().selectSystemOption(sys_opt);
}

bool
ProxyWatcher::isFlagSet(DictRef const& dict, CFStringRef key)
{
	CFNumberRef val = (CFNumberRef)CFDictionaryGetValue(dict, key);
	if (!val || CFGetTypeID(val) != CFNumberGetTypeID()) {
		return false;
	}
	int flag = 0;
	CFNumberGetValue(val, kCFNumberIntType, &flag);
	return flag != 0;
}

void
ProxyWatcher::processHttpProxy(
	DictRef const& proxies, Forwarding::SystemOption& sys_opt)
{
	CFStringRef host_str = (CFStringRef)CFDictionaryGetValue(
		proxies, kSCPropNetProxiesHTTPProxy
	);
	if (!host_str || CFGetTypeID(host_str) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	CFNumberRef port_num = (CFNumberRef)CFDictionaryGetValue(
		proxies, kSCPropNetProxiesHTTPPort
	);
	if (!port_num || CFGetTypeID(port_num) != CFNumberGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	char host[256];
	if (!CFStringGetCString(host_str, host, sizeof(host), kCFStringEncodingASCII)) {
		throw GenericError("HTTP proxy host contains international characters or is too long");
	}
	
	int port = 0;
	CFNumberGetValue(port_num, kCFNumberIntType, &port);
	
	ProxyDescriptor proxy_desc;
	proxy_desc.setType(ProxyDescriptor::HTTP);
	proxy_desc.setAddr(SymbolicInetAddr(host, port));
	
	sys_opt.proxyChain().push_back(proxy_desc);
	readProxyBypassList(proxies, sys_opt);
}

void
ProxyWatcher::processSocksProxy(
	DictRef const& proxies, Forwarding::SystemOption& sys_opt)
{
	CFStringRef host_str = (CFStringRef)CFDictionaryGetValue(
		proxies, kSCPropNetProxiesSOCKSProxy
	);
	if (!host_str || CFGetTypeID(host_str) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	CFNumberRef port_num = (CFNumberRef)CFDictionaryGetValue(
		proxies, kSCPropNetProxiesSOCKSPort
	);
	if (!port_num || CFGetTypeID(port_num) != CFNumberGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	char host[256];
	if (!CFStringGetCString(host_str, host, sizeof(host), kCFStringEncodingASCII)) {
		throw GenericError("SOCKS proxy host contains international characters or is too long");
	}
	
	int port = 0;
	CFNumberGetValue(port_num, kCFNumberIntType, &port);
	
	ProxyDescriptor proxy_desc;
	proxy_desc.setType(ProxyDescriptor::SOCKS5); // making a guess about SOCKS5
	proxy_desc.setAddr(SymbolicInetAddr(host, port));
	
	sys_opt.proxyChain().push_back(proxy_desc);
	readProxyBypassList(proxies, sys_opt);
}

void
ProxyWatcher::processProxyAutoDiscovery(
	DictRef const& proxies, Forwarding::SystemOption& sys_opt)
{
	sys_opt.setErrorMessage("Unsupported feature: proxy auto discovery");
}

void
ProxyWatcher::processProxyAutoConfig(
	DictRef const& proxies, Forwarding::SystemOption& sys_opt)
{
	sys_opt.setErrorMessage("Unsupported feature: proxy auto configuration");
}

void
ProxyWatcher::readProxyBypassList(
	DictRef const& proxies, Forwarding::SystemOption& sys_opt)
{
	if (isFlagSet(proxies, CFSTR("ExcludeSimpleHostnames"))) {
		sys_opt.bypassList().push_back(
			Forwarding::BypassPatternConstPtr(
				new Forwarding::SimpleHostnamesMatchPattern
			)
		);
	}
	
	CFArrayRef arr = (CFArrayRef)CFDictionaryGetValue(
		proxies, kSCPropNetProxiesExceptionsList
	);
	if (!arr) {
		return;
	}
	if (CFGetTypeID(arr) != CFArrayGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	CFIndex const count = CFArrayGetCount(arr);
	for (CFIndex i = 0; i < count; ++i) {
		CFStringRef item = (CFStringRef)CFArrayGetValueAtIndex(arr, i);
		if (!item || CFGetTypeID(item) != CFStringGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		Forwarding::Utf8String const host_mask(CFUtils::stringToUtf8(item));
		sys_opt.bypassList().push_back(
			Forwarding::BypassPatternConstPtr(
				new Forwarding::HostMaskMatchPattern(host_mask)
			)
		);
	}
}

/**
 * Remember that SystemOption can be overriden by an option in forwarding.xml
 * having the same name.  This function sets an error that can't be overriden.
 */
void
ProxyWatcher::setUnmaskableError(std::string const& msg)
{
	// Options with empty names are not allowed, so nothing can override us.
	Forwarding::Utf8String name;
	Forwarding::SystemOption sys_opt(name);
	sys_opt.setErrorMessage(msg);
	GlobalState::WriteAccessor()->forwardingResolver().selectSystemOption(sys_opt);
}


/*================== ProxyWatcher::GenericError ====================*/

ProxyWatcher::GenericError::GenericError(std::string const& msg)
:	m_msg(msg)
{
}

ProxyWatcher::GenericError::GenericError(char const* msg)
:	m_msg(msg)
{
}

ProxyWatcher::GenericError::~GenericError()
{
}
