/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTTPCACHE_LIMITEDSPACEMANAGER_H_
#define HTTPCACHE_LIMITEDSPACEMANAGER_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "AtomicCounter.h"
#include "IntrusivePtr.h"
#include "IntrusiveList.h"
#include "AbstractCommand.h"
#include "types.h"
#include <ace/config-lite.h>
#include <ace/Basic_Types.h> // for off_t
#include <ace/Synch.h>
#include <map>
#include <assert.h>

namespace HttpCache
{

// Declared later in this file:
class LimitedSpaceObject;
class ResizeRequest;
class GrowRequest;
template<typename Key> class LimitedSpaceManager;

namespace detail
{

class ResizeRequestBase
{
public:
	ResizeRequestBase(off_t requested_size) : m_requestedSize(requested_size) {}
	
	off_t getRequestedSize() const { return m_requestedSize; } 
private:
	off_t const m_requestedSize;
};


/**
 * \brief This interface is a way for LimitedSpaceObject to access its
 *        LimitedSpaceManager.
 */
class ManagerHandle
{
public:	
	virtual ~ManagerHandle() {}
	
	virtual void ref() const = 0;
	
	virtual void unref() const = 0;
	
	/**
	 * \brief Request permission for resizing an object.
	 * \return true if permission is granted, false otherwise.
	 *
	 * If permission is granted, commitResize() must follow after the
	 * actual resizing took place.
	 */
	virtual bool requestResize(
		LimitedSpaceObject& obj, ResizeRequestBase& request,
		IntrusiveListNode& node, off_t requested_size) = 0;
	
	/**
	 * \brief Complete the resizing operation.
	 *
	 * If resizing didn't take place, -1 is to be passed as \p new_size.
	 * Otherwise, \p new_size must be less or equal to the requested size.
	 */
	virtual void commitResize(
		LimitedSpaceObject& obj, ResizeRequestBase& request,
		IntrusiveListNode& node, off_t new_size, bool only_grow) = 0;
	
	/**
	 * \brief Called when an erased object looses its last reference.
	 */
	virtual void onGhostGone(off_t confirmed_size) = 0;
};

} // namespace detail


class LimitedSpaceObject
{
	friend class ResizeRequest;
	friend class GrowRequest;
	friend class LimitedSpaceManagerBase;
	template<typename Key> friend class LimitedSpaceManager;
	DECLARE_NON_COPYABLE(LimitedSpaceObject)
public:
	void ref() const { ++m_numRefs; }
	
	void unref() const;
	
	~LimitedSpaceObject();
private:
	typedef IntrusiveList<detail::ResizeRequestBase const> ReservationList;
	
	LimitedSpaceObject(detail::ManagerHandle* manager_handle, off_t size);
	
	off_t getConfirmedSize() const { return m_confirmedSize; }
	
	void setConfirmedSize(off_t size) { m_confirmedSize = size; }
	
	off_t getPotentialSize() const;
	
	off_t getPotentialSizeWith(off_t size_request) const;
	
	bool hasExactlyOneRef() const { return (m_numRefs += 0) == 1; }
	
	ReservationList& reservations() { return m_reservations; }
	
	void addReservation(detail::ResizeRequestBase& request, IntrusiveListNode& node);
	
	/**
	 * \brief Called when an object is being erased, but still have references.
	 * \note This method may be called only once.
	 */
	void becomeGhost() { m_numRefs += m_sGhostOffset; }
	
	bool requestResize(detail::ResizeRequestBase& request,
		IntrusiveListNode& node, off_t requested_size) {
		return m_ptrManagerHandle->requestResize(*this, request, node, requested_size);
	}
	
	void commitResize(detail::ResizeRequestBase& request,
		IntrusiveListNode& node, off_t new_size, bool only_grow) {
		m_ptrManagerHandle->commitResize(*this, request, node, new_size, only_grow);
	}
	
	static int32_t const m_sGhostOffset = int32_t(1) << (sizeof(int32_t) * 8 - 1);
	off_t m_confirmedSize;
	IntrusivePtr<detail::ManagerHandle> m_ptrManagerHandle;
	
	/**
	 * \brief Granted resize requests ordered by requested size, smallest first.
	 */
	ReservationList m_reservations;
	
	mutable AtomicCounter<ACE_MT_SYNCH> m_numRefs;
};


class LimitedSpaceManagerBase : private detail::ManagerHandle
{
	template<typename Key> friend class LimitedSpaceManager;
	DECLARE_NON_COPYABLE(LimitedSpaceManagerBase)
public:
	enum CollisionResolution { ERROR_IF_EXISTS, ERASE_EXISTING };
	
	explicit LimitedSpaceManagerBase(
		uint64_t space_limit, IntrusivePtr<AbstractCommand> const& gc_invoker);
	
	virtual ~LimitedSpaceManagerBase();
	
	virtual void ref() const;
	
	virtual void unref() const;
	
	/**
	 * \brief Change the space limit.
	 *
	 * If we lower the limit so that the used space exceedes the new limit,
	 * it won't cause any objects to be erased.  However, when an object
	 * tries to grow further, the garbage collector will be invoked.  So, over
	 * time the used space will shrink to fit in the new limit.
	 */
	virtual void setSpaceLimit(uint64_t limit);
private:
	typedef ACE_Thread_Mutex Mutex;
	
	virtual bool requestResize(
		LimitedSpaceObject& obj, detail::ResizeRequestBase& request,
		IntrusiveListNode& node, off_t requested_size);
	
	virtual void commitResize(
		LimitedSpaceObject& obj, detail::ResizeRequestBase& request,
		IntrusiveListNode& node, off_t new_size, bool only_grow);
		
	virtual void onGhostGone(off_t confirmed_size);
	
	mutable AtomicCounter<ACE_MT_SYNCH> m_refCounter;
	mutable Mutex m_mutex;
	IntrusivePtr<AbstractCommand> const m_ptrGcInvoker;
	uint64_t m_spaceLimit;
	
	/**
	 * Used space is a sum of potential sizes of all objects belonging to this
	 * manager, including the ones erased but still referenced.
	 * The potential size is max(confirmed_size, max(pending_resize_requests)).
	 */
	uint64_t m_usedSpace;
};


template<typename Key>
class LimitedSpaceManager : public LimitedSpaceManagerBase
{
	DECLARE_NON_COPYABLE(LimitedSpaceManager)
private:
	/**
	 * \brief Constructor is private. Use create() instead.
	 */
	explicit LimitedSpaceManager(
		uint64_t space_limit, IntrusivePtr<AbstractCommand> const& gc_invoker);
public:
	virtual ~LimitedSpaceManager();
	
	/**
	 * \brief Creates a new instance.
	 *
	 * \note \p gc_invoker may be null.
	 */
	static IntrusivePtr<LimitedSpaceManager> create(
		uint64_t space_limit, IntrusivePtr<AbstractCommand> const& gc_invoker);
	
	/**
	 * \brief Create a new object with a given size.
	 * \return A new object on success, null on failure.
	 *
	 * A failure may be caused by exceeding the space limit, or by trying to
	 * create an object that already exists.
	 * \note Creating an object with zero size will not fail even if
	 *       used space already exceedes the space limit.
	 * \see setSpaceLimit()
	 */
	IntrusivePtr<LimitedSpaceObject> newObject(Key const& key, off_t size,
		CollisionResolution collision_resolution = ERROR_IF_EXISTS);
	
	/**
	 * \brief Open an existing object.
	 * \return A pointer to object on success or a null pointer if the object
	 *         doesn't exist.
	 */
	IntrusivePtr<LimitedSpaceObject> openObject(Key const& key);
	
	/**
	 * \brief Erase an object.
	 *
	 * It is possible to erase an object that still has references.
	 * In this case, the object will still exist and take space, but
	 * it won't be associated with a key anymore, so a new object with the
	 * same key can be created. Naturally, the space taken by such an object
	 * will be freed when all of its references are gone.
	 */
	void eraseObject(Key const& key);
	
	/**
	 * \brief Output all stored keys in their natural order.
	 *
	 * The \p inserter will be used like this:\n
	 * *inserter = key;\n
	 * ++inserter;
	 */
	template<typename Inserter>
	void listKeys(Inserter inserter) const;
private:
	typedef std::map<Key, IntrusivePtr<LimitedSpaceObject> > ObjectMap;
	
	void eraseLocked(typename ObjectMap::iterator const& it);
	
	ObjectMap m_objects;
};


class ResizeRequest : private detail::ResizeRequestBase
{
	DECLARE_NON_COPYABLE(ResizeRequest)
public:
	/**
	 * \brief Request resizing an object.
	 *
	 * If we work with files, ResizeRequest is to be used for truncating,
	 * including tuncating to zero and truncating past the end of the file.
	 * Note that even if you think you are shrinking a file, you may in fact
	 * be extending it, because it was shrinked even more through a different
	 * handle. The same is true the other way around. Because of this, resizing
	 * files should be synchronized with each other and with writes.  Consider
	 * this example:\n
	 * Thread A is writing at the end of a file.\n
	 * Thread B is truncating the same file.\n
	 * The file size after both operations are finished depends on which
	 * operation took place first.
	 */
	ResizeRequest(IntrusivePtr<LimitedSpaceObject> const& obj, off_t requested_size);
	
	/**
	 * \brief The destructor commits the new object size.
	 */
	~ResizeRequest();
	
	/**
	 * \brief Check whether the request was granted or not.
	 */
	bool requestGranted() const { return m_requestGranted; }
	
	/**
	 * \brief Confirm the successful resizing.
	 *
	 * If not called, it is assumed that no resizing took place.
	 */
	void confirm() { m_confirmed = true; }
private:
	IntrusiveListNode m_listNode;
	IntrusivePtr<LimitedSpaceObject> m_ptrObj;
	bool m_requestGranted;
	bool m_confirmed;
};


class GrowRequest : private detail::ResizeRequestBase
{
	DECLARE_NON_COPYABLE(GrowRequest)
public:
	/**
	 * \brief Reserves space for possible object growth.
	 *
	 * If we work with files, GrowRequest is to be used with writes, including
	 * overwrites.  As for overwrites, you can never be sure you are not
	 * extending the file, as it may have been truncated through another handle.
	 */
	GrowRequest(IntrusivePtr<LimitedSpaceObject> const& obj, off_t requested_size);
	
	/**
	 * \brief The destructor commits the new object size.
	 */
	~GrowRequest();
	
	/**
	 * \brief Check whether the request was granted or not.
	 */
	bool requestGranted() const { return m_requestGranted; }
	
	/**
	 * \brief Set the new size of the object.
	 *
	 * If not called, it is assumed that the size didn't change.
	 * The new size may be less or equal to the requested size.
	 * If the request was not granted, this method must not be called at all.
	 */
	void setNewSize(off_t size) { m_newSize = size; }
private:
	off_t m_newSize;
	IntrusiveListNode m_listNode;
	IntrusivePtr<LimitedSpaceObject> m_ptrObj;
	bool m_requestGranted;
};


template<typename Key>
LimitedSpaceManager<Key>::LimitedSpaceManager(
	uint64_t const space_limit, IntrusivePtr<AbstractCommand> const& gc_invoker)
:	LimitedSpaceManagerBase(space_limit, gc_invoker)
{
}

template<typename Key>
LimitedSpaceManager<Key>::~LimitedSpaceManager()
{
}

template<typename Key>
inline IntrusivePtr<LimitedSpaceManager<Key> >
LimitedSpaceManager<Key>::create(
	uint64_t const space_limit, IntrusivePtr<AbstractCommand> const& gc_invoker)
{
	typedef LimitedSpaceManager<Key> Manager;
	return IntrusivePtr<Manager>(new Manager(space_limit, gc_invoker));
}

template<typename Key>
IntrusivePtr<LimitedSpaceObject>
LimitedSpaceManager<Key>::newObject(
	Key const& key, off_t const object_size,
	CollisionResolution const collision_resolution)
{
	assert(object_size >= 0);
	
	ACE_Guard<Mutex> guard(m_mutex);
	
	if (object_size > 0 && m_usedSpace + object_size > m_spaceLimit) {
		return IntrusivePtr<LimitedSpaceObject>();
	}
	
	typename ObjectMap::iterator const it(m_objects.lower_bound(key));
	if (it != m_objects.end() && !m_objects.key_comp()(key, it->first)) {
		// Object already exists.
		if (collision_resolution == ERROR_IF_EXISTS) {
			return IntrusivePtr<LimitedSpaceObject>();
		} else { // ERASE_EXISTING
			assert(collision_resolution == ERASE_EXISTING);
			eraseLocked(it);
		}
	}
	
	IntrusivePtr<LimitedSpaceObject> const obj(
		new LimitedSpaceObject(this, object_size)
	);
	m_objects.insert(it, typename ObjectMap::value_type(key, obj));
	m_usedSpace += object_size;
	
	return obj;
}

template<typename Key>
IntrusivePtr<LimitedSpaceObject>
LimitedSpaceManager<Key>::openObject(Key const& key)
{	
	ACE_Guard<Mutex> guard(m_mutex);
	
	typename ObjectMap::iterator const it(m_objects.find(key));
	if (it != m_objects.end()) {
		return it->second;
	}
	
	return IntrusivePtr<LimitedSpaceObject>();
}

template<typename Key>
void
LimitedSpaceManager<Key>::eraseObject(Key const& key)
{
	ACE_Guard<Mutex> guard(m_mutex);
	
	typename ObjectMap::iterator const it(m_objects.find(key));
	if (it == m_objects.end()) {
		return;
	}
	
	eraseLocked(it);
}

template<typename Key>
template<typename Inserter>
void
LimitedSpaceManager<Key>::listKeys(Inserter inserter) const
{
	ACE_Guard<Mutex> guard(m_mutex);
	
	typename ObjectMap::const_iterator it(m_objects.begin());
	typename ObjectMap::const_iterator const end(m_objects.end());
	for (; it != end; ++it) {
		*inserter = it->first;
		++inserter;
	}
}

template<typename Key>
void
LimitedSpaceManager<Key>::eraseLocked(typename ObjectMap::iterator const& it)
{	
	if (it->second->hasExactlyOneRef()) {
		// The only reference is from m_objects.
		m_usedSpace -= it->second->getConfirmedSize();
	} else {
		it->second->becomeGhost();
	}
	m_objects.erase(it);
}

} // namespace HttpCache

#endif
