/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Forwarding.h"
#include "HtmlEscaper.h"
#include "XmlLexer.h"
#include "BString.h"
#include "BStringPOD.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "StringUtils.h"
#include "ConfErrorHandler.h"
#include "ConfError.h"
#include "SymbolicInetAddr.h"
#include "IPv4SubnetParser.h"
#include "URI.h"
#include "ArraySize.h"
#include "types.h"
#include <ace/OS_NS_arpa_inet.h> // for ACE_OS::inet_aton()
#include <boost/lexical_cast.hpp>
#include <istream>
#include <ostream>
#include <vector>
#include <algorithm>
#include <assert.h>
#include <stddef.h>
#include <ctype.h>

// for ntohl()
#ifdef _WIN32
#include <winsock2.h>
#else
#include <arpa/inet.h>
#endif

namespace Forwarding
{

namespace impl {}
using namespace impl;


/*============================= impl::Indent ===========================*/

namespace impl
{

class Indent
{
public:
	explicit Indent(int level) : m_level(level) {}
	
	int getLevel() const { return m_level; }
private:
	int m_level;
};

std::ostream& operator<<(std::ostream& strm, Indent const& indent)
{
	for (int i = 0; i < indent.getLevel(); ++i) {
		strm << '\t';
	}
	return strm;
}

} // namespace impl


/*========================== impl::DomainMatcher =======================*/

namespace impl
{

class DomainMatcher : public Matcher
{
public:
	DomainMatcher(BString const& domain) : m_domain(domain) {}
	
	virtual bool matches(URI const& uri) const;
private:
	BString const m_domain;
};


bool
DomainMatcher::matches(URI const& uri) const
{
	BString host(uri.getHost());
	
	if (!host.empty() && host.end()[-1] == '.') {
		// absolute domain like "www.slashdot.org."
		host.trimBack(1);
	}
	
	bool const ends_with = StringUtils::ciEndsWith(
		host.begin(), host.end(), m_domain.begin(), m_domain.end()
	);
	if (ends_with) {
		if (host.size() == m_domain.size()) {
			// exact match
			return true;
		}
		if (host[host.size() - m_domain.size() - 1] == '.') {
			// subdomain
			return true;
		}
	}
	return false;
}

} // namespace impl


/*========================== DomainMatchPattern ========================*/

MatcherConstPtr
DomainMatchPattern::getMatcher() const
{
	return MatcherConstPtr(new DomainMatcher(BString(m_pattern.raw())));
}

void
DomainMatchPattern::toStream(std::ostream& strm, int indent) const
{
	strm << Indent(indent) << "<domain>";
	strm << HtmlEscaper::escape(m_pattern.raw()) << "</domain>\n";
}


/*======================== impl::IPv4SubnetMatcher =====================*/

namespace impl
{

class IPv4SubnetMatcher : public Matcher
{
public:
	IPv4SubnetMatcher(uint32_t net_addr, unsigned net_bits);
	
	virtual bool matches(URI const& uri) const;
private:
	uint32_t m_netAddr;
	unsigned m_netBits;
};


IPv4SubnetMatcher::IPv4SubnetMatcher(uint32_t net_addr, unsigned net_bits)
:	m_netAddr(net_addr),
	m_netBits(net_bits)
{
}

bool
IPv4SubnetMatcher::matches(URI const& uri) const
{
	struct in_addr addr;
	if (!ACE_OS::inet_aton(uri.getHost().toStdString().c_str(), &addr)) {
		// Not an IPv4 address.
		return false;
	}
	
	uint32_t const ip = ntohl(addr.s_addr);
	uint32_t const mask = ~uint32_t(0) << (32 - m_netBits);
	return ((m_netAddr ^ ip) & mask) == 0;
}

} // namespace impl


/*========================== SubnetMatchPattern ========================*/

MatcherConstPtr
SubnetMatchPattern::getMatcher() const
{
	MatcherConstPtr matcher;
	IPv4SubnetParser parser;
	if (parser.parse(m_pattern.raw())) {
		matcher.reset(new IPv4SubnetMatcher(
			parser.getNetAddr(), parser.getNetBits()
		));
	}
	return matcher;
}

void
SubnetMatchPattern::toStream(std::ostream& strm, int indent) const
{
	strm << Indent(indent) << "<subnet>";
	strm << HtmlEscaper::escape(m_pattern.raw()) << "</subnet>\n";
}


/*========================= impl::HostMaskMatcher ======================*/

namespace impl
{

class HostMaskMatcher : public Matcher
{
public:
	HostMaskMatcher(BString const& mask);
	
	virtual bool matches(URI const& uri) const;
private:
	static bool trimPrefix(BString& host, BString const& prefix);
	
	static bool trimSuffix(BString& host, BString const& suffix);
	
	static bool trimNearestMatch(BString& host, BString const& text);
	
	/**
	 * \brief Components of host mask separated by *.
	 *
	 * A token can't be empty, but there may be no tokens.
	 */
	std::vector<BString> m_tokens;
	
	/**
	 * \brief Does the host mask begin with a wildcard?
	 */
	bool m_leadingWildcard;
	
	/**
	 * \brief Does the host mask end with a wildcard?
	 * \note If the host consists of a single wildcard, then
	 *       m_leadingWildcard is true and m_trailingWildcard is false.
	 */
	bool m_trailingWildcard;
};


HostMaskMatcher::HostMaskMatcher(BString const& mask)
:	m_leadingWildcard(false),
	m_trailingWildcard(false)
{
	if (mask.empty()) {
		return;
	}
	
	m_leadingWildcard = (mask.begin()[0] == '*');
	if (mask.size() == 1) {
		return;
	}
	m_trailingWildcard = (mask.end()[-1] == '*');
	
	char const* p = mask.begin();
	char const* const end = mask.end();
	char const* ap = 0;
	for (;; p = ap + 1) {
		ap = StringUtils::find(p, end, '*');
		if (p != ap) {
			// insert text before asterisk
			m_tokens.push_back(BString(mask, p, ap));
		}
		if (ap == end) {
			break;
		}
	}
}

bool
HostMaskMatcher::matches(URI const& uri) const
{
	BString host(uri.getHost());
	
	if (!host.empty() && host.end()[-1] == '.') {
		// absolute domain like "www.slashdot.org."
		host.trimBack(1);
	}
	
	if (m_tokens.empty()) {
		return m_leadingWildcard;
	}
	
	BString const* pos = &m_tokens[0];
	BString const* end = pos + m_tokens.size();
	
	if (!m_leadingWildcard) {
		if (!trimPrefix(host, pos[0])) {
			return false;
		} else {
			++pos;
		}
	}
	
	if (!m_trailingWildcard && pos != end) {
		if (!trimSuffix(host, end[-1])) {
			return false;
		} else {
			--end;
		}
	}
	
	for (; pos != end; ++pos) {
		if (!trimNearestMatch(host, pos[0])) {
			return false;
		}
	}
	
	return true;
}

bool
HostMaskMatcher::trimPrefix(BString& host, BString const& prefix)
{
	assert(!prefix.empty());
	
	if (StringUtils::ciStartsWith(host.begin(), host.end(), prefix.begin(), prefix.end())) {
		host.trimFront(prefix.size());
		return true;
	} else {
		return false;
	}
}

bool
HostMaskMatcher::trimSuffix(BString& host, BString const& suffix)
{
	assert(!suffix.empty());
	
	if (StringUtils::ciEndsWith(host.begin(), host.end(), suffix.begin(), suffix.end())) {
		host.trimBack(suffix.size());
		return true;
	} else {
		return false;
	}
}

bool
HostMaskMatcher::trimNearestMatch(BString& host, BString const& text)
{
	assert(!text.empty());
	
	if (text.size() > host.size()) {
		return false;
	}
	
	char const first_lc = tolower(text.begin()[0]);
	char const first_uc = toupper(text.begin()[0]);
	char const* h = host.begin();
	char const* h_end = host.end() - text.size() + 1;
	for (; h != h_end; ++h) {
		if (*h == first_lc || *h == first_uc) {
			if (StringUtils::ciStartsWith(h, host.end(), text.begin(), text.end())) {
				host.trimFront(h - host.begin() + text.size());
				return true;
			}
		}
	}
	return false;
}

} // namespace impl


/*========================= HostMaskMatchPattern =======================*/

MatcherConstPtr
HostMaskMatchPattern::getMatcher() const
{
	return MatcherConstPtr(new HostMaskMatcher(BString(m_pattern.raw())));
}

void
HostMaskMatchPattern::accept(BypassVisitor& visitor) const
{
	visitor.visit(*this);
}

void
HostMaskMatchPattern::toStream(std::ostream& strm, int indent) const
{
	strm << Indent(indent) << "<host-mask>";
	strm << HtmlEscaper::escape(m_pattern.raw()) << "</host-mask>\n";
}


/*===================== impl::SimpleHostnamesMatcher ===================*/

namespace impl
{

class SimpleHostnamesMatcher : public Matcher
{
public:
	virtual bool matches(URI const& uri) const;
};

} // namespace impl


bool
SimpleHostnamesMatcher::matches(URI const& uri) const
{
	BString const& host = uri.getHost();
	return StringUtils::find(host.begin(), host.end(), '.') == host.end();
}


/*===================== SimpleHostnamesMatchPattern ====================*/

MatcherConstPtr
SimpleHostnamesMatchPattern::getMatcher() const
{
	return MatcherConstPtr(new SimpleHostnamesMatcher);
}

void
SimpleHostnamesMatchPattern::accept(BypassVisitor& visitor) const
{
	visitor.visit(*this);
}

void
SimpleHostnamesMatchPattern::toStream(std::ostream& strm, int indent) const
{
	strm << Indent(indent) << "<simple-hostnames/>\n";
};


/*=========================== impl::XmlLoader ==========================*/

namespace impl
{

/**
 * \brief Parser for forwarding.xml file.
 */
class XmlLoader : private XmlLexer
{
public:
	XmlLoader();
	
	virtual ~XmlLoader();
	
	Config& config() { return m_config; }
	
	bool load(std::istream& strm, ConfErrorHandler& eh);
private:
	virtual void processProcInstNode(
		Iterator const& begin, Iterator const& end) {}
	
	virtual void processDocTypeNode(
		Iterator const& begin, Iterator const& end) {}
	
	virtual void processTextNode(
		Iterator const& begin, Iterator const& end);
	
	virtual void processCDataNode(
		Iterator const& begin, Iterator const& end);
	
	virtual void processCommentNode(
		Iterator const& begin, Iterator const& end) {}
	
	virtual void processOpeningTagName(
		Iterator const& begin, Iterator const& end);
	
	virtual void processAttrName(
		Iterator const& begin, Iterator const& end);
	
	virtual void processAttrValueWithQuotes(
		Iterator const& begin, Iterator const& end);
	
	virtual void processOpeningTag(bool empty_tag);
	
	virtual void processClosingTag(
		Iterator const& begin, Iterator const& end);
	
	virtual void processParseError(Position const& pos);
	
	void handleText(Iterator const& begin, Iterator const& end);
	
	void handleClosingTag(BString const& name, std::string const& text);
	
	void handleOpeningForwarding(Position const& pos);
	
	void handleOpeningOption(Position const& pos);
	
	void handleOpeningBypass(Position const& pos);
	
	void handleOpeningProxyChain(Position const& pos);
	
	void handleOpeningProxy(Position const& pos);
	
	void handleOpeningType(Position const& pos);
	
	void handleOpeningHost(Position const& pos);
	
	void handleOpeningPort(Position const& pos);
	
	void handleOpeningUser(Position const& pos);
	
	void handleOpeningPass(Position const& pos);
	
	void handleClosingForwarding(std::string const& text);
	
	void handleClosingOption(std::string const& text);
	
	void handleClosingHostMask(std::string const& text);
	
	void handleClosingSimpleHostnames(std::string const& text);
	
	void handleClosingProxy(std::string const& text);
	
	void handleClosingType(std::string const& text);
	
	void handleClosingHost(std::string const& text);
	
	void handleClosingPort(std::string const& text);
	
	void handleClosingUser(std::string const& text);
	
	void handleClosingPass(std::string const& text);
	
	void ensureOptionNamesUnique();
	
	static Utf8String mutateName(Utf8String const& name);
	
	bool isContentAllowedHere() const;
	
	bool isTagAllowedHere(BString const& tag) const;
	
	void issueError(std::string const& msg, int line = 0, int col = 0);
	
	void issueWarning(std::string const& msg, int line = 0, int col = 0);
	
	ConfErrorHandler* m_pErrorHandler;
	BString m_curTagName;
	BString m_curAttrName;
	int m_curTagLine; // zero based
	SplittableBuffer m_text;
	std::vector<BString> m_parentTags;
	// m_parentTags[m_ignoreFrom] and beyond are to be ignored
	int m_ignoreFrom; // -1 when not ignoring
	Config m_config;
	Option m_option;
	ProxyDescriptor m_proxyDescriptor;
};


XmlLoader::XmlLoader()
:	m_pErrorHandler(0),
	m_curTagLine(0),
	m_ignoreFrom(-1),
	m_option(Utf8String())
{
}

XmlLoader::~XmlLoader()
{
}

bool
XmlLoader::load(std::istream& strm, ConfErrorHandler& eh)
{
	m_pErrorHandler = &eh;
	SBOutStream buffer;
	while (!isAborted()) {
		buffer.readChunk(strm);
		consume(buffer.data(), false);
		if (strm.bad()) {
			abort();
			issueError("Read error");
			break;
		}
		if (strm.eof()) {
			consume(buffer.data(), true);
			break;
		}
	}
	return !isAborted();
}

void
XmlLoader::processTextNode(
	Iterator const& begin, Iterator const& end)
{
	if (m_ignoreFrom != -1) {
		return;
	}
	
	handleText(begin, end);
}

void
XmlLoader::processCDataNode(
	Iterator const& begin, Iterator const& end)
{
	if (m_ignoreFrom != -1) {
		return;
	}
	
	BString const cdata_start("<![CDATA[");
	BString const cdata_end("]]>");
	Iterator const text_begin(begin + cdata_start.size());
	Iterator const text_end(end - cdata_end.size());
	handleText(text_begin, text_end);
}

namespace
{

struct OpeningTagHandler
{
	typedef void (XmlLoader::*Fp)(XmlLexer::Position const&);
	
	BStringPOD tag;
	Fp func;
	
	operator BString() const { return tag; }
};

}

void
XmlLoader::processOpeningTagName(
	Iterator const& begin, Iterator const& end)
{
	m_curTagName = SplittableBuffer::toBString(begin, end);
	Position pos(tokenPosition(begin));
	m_curTagLine = pos.line;
	
	if (m_ignoreFrom != -1) {
		return;
	}
	
	if (!isTagAllowedHere(m_curTagName)) {
		issueWarning(
			"Unexpected tag: "+m_curTagName.toStdString(),
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
		return;
	}
	
	static OpeningTagHandler const handlers[] = {
		{ { "forwarding" }, &XmlLoader::handleOpeningForwarding },
		{ { "option" }, &XmlLoader::handleOpeningOption },
		{ { "bypass" }, &XmlLoader::handleOpeningBypass },
		{ { "proxy-chain" }, &XmlLoader::handleOpeningProxyChain },
		{ { "proxy" }, &XmlLoader::handleOpeningProxy },
		{ { "type" }, &XmlLoader::handleOpeningType },
		{ { "host" }, &XmlLoader::handleOpeningHost },
		{ { "port" }, &XmlLoader::handleOpeningPort },
		{ { "user" }, &XmlLoader::handleOpeningUser },
		{ { "pass" }, &XmlLoader::handleOpeningPass }
	};
	
	OpeningTagHandler const* const handlers_end = handlers + ARRAY_SIZE(handlers);
	OpeningTagHandler const* handler = std::find(handlers, handlers_end, m_curTagName);
	if (handler != handlers_end) {
		(this->*handler->func)(pos);
	}
}

void
XmlLoader::processAttrName(
	Iterator const& begin, Iterator const& end)
{
	if (m_ignoreFrom != -1) {
		return;
	}
	
	m_curAttrName = SplittableBuffer::toBString(begin, end);
}

void
XmlLoader::processAttrValueWithQuotes(
	Iterator const& begin, Iterator const& end)
{
	if (m_ignoreFrom != -1) {
		return;
	}
	
	BString name;
	m_curAttrName.swap(name);
	BString value(SplittableBuffer::toBString(begin, end));
	value.trimFront(1);
	value.trimBack(1);
	value = HtmlEscaper::unescape(value);
	
	bool bad_attr_name = false;
	
	if (m_curTagName == BString("option")) {
		if (name == BString("name")) {
			m_option.setName(Utf8String(value.toStdString()));
		} else if (name == BString("selected")) {
			m_option.setSelected();
		} else {
			bad_attr_name = true;
		}
	} else {
		bad_attr_name = true;
	}
	
	if (bad_attr_name) {
		issueWarning(
			"Unexpected attribute \""+name.toStdString()+"\"",
			tokenPosition(begin).line + 1
		);
	}
}

void
XmlLoader::processOpeningTag(bool empty_tag)
{
	BString name;
	name.swap(m_curTagName);
	SplittableBuffer text;
	text.appendDestructive(m_text);
	
	if (empty_tag) {
		handleClosingTag(name, text.toString());
	} else {
		m_parentTags.push_back(name);
	}
}

void
XmlLoader::processClosingTag(
	Iterator const& begin, Iterator const& end)
{
	BString name(SplittableBuffer::toBString(begin, end));
	SplittableBuffer text;
	text.appendDestructive(m_text);
	
	if (m_parentTags.empty() || m_parentTags.back() != name) {
		Position pos(tokenPosition(begin));
		issueError("Parse error: unmatched tag", pos.line + 1, pos.col + 1);
		abort();
		return;
	}
	
	m_parentTags.pop_back();
	
	m_curTagLine = tokenPosition(begin).line;
	handleClosingTag(name, text.toString());
}

void
XmlLoader::processParseError(Position const& pos)
{
	issueError("Parse error", pos.line + 1, pos.col + 1);
	abort();
}

void
XmlLoader::handleText(
	Iterator const& begin, Iterator const& end)
{
	m_text.append(begin, end);
	
	Iterator const first_non_space(StringUtils::ltrim(begin, end));
	if (first_non_space != end) {
		if (!isContentAllowedHere()) {
			Position pos(tokenPosition(first_non_space));
			issueWarning(
				"Unexpected character",
				pos.line + 1, pos.col + 1
			);
		}
	}
}

namespace
{

struct ClosingTagHandler
{
	typedef void (XmlLoader::*Fp)(std::string const&);
	
	BStringPOD tag;
	Fp func;
	
	operator BString() const { return tag; }
};

}

void
XmlLoader::handleClosingTag(
	BString const& name, std::string const& text)
{
	if (m_ignoreFrom != -1) {
		if (m_ignoreFrom == (int)m_parentTags.size()) {
			// closing the last ignored tag
			m_ignoreFrom = -1;
		}
		return;
	}
	
	static ClosingTagHandler const handlers[] = {
		{ { "forwarding" }, &XmlLoader::handleClosingForwarding },
		{ { "option" }, &XmlLoader::handleClosingOption },
		{ { "host-mask" }, &XmlLoader::handleClosingHostMask },
		{ { "simple-hostnames" }, &XmlLoader::handleClosingSimpleHostnames },
		{ { "proxy" }, &XmlLoader::handleClosingProxy },
		{ { "type" }, &XmlLoader::handleClosingType },
		{ { "host" }, &XmlLoader::handleClosingHost },
		{ { "port" }, &XmlLoader::handleClosingPort },
		{ { "user" }, &XmlLoader::handleClosingUser },
		{ { "pass" }, &XmlLoader::handleClosingPass }
	};
	
	ClosingTagHandler const* const handlers_end = handlers + ARRAY_SIZE(handlers);
	ClosingTagHandler const* handler = std::find(handlers, handlers_end, name);
	if (handler != handlers_end) {
		(this->*handler->func)(text);
	}
}

void
XmlLoader::handleOpeningForwarding(Position const& pos)
{
	if (!m_config.options().empty()) {
		issueError("Multiple <forwarding> tags", pos.line + 1, pos.col + 1);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleOpeningOption(Position const&)
{
	(void)Option(Utf8String()).swap(m_option);
}

void
XmlLoader::handleOpeningBypass(Position const& pos)
{
	if (!m_option.bypassList().empty()) {
		issueError(
			"Multiple <bypass> tags for a single <option>",
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleOpeningProxyChain(Position const& pos)
{
	if (!m_option.proxyChain().empty()) {
		issueError(
			"Multiple <proxy-chain> tags for a single <option>",
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleOpeningProxy(Position const&)
{
	ProxyDescriptor().swap(m_proxyDescriptor);
}

void
XmlLoader::handleOpeningType(Position const& pos)
{
	if (m_proxyDescriptor.getType() != ProxyDescriptor::INVALID) {
		issueError(
			"Multiple <type> tags for a single <proxy>",
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleOpeningHost(Position const& pos)
{
	if (!m_proxyDescriptor.getAddr().getHost().empty()) {
		issueError(
			"Multiple <host> tags for a single <proxy>",
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleOpeningPort(Position const& pos)
{
	ProxyDescriptor pdesc;
	if (m_proxyDescriptor.getAddr().getPort() != pdesc.getAddr().getPort()) {
		issueError(
			"Multiple <port> tags for a single <proxy>",
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleOpeningUser(Position const& pos)
{
	if (!m_proxyDescriptor.getUserName().empty()) {
		issueError(
			"Multiple <user> tags for a single <proxy>",
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleOpeningPass(Position const& pos)
{
	if (!m_proxyDescriptor.getPassword().empty()) {
		issueError(
			"Multiple <pass> tags for a single <proxy>",
			pos.line + 1, pos.col + 1
		);
		m_ignoreFrom = m_parentTags.size();
	}
}

void
XmlLoader::handleClosingForwarding(std::string const&)
{
	ensureOptionNamesUnique();
}

void
XmlLoader::handleClosingOption(std::string const&)
{
	m_config.options().push_back(m_option);
}

void
XmlLoader::handleClosingHostMask(std::string const& text)
{
	BypassPatternConstPtr pattern(
		new HostMaskMatchPattern(Utf8String(text))
	);
	m_option.bypassList().push_back(pattern);
	if (!pattern->getMatcher()) {
		issueError("Invaid syntax for <host-mask>", m_curTagLine + 1);
	}
}

void
XmlLoader::handleClosingSimpleHostnames(std::string const&)
{
	m_option.bypassList().push_back(
		BypassPatternConstPtr(new SimpleHostnamesMatchPattern)
	);
}

void
XmlLoader::handleClosingProxy(std::string const& text)
{
	m_option.proxyChain().push_back(m_proxyDescriptor);
}

void
XmlLoader::handleClosingType(std::string const& text)
{
	m_proxyDescriptor.setType(ProxyDescriptor::resolveType(text));
	if (m_proxyDescriptor.getType() == ProxyDescriptor::INVALID) {
		issueError("Invalid proxy type", m_curTagLine + 1);
	}
}

void
XmlLoader::handleClosingHost(std::string const& text)
{
	SymbolicInetAddr addr(m_proxyDescriptor.getAddr());
	addr.setHost(text);
	m_proxyDescriptor.setAddr(addr);
}

void
XmlLoader::handleClosingPort(std::string const& text)
{
	try {
		uint16_t port = boost::lexical_cast<uint16_t>(text);
		SymbolicInetAddr addr(m_proxyDescriptor.getAddr());
		addr.setPort(port);
		m_proxyDescriptor.setAddr(addr);
	} catch (boost::bad_lexical_cast const&) {
		issueError("Invalid proxy port", m_curTagLine + 1);
	}
}

void
XmlLoader::handleClosingUser(std::string const& text)
{
	m_proxyDescriptor.setUserName(text);
}

void
XmlLoader::handleClosingPass(std::string const& text)
{
	m_proxyDescriptor.setPassword(text);
}

namespace
{

class OptionNameIs
{
public:
	OptionNameIs(Utf8String const& name) : m_name(name) {}
	
	bool operator()(Option const& opt) const {
		return opt.getName() == m_name;
	}
private:
	Utf8String m_name;
};

}

void
XmlLoader::ensureOptionNamesUnique()
{
	OptionList& options = m_config.options();
	OptionList::iterator const begin(options.begin());
	OptionList::iterator const end(options.end());
	OptionList::iterator it(begin);
	for (; it != end; ++it) {
		while (std::find_if(begin, it, OptionNameIs(it->getName())) != it) {
			it->setName(mutateName(it->getName()));
		}
	}
}

Utf8String
XmlLoader::mutateName(Utf8String const& name)
{
	// "Str" becomes "Str 1", "Str 1" becomes "Str 2", ...
	
	unsigned trailing_number = 0;
	unsigned scale = 1;
	size_t i = name.raw().size();
	for (; i > 0; --i, scale *= 10) {
		char const ch = name.raw().c_str()[i-1];
		if (ch >= '0' && ch <= '9') {
			trailing_number += unsigned(ch - '0') * scale;
		} else {
			break;
		}
	}
	
	// i now points to the first digit of the trailing number
	
	std::ostringstream strm;
	strm.write(name.raw().c_str(), i);
	if (i == name.raw().size()) {
		strm << ' ';
	}
	strm << (trailing_number + 1);
	
	return Utf8String(strm.str());
}

bool
XmlLoader::isContentAllowedHere() const
{
	if (m_parentTags.empty()) {
		return false;
	}
	BString const& tag = m_parentTags.back();
	
	static BStringPOD const tags[] = {
		{ "host-mask" },
		{ "type" },
		{ "host" },
		{ "port" },
		{ "user" },
		{ "pass" }
	};
	BStringPOD const* const tags_end = tags + ARRAY_SIZE(tags);
	if (std::find(tags, tags_end, tag) != tags_end) {
		return true;
	}
	return false;
}

bool
XmlLoader::isTagAllowedHere(BString const& tag) const
{
	if (m_parentTags.empty()) {
		return tag == BString("forwarding");
	}
	BString const& parent = m_parentTags.back();
	
	struct ParentChild
	{
		BStringPOD parent;
		BStringPOD child;
	};
	static ParentChild const tags[] = {
		{ { "forwarding" }, { "option" } },
		{ { "option" }, { "bypass" } },
		{ { "option" }, { "proxy-chain" } },
		{ { "bypass" }, { "host-mask" } },
		{ { "bypass" }, { "simple-hostnames" } },
		{ { "proxy-chain" }, { "proxy" } },
		{ { "proxy" }, { "type" } },
		{ { "proxy" }, { "host" } },
		{ { "proxy" }, { "port" } },
		{ { "proxy" }, { "user" } },
		{ { "proxy" }, { "pass" } }
	};
	
	for (unsigned i = 0; i < ARRAY_SIZE(tags); ++i) {
		if (tags[i].parent == parent && tags[i].child == tag) {
			return true;
		}
	}
	
	return false;
}

void
XmlLoader::issueError(std::string const& msg, int line, int col)
{
	ConfError err(ConfError::T_ERROR, msg, line, col);
	m_pErrorHandler->handleError(err);
}

void
XmlLoader::issueWarning(std::string const& msg, int line, int col)
{
	ConfError err(ConfError::T_WARNING, msg, line, col);
	m_pErrorHandler->handleError(err);
}

} // namespace impl


/*============================== BypassList ============================*/

void
BypassList::accept(Forwarding::BypassVisitor& visitor) const
{
	const_iterator it(this->begin());
	const_iterator const end(this->end());
	for (; it != end; ++it) {
		(*it)->accept(visitor);
	}
}

void
BypassList::toStream(std::ostream& strm, int indent) const
{
	if (empty()) {
		return;
	}
	
	strm << Indent(indent) << "<bypass>\n";
	
	const_iterator it(this->begin());
	const_iterator const end(this->end());
	for (; it != end; ++it) {
		(*it)->toStream(strm, indent + 1);
	}
	
	strm << Indent(indent) << "</bypass>\n";
}


/*========================== ProxyDescriptorList =======================*/

void
ProxyDescriptorList::toStream(std::ostream& strm, int indent) const
{
	if (empty()) {
		return;
	}
	
	strm << Indent(indent) << "<proxy-chain>\n";
	
	const_iterator it(this->begin());
	const_iterator const end(this->end());
	for (; it != end; ++it) {
		itemToStream(*it, strm, indent + 1);
	}
	
	strm << Indent(indent) << "</proxy-chain>\n";
}


void
ProxyDescriptorList::itemToStream(
	ProxyDescriptor const& item, std::ostream& strm, int indent) const
{
	strm << Indent(indent) << "<proxy>\n";
	
	Indent ind(indent + 1);
	strm << ind << "<type>" << item.getTypeAsString() << "</type>\n";
	strm << ind << "<host>";
	strm << HtmlEscaper::escape(item.getAddr().getHost()) << "</host>\n";
	strm << ind << "<port>" << item.getAddr().getPort() << "</port>\n";
	if (!item.getUserName().empty() || !item.getPassword().empty()) {
		strm << ind << "<user>";
		strm << HtmlEscaper::escape(item.getUserName()) << "</user>\n";
	}
	if (!item.getPassword().empty()) {
		strm << ind << "<pass>";
		strm << HtmlEscaper::escape(item.getPassword()) << "</pass>\n";
	}
	
	strm << Indent(indent) << "</proxy>\n";
}


/*=============================== Option ===============================*/

Option::Option(Utf8String const& name)
:	m_name(name),
	m_isSelected(false)
{
}

Option::~Option()
{
}

void
Option::swap(Option& other)
{
	m_name.swap(other.m_name);
	std::swap(m_isSelected, other.m_isSelected);
	m_bypassList.swap(other.m_bypassList);
	m_proxyChain.swap(other.m_proxyChain);
}

void
Option::toStream(std::ostream& strm, int indent) const
{
	strm << Indent(indent) << "<option name=\"";
	strm << HtmlEscaper::escape(m_name.raw()) << '\"';
	if (m_isSelected) {
		strm << " selected=\"selected\"";
	}
	strm << ">\n";
	
	m_bypassList.toStream(strm, indent + 1);
	m_proxyChain.toStream(strm, indent + 1);
	
	strm << Indent(indent) << "</option>\n";
}


/*============================== OptionList ============================*/

void
OptionList::toStream(std::ostream& strm, int indent) const
{
	const_iterator it(this->begin());
	const_iterator const end(this->end());
	for (; it != end; ++it) {
		strm << "\n"; // an empty line between options
		it->toStream(strm, indent);
	}
}


/*============================== Config ================================*/

Config::Config()
{
}

Config::~Config()
{
}

void
Config::swap(Config& other)
{
	m_options.swap(other.m_options);
}

void
Config::toStream(std::ostream& strm) const
{
	strm << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
	strm << "<forwarding>\n";
	
	m_options.toStream(strm, 1);
	
	strm << "</forwarding>\n";
}

bool
Config::fromStream(std::istream& strm, ConfErrorHandler& eh)
{
	XmlLoader loader;
	if (loader.load(strm, eh)) {
		swap(loader.config());
		return true;
	} else {
		return false;
	}
}


/*============================ SystemOption ============================*/

SystemOption::SystemOption(Utf8String const& name)
:	m_name(name)
{
}

SystemOption::~SystemOption()
{
}


/*============================== Resolver ==============================*/

Resolver::Resolver()
:	m_ptrEmptyProxyChain(new ProxyChain),
	m_pSelectedOption(0)
{
}

Resolver::Resolver(Config const& config)
:	m_ptrEmptyProxyChain(new ProxyChain),
	m_pSelectedOption(0)
{
	OptionList const& list = config.options();
	OptionList::const_iterator it(list.begin());
	OptionList::const_iterator const end(list.end());
	for (; it != end; ++it) {
		Opt& opt = m_options[it->getName()];
		createBypassSet(it->bypassList()).swap(opt.bypassSet);
		createProxyChain(it->proxyChain()).swap(opt.proxyChain);
		if (it->isSelected()) {
			m_pSelectedOption = &opt;
		}
	}
}

Resolver::~Resolver()
{
}

void
Resolver::selectOption(Utf8String const& option_name)
{
	std::map<Utf8String, Opt>::iterator it(m_options.find(option_name));
	if (it == m_options.end()) {
		throw ResolverException("option not found");
	} else {
		m_pSelectedOption = &it->second;
	}
}

void
Resolver::selectSystemOption(SystemOption const& option)
{
	std::map<Utf8String, Opt>::iterator it(m_options.find(option.getName()));
	if (it != m_options.end()) {
		m_ptrSystemOption.reset(0);
		m_pSelectedOption = &it->second;
	} else {
		m_ptrSystemOption.reset(new SysOpt);
		createBypassSet(option.bypassList()).swap(m_ptrSystemOption->bypassSet);
		createProxyChain(option.proxyChain()).swap(m_ptrSystemOption->proxyChain);
		m_ptrSystemOption->errorMessage = option.getErrorMessage();
		m_pSelectedOption = m_ptrSystemOption.get();
	}
}

ProxyChainConstPtr
Resolver::resolve(URI const& url) const
{
	if (!m_pSelectedOption) {
		return m_ptrEmptyProxyChain;
	} else if (m_pSelectedOption == m_ptrSystemOption.get()) {
		if (!m_ptrSystemOption->errorMessage.empty()) {
			throw ResolverException(m_ptrSystemOption->errorMessage);
		}
	}
	
	if (matches(url, m_pSelectedOption->bypassSet)) {
		return m_ptrEmptyProxyChain;
	} else {
		return m_pSelectedOption->proxyChain;
	}
}

void
Resolver::swap(Resolver& other)
{
	m_options.swap(other.m_options);
	SysOpt* opt1 = m_ptrSystemOption.release();
	SysOpt* opt2 = other.m_ptrSystemOption.release();
	m_ptrSystemOption.reset(opt2);
	other.m_ptrSystemOption.reset(opt1);
	std::swap(m_pSelectedOption, other.m_pSelectedOption);
}

Resolver::BypassSet
Resolver::createBypassSet(BypassList const& list)
{
	BypassSet set;
	set.reserve(list.size());
	BypassList::const_iterator it(list.begin());
	BypassList::const_iterator const end(list.end());
	for (; it != end; ++it) {
		MatcherConstPtr matcher((*it)->getMatcher());
		if (matcher) {
			set.push_back(matcher);
		}
	}
	return set;
}

ProxyChainConstPtr
Resolver::createProxyChain(ProxyDescriptorList const& proxies)
{
	std::vector<ProxyDescriptor> vec(proxies.begin(), proxies.end());
	std::auto_ptr<ProxyChain> proxy_chain(new ProxyChain);
	proxy_chain->swap(vec);
	return ProxyChainConstPtr(proxy_chain.release());
}

bool
Resolver::matches(URI const& url, BypassSet const& bypass_set)
{
	BypassSet::const_iterator it(bypass_set.begin());
	BypassSet::const_iterator const end(bypass_set.end());
	for (; it != end; ++it) {
		if ((*it)->matches(url)) {
			return true;
		}
	}
	return false;
}

} // namespace Forwarding
