/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ConfigFileStructure.h"
#include "Conf.h"
#include "ConfError.h"
#include "ConfErrorHandler.h"
#include "Color.h"
#include "SymbolicInetAddr.h"
#include "ProxyDescriptor.h"
#include "PortSet.h"
#include "ArraySize.h"
#include "StringUtils.h"
#include "types.h"
#include <ace/config-lite.h>
#include <boost/lexical_cast.hpp>
#include <set>
#include <string>
#include <memory>
#include <istream>
#include <ostream>
#include <sstream>
#include <algorithm>
#include <cstring>
#include <stddef.h>

using namespace std;

class ConfigFileStructure::Reader : public ConfIO
{
public:
	Reader(std::list<Element>& elements, Config& target, ConfErrorHandler& eh);
	
	virtual ~Reader();
	
	bool finalValidation();
private:
	typedef bool (Reader::*ParamHandlerPtr)(std::string const& value, int line);
	
	struct ParamHandlerNode
	{
		char const* name;
		ParamHandlerPtr handler;
		
		bool operator==(std::string const& rhs) const {
			return name == rhs;
		}
	};
	
	virtual bool processElement(Element const& element, int line);
	
	virtual bool processError(ConfError const& error);
	
	bool validateSection(std::string const& section, int line);
	
	bool processKeyValue(
		std::string const& section, std::string const& key,
		std::string const& value, int line);
	
	bool handleParam(
		ParamHandlerNode const* begin, ParamHandlerNode const* end,
		std::string const& key, std::string const& value, int line);
	
	bool handleListenAddress(std::string const& value, int line);
	
	bool processListenAddr(
		char const* begin, char const* end,
		int line, std::list<SymbolicInetAddr>& addrs);
	
	bool handleClientCompression(std::string const& value, int line);
	
	bool handleAdBorder(std::string const& value, int line);
	
	bool handlePageCleanup(std::string const& value, int line);
	
	bool handleTrayIconAnimation(std::string const& value, int line);
	
	bool handleNoFlash(std::string const& value, int line);
	
	bool handleMaxScriptNestLevel(std::string const& value, int line);
	
	bool handleMaxScriptFetchSize(std::string const& value, int line);
	
	bool handleMaxScriptEvalSize(std::string const& value, int line);
	
	bool handleSaveTrafficThreshold(std::string const& value, int line);
	
	bool handleReportClientIP(std::string const& value, int line);
	
	bool handleAllowedTunnelPorts(std::string const& value, int line);
	
	bool handleUseProxy(std::string const& value, int line);
	
	bool handleProxyType(std::string const& value, int line);
	
	bool handleProxyHost(std::string const& value, int line);
	
	bool handleProxyPort(std::string const& value, int line);
	
	bool handleProxyUser(std::string const& value, int line);
	
	bool handleProxyPass(std::string const& value, int line);
	
	bool handleNoProxyFor(std::string const& value, int line);
	
	std::list<Element>& m_rElements;
	Config& m_rTarget;
	ConfErrorHandler& m_rErrorHandler;
	std::string m_curSection;
	std::set<std::string> m_sections;
	std::set<std::string> m_keys; // within the current section
	int m_proxyTypeLine;
	int m_proxyHostLine;
	int m_proxyPortLine;
	int m_proxyUserLine;
	int m_proxyPassLine;
};


class ConfigFileStructure::Updater
{
public:
	typedef ConfIO::Element Element;
	
	Updater(std::list<Element>& elements, Config const& source);
	
	~Updater();
	
	void update();
private:
	typedef void (Updater::*SectionHandlerPtr)(
		std::list<Element>::iterator const& begin,
		std::list<Element>::iterator const& end);
	
	typedef void (Updater::*ParamHandlerPtr)(Element& el);
	
	struct SectionHandlerNode
	{
		bool processed;
		char const* name;
		SectionHandlerPtr handler;
		
		bool operator==(std::string const& rhs) const {
			return name == rhs;
		}
	};
	
	struct ParamHandlerNode
	{
		bool processed;
		char const* name;
		ParamHandlerPtr handler;
		
		bool operator==(std::string const& rhs) const {
			return name == rhs;
		}
	};
	
	void createMissingGlobalSection();
	
	void processSections(
		SectionHandlerNode* h_begin,
		SectionHandlerNode* h_end);
	
	void executeSectionHandler(
		SectionHandlerNode* h_begin,
		SectionHandlerNode* h_end,
		std::string const& section,
		std::list<Element>::iterator const& begin,
		std::list<Element>::iterator const& end);
	
	void handleGlobalSection(
		std::list<Element>::iterator const& begin,
		std::list<Element>::iterator const& end);
	
	void handleUnknownSection(
		std::string const& section, 
		std::list<Element>::iterator const& begin,
		std::list<Element>::iterator const& end);
	
	void processParams(
		std::list<Element>::iterator const& begin,
		std::list<Element>::iterator const& end,
		ParamHandlerNode* h_begin,
		ParamHandlerNode* h_end);
	
	void handleListenAddress(Element& el);
	
	void handleClientCompression(Element& el);
	
	void handleAdBorder(Element& el);
	
	void handlePageCleanup(Element& el);
	
	void handleTrayIconAnimation(Element& el);
	
	void handleMaxScriptNestLevel(Element& el);
	
	void handleMaxScriptFetchSize(Element& el);
	
	void handleMaxScriptEvalSize(Element& el);
	
	void handleSaveTrafficThreshold(Element& el);
	
	void handleReportClientIP(Element& el);
	
	void handleAllowedTunnelPorts(Element& el);
	
	static void commentOut(Element& el);
	
	std::list<Element>& m_rElements;
	Config const& m_rSource;
};


void
ConfigFileStructure::clear()
{
	m_elements.clear();
}

void
ConfigFileStructure::load(
	std::string const& text, Config& target, ConfErrorHandler& eh)
{
	clear();
	Reader reader(m_elements, target, eh);
	reader.read(text);
	reader.finalValidation();
}

void
ConfigFileStructure::updateWith(Config const& config)
{
	Updater updater(m_elements, config);
	updater.update();
}

void
ConfigFileStructure::swap(ConfigFileStructure& other)
{
	m_elements.swap(other.m_elements);
}

void
ConfigFileStructure::toStream(std::ostream& strm) const
{
	ConfIO::write(strm, m_elements, ConfIO::DISABLE_HEREDOC);
}


/*========================== ConfigFileStructure::Reader ===========================*/

ConfigFileStructure::Reader::Reader(
	list<Element>& elements, Config& target, ConfErrorHandler& eh)
:	m_rElements(elements),
	m_rTarget(target),
	m_rErrorHandler(eh),
	m_curSection("global"), // backward compatibilty
	m_proxyTypeLine(0),
	m_proxyHostLine(0),
	m_proxyPortLine(0),
	m_proxyUserLine(0),
	m_proxyPassLine(0)
{
}

ConfigFileStructure::Reader::~Reader()
{
}

bool
ConfigFileStructure::Reader::processElement(Element const& element, int line)
{
	if (element.getType() == Element::SECTION) {
		m_curSection = element.getText();
		if (!validateSection(m_curSection, line)) {
			return false;
		}
		m_sections.insert(element.getText());
		m_keys.clear();
	} else if (element.getType() == Element::KEY_VALUE) {
		if (!processKeyValue(m_curSection, element.getKey(), element.getValue(), line)) {
			return false;
		}
		m_keys.insert(element.getKey());
	}
	m_rElements.push_back(element);
	return true;
}

bool
ConfigFileStructure::Reader::processError(ConfError const& error)
{
	return m_rErrorHandler.handleError(error);
}

bool
ConfigFileStructure::Reader::validateSection(std::string const& section, int line)
{
	if (m_sections.find(section) != m_sections.end()) {
		ConfError err(
			ConfError::T_ERROR,
			"Duplicate section: "+section, line
		);
		return m_rErrorHandler.handleError(err);
	} else if (section != "global" && section != "forwarding") {
		ConfError err(
			ConfError::T_WARNING,
			"Unknown section: "+section, line
		);
		return m_rErrorHandler.handleError(err);
	}
	return true;
}

bool
ConfigFileStructure::Reader::finalValidation()
{
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	
	if (forwarding.isNextHopProxyEnabled()) {	
		if (forwarding.nextHopProxy().getType() == ProxyDescriptor::INVALID) {
			// If it's not given at all, then it's HTTP,
			// for backwards compatibility.
			forwarding.setNextHopProxyEnabled(false);
			m_rTarget.setObsoleteForwardingInfo(forwarding);
			ConfError err(
				ConfError::T_ERROR,
				"Illegal value for \"proxy_type\"",
				m_proxyTypeLine
			);
			return m_rErrorHandler.handleError(err);
		}
		
		if (forwarding.nextHopProxy().getAddr().getHost().empty()) {
			forwarding.setNextHopProxyEnabled(false);
			m_rTarget.setObsoleteForwardingInfo(forwarding);
			ConfError err(
				ConfError::T_ERROR,
				"Illegal value for \"proxy_host\"",
				m_proxyHostLine
			);
			return m_rErrorHandler.handleError(err);
		}
		
		if (forwarding.nextHopProxy().getAddr().getPort() <= 0) {
			forwarding.setNextHopProxyEnabled(false);
			m_rTarget.setObsoleteForwardingInfo(forwarding);
			ConfError err(
				ConfError::T_ERROR,
				"Illegal value for \"proxy_port\"",
				m_proxyPortLine
			);
			return m_rErrorHandler.handleError(err);
		}
		
		if (!(forwarding.nextHopProxy().getUserName().empty()
		      && forwarding.nextHopProxy().getPassword().empty())
		    && forwarding.nextHopProxy().getType() != ProxyDescriptor::SOCKS4
		    && forwarding.nextHopProxy().getType() != ProxyDescriptor::SOCKS4A
		    && forwarding.nextHopProxy().getType() != ProxyDescriptor::SOCKS5) {
			ConfError err(
				ConfError::T_WARNING,
				"Sorry, authentication is only implemented for socks proxies",
				m_proxyUserLine
			);
			return m_rErrorHandler.handleError(err);
		}
		
		if (!forwarding.nextHopProxy().getPassword().empty()
		    && (forwarding.nextHopProxy().getType() == ProxyDescriptor::SOCKS4
		        || forwarding.nextHopProxy().getType() == ProxyDescriptor::SOCKS4A)) {
			ConfError err(
				ConfError::T_WARNING,
				"Socks4 proxies don't use password",
				m_proxyPassLine
			);
			return m_rErrorHandler.handleError(err);
		}
	}
	
	return true;
}

bool
ConfigFileStructure::Reader::processKeyValue(
	string const& section, string const& key,
	string const& value, int line)
{
	if (m_keys.find(key) != m_keys.end()) {
		ConfError err(
			ConfError::T_ERROR,
			"Duplicate parameter", line
		);
		return m_rErrorHandler.handleError(err);
	}
	if (section == "global") {
		static ParamHandlerNode const handlers[] = {
			{ "listen_address", &Reader::handleListenAddress },
			{ "client_compression", &Reader::handleClientCompression },
			{ "ad_border", &Reader::handleAdBorder },
			{ "page_cleanup", &Reader::handlePageCleanup },
			{ "tray_icon_animation", &Reader::handleTrayIconAnimation },
			{ "no_flash", &Reader::handleNoFlash },
			{ "max_script_nest_level", &Reader::handleMaxScriptNestLevel },
			{ "max_script_fetch_size", &Reader::handleMaxScriptFetchSize },
			{ "max_script_eval_size", &Reader::handleMaxScriptEvalSize },
			{ "save_traffic_threshold", &Reader::handleSaveTrafficThreshold },
			{ "report_client_ip", &Reader::handleReportClientIP },
			{ "allowed_tunnel_ports", &Reader::handleAllowedTunnelPorts }
		};
		return handleParam(
			handlers, handlers + ARRAY_SIZE(handlers),
			key, value, line
		);
		
	} else if (section == "forwarding") {
		static ParamHandlerNode const handlers[] = {
			{ "use_proxy", &Reader::handleUseProxy },
			{ "proxy_type", &Reader::handleProxyType },
			{ "proxy_host", &Reader::handleProxyHost },
			{ "proxy_port", &Reader::handleProxyPort },
			{ "proxy_user", &Reader::handleProxyUser },
			{ "proxy_pass", &Reader::handleProxyPass },
			{ "no_proxy_for", &Reader::handleNoProxyFor }
		};
		return handleParam(
			handlers, handlers + ARRAY_SIZE(handlers),
			key, value, line
		);
	}
	
	// parameters in unknown sections are ignored,
	// but a warning is issued by validateSection
	return true;
}

bool
ConfigFileStructure::Reader::handleParam(
	ParamHandlerNode const* begin, ParamHandlerNode const* end,
	std::string const& key, std::string const& value, int line)
{
	ParamHandlerNode const* node = std::find(begin, end, key);
	if (node != end) {
		return (this->*node->handler)(value, line);
	} else {
		ConfError err(
			ConfError::T_WARNING,
			"Unknown paramenter: "+key, line
		);
		return m_rErrorHandler.handleError(err);
	}
}

bool
ConfigFileStructure::Reader::handleListenAddress(std::string const& value, int line)
{
	list<SymbolicInetAddr> addrs;
	char const* begin = value.c_str();
	char const* const end = begin + value.size();
	char const* rec_end;
	
	do {
		rec_end = StringUtils::find(begin, end, ',');
		if (!processListenAddr(begin, rec_end, line, addrs)) {
			ConfError err(
				ConfError::T_ERROR,
				"Error parsing listen address \""
				+string(begin, rec_end - begin)+"\"",
				line
			);
			if (!m_rErrorHandler.handleError(err)) {
				return false;
			}
		}
		begin = rec_end + 1;
	} while (rec_end != end);
	
	m_rTarget.setListenAddresses(addrs);
	return true;
}

bool
ConfigFileStructure::Reader::processListenAddr(
	char const* begin, char const* end,
	int line, std::list<SymbolicInetAddr>& addrs)
{
	begin = StringUtils::ltrim(begin, end);
	end = StringUtils::rtrim(begin, end);
	char const* colon_pos = StringUtils::rfind(begin, end, ':');
	
	if (colon_pos == end) {
		return false;
	}
	
	string host(begin, colon_pos - begin);
	BString tmp;
	BString port_str(tmp, colon_pos + 1, end);
	unsigned int port;
	try {
		port = boost::lexical_cast<uint16_t>(port_str);
	} catch (boost::bad_lexical_cast& e) {
		return false;
	}
	
	addrs.push_back(SymbolicInetAddr(host, port));
	return true;
}

bool
ConfigFileStructure::Reader::handleClientCompression(std::string const& value, int line)
{
	if (value != "yes" && value != "no") {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"client_compression\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_rTarget.setClientCompressionEnabled(value == "yes");
	return true;
}

bool
ConfigFileStructure::Reader::handleAdBorder(std::string const& value, int line)
{
	auto_ptr<Color> color;
	if (value != "none") {
		char const* begin = value.c_str();
		char const* end = begin + value.size();
		uint32_t val = StringUtils::parseUnsignedHex<uint32_t>(begin, end);
		if (value.size() != 6 || end != begin + 6) {
			ConfError err(
				ConfError::T_ERROR,
				"Illegal value for \"ad_border\"", line
			);
			return m_rErrorHandler.handleError(err);
		}
		uint8_t r = val;
		uint8_t g = val >> 8;
		uint8_t b = val >> 16;
		color.reset(new Color(r, g, b));
	}
	m_rTarget.setBorderColor(color);
	return true;
}

bool
ConfigFileStructure::Reader::handlePageCleanup(std::string const& value, int line)
{
	Config::PageCleanupLevel cleanup = Config::CLEANUP_OFF;
	if (value == "maximum") {
		cleanup = Config::CLEANUP_MAXIMUM;
	} else if (value == "safe") {
		cleanup = Config::CLEANUP_SAFE;
	} else if (value == "off") {
		cleanup = Config::CLEANUP_OFF;
	} else {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"page_cleanup\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_rTarget.setPageCleanupLevel(cleanup);
	return true;
}

bool
ConfigFileStructure::Reader::handleTrayIconAnimation(std::string const& value, int line)
{
	if (value != "yes" && value != "no") {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"tray_icon_animation\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_rTarget.setTrayAnimationEnabled(value == "yes");
	return true;
}

bool
ConfigFileStructure::Reader::handleNoFlash(std::string const& value, int line)
{
	if (value != "no") {
		ConfError err(
			ConfError::T_WARNING,
			"The \"no_flash\" parameter has been removed.", line
		);
		return m_rErrorHandler.handleError(err);
	}
	return true;
}

bool
ConfigFileStructure::Reader::handleMaxScriptNestLevel(std::string const& value, int line)
{
	size_t val;
	try {
		val = boost::lexical_cast<size_t>(value);
	} catch (boost::bad_lexical_cast& e) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"max_script_nest_level\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_rTarget.setMaxScriptNestLevel(val);
	return true;
}

bool
ConfigFileStructure::Reader::handleMaxScriptFetchSize(std::string const& value, int line)
{
	size_t val;
	try {
		val = boost::lexical_cast<size_t>(value);
	} catch (boost::bad_lexical_cast& e) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"max_script_fetch_size\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_rTarget.setMaxScriptFetchSize(val);
	return true;
}

bool
ConfigFileStructure::Reader::handleMaxScriptEvalSize(std::string const& value, int line)
{
	size_t val;
	try {
		val = boost::lexical_cast<size_t>(value);
	} catch (boost::bad_lexical_cast& e) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"max_script_eval_size\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_rTarget.setMaxScriptEvalSize(val);
	return true;
}

bool
ConfigFileStructure::Reader::handleSaveTrafficThreshold(std::string const& value, int line)
{
	size_t val;
	try {
		val = boost::lexical_cast<size_t>(value);
	} catch (boost::bad_lexical_cast& e) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"save_traffic_threshold\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_rTarget.setSaveTrafficThreshold(val);
	return true;
}

bool
ConfigFileStructure::Reader::handleReportClientIP(std::string const& value, int line)
{
	if (value.empty()) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"report_client_ip\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	if (value == "yes") {
		m_rTarget.setReportClientIP(Config::REPORT_IP_ON);
	} else if (value == "no") {
		m_rTarget.setReportClientIP(Config::REPORT_IP_OFF);
	} else {
		m_rTarget.setReportClientIP(Config::REPORT_IP_FIXED, value);
	}
	return true;
}

bool
ConfigFileStructure::Reader::handleAllowedTunnelPorts(std::string const& value, int line)
{
	PortSet ports;
	if (!ports.fromString(value)) {
		ConfError err(
			ConfError::T_ERROR,
			"Error parsing \"allowed_tunnel_ports\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	
	m_rTarget.setAllowedTunnelPorts(ports);
	return true;
}

bool
ConfigFileStructure::Reader::handleUseProxy(std::string const& value, int line)
{
	if (value != "yes" && value != "no") {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"use_proxy\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	forwarding.setNextHopProxyEnabled(value == "yes");
	m_rTarget.setObsoleteForwardingInfo(forwarding);
	return true;
}

bool
ConfigFileStructure::Reader::handleProxyType(std::string const& value, int line)
{
	m_proxyTypeLine = line;
	ProxyDescriptor::ProxyType type = ProxyDescriptor::INVALID;
	if (value == "http") {
		type = ProxyDescriptor::HTTP;
	} else if (value == "socks4") {
		type = ProxyDescriptor::SOCKS4;
	} else if (value == "socks4a") {
		type = ProxyDescriptor::SOCKS4A;
	} else if (value == "socks5") {
		type = ProxyDescriptor::SOCKS5;
	}
	
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	forwarding.nextHopProxy().setType(type);
	m_rTarget.setObsoleteForwardingInfo(forwarding);
	return true;
}

bool
ConfigFileStructure::Reader::handleProxyHost(std::string const& value, int line)
{
	m_proxyHostLine = line;
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	SymbolicInetAddr addr(forwarding.nextHopProxy().getAddr());
	addr.setHost(value);
	forwarding.nextHopProxy().setAddr(addr);
	m_rTarget.setObsoleteForwardingInfo(forwarding);
	return true;
}

bool
ConfigFileStructure::Reader::handleProxyPort(std::string const& value, int line)
{
	m_proxyPortLine = line;
	int port = -1;
	if (!value.empty() && value != "0" && value != "-1") {
		try {
			port = boost::lexical_cast<uint16_t>(value);
		} catch (boost::bad_lexical_cast& e) {
			ConfError err(
				ConfError::T_ERROR,
				"Illegal value for \"proxy_port\"", line
			);
			return m_rErrorHandler.handleError(err);
		}
	}
	
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	SymbolicInetAddr addr(forwarding.nextHopProxy().getAddr());
	addr.setPort(port);
	forwarding.nextHopProxy().setAddr(addr);
	m_rTarget.setObsoleteForwardingInfo(forwarding);
	
	return true;
}

bool
ConfigFileStructure::Reader::handleProxyUser(std::string const& value, int line)
{
	m_proxyUserLine = line;
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	forwarding.nextHopProxy().setUserName(value);
	m_rTarget.setObsoleteForwardingInfo(forwarding);
	return true;
}

bool
ConfigFileStructure::Reader::handleProxyPass(std::string const& value, int line)
{
	m_proxyPassLine = line;
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	forwarding.nextHopProxy().setPassword(value);
	m_rTarget.setObsoleteForwardingInfo(forwarding);
	return true;
}

bool
ConfigFileStructure::Reader::handleNoProxyFor(std::string const& value, int line)
{
	ObsoleteForwardingInfo forwarding(m_rTarget.getObsoleteForwardingInfo());
	char const* begin = value.c_str();
	char const* end = begin + value.size();
	char const* rec_end;
	
	do {
		rec_end = StringUtils::find(begin, end, ',');
		rec_end = StringUtils::find(begin, rec_end, ';');
		char const* b = StringUtils::ltrim(begin, rec_end);
		char const* e = StringUtils::rtrim(b, rec_end);
		if (b != e) {
			forwarding.proxyBypassList().push_back(string(b, e - b));
		}
		begin = rec_end + 1;
	} while (rec_end != end);
	
	m_rTarget.setObsoleteForwardingInfo(forwarding);
	return true;
}


/*======================== ConfigFileStructure::Updater ==========================*/

ConfigFileStructure::Updater::Updater(list<Element>& elements, Config const& source)
:	m_rElements(elements),
	m_rSource(source)
{
}

ConfigFileStructure::Updater::~Updater()
{
}

void
ConfigFileStructure::Updater::update()
{
	createMissingGlobalSection();
	
	static SectionHandlerNode const handlers_proto[] = {
		{ false, "global", &Updater::handleGlobalSection }
	};
	SectionHandlerNode handlers[ARRAY_SIZE(handlers_proto)];
	memcpy(handlers, handlers_proto, sizeof(handlers));
	
	processSections(handlers, handlers + ARRAY_SIZE(handlers));
}

void
ConfigFileStructure::Updater::createMissingGlobalSection()
{
	// the [global] section wasn't present in some ancient versions
	list<Element>::iterator it = m_rElements.begin();
	list<Element>::iterator const end = m_rElements.end();
	for (; it != end; ++it) {
		if (it->getType() == Element::SECTION) {
			break;
		} else if (it->getType() == Element::KEY_VALUE) {
			m_rElements.push_front(Element("global"));
			break;
		}
	}
}

void
ConfigFileStructure::Updater::processSections(
	SectionHandlerNode* h_begin, SectionHandlerNode* h_end)
{
	list<Element>::iterator it = m_rElements.begin();
	list<Element>::iterator const end = m_rElements.end();
	list<Element>::iterator section = end;
	for (; it != end; ++it) {
		if (it->getType() == Element::SECTION) {
			if (section != end) {
				executeSectionHandler(
					h_begin, h_end, section->getValue(), section, it
				);
			}
			section = it;
		}
	}
	if (section != end) {
		executeSectionHandler(
			h_begin, h_end, section->getValue(), section, end
		);
	}
	
	// Now process sections that weren't encountered in the config file
	
	for (SectionHandlerNode* node = h_begin; node != h_end; ++node) {
		if (node->processed) {
			continue;
		}
		m_rElements.insert(end, Element()); // blank line
		section = m_rElements.insert(
			end, Element(node->name) // section
		);
		(this->*node->handler)(section, end);
		node->processed = true;
	}
}

void
ConfigFileStructure::Updater::executeSectionHandler(
	SectionHandlerNode* h_begin,
	SectionHandlerNode* h_end,
	std::string const& section,
	std::list<Element>::iterator const& begin,
	std::list<Element>::iterator const& end)
{
	SectionHandlerNode* node = std::find(h_begin, h_end, section);
	if (node == h_end) {
		handleUnknownSection(section, begin, end);
	} else {
		(this->*node->handler)(begin, end);
		node->processed = true;
	}
}

void
ConfigFileStructure::Updater::handleGlobalSection(
	std::list<Element>::iterator const& begin,
	std::list<Element>::iterator const& end)
{
	static ParamHandlerNode const handlers_proto[] = {
		{ false, "listen_address", &Updater::handleListenAddress },
		{ false, "client_compression", &Updater::handleClientCompression },
		{ false, "ad_border", &Updater::handleAdBorder },
		{ false, "page_cleanup", &Updater::handlePageCleanup },
		{ false, "tray_icon_animation", &Updater::handleTrayIconAnimation },
		{ false, "max_script_nest_level", &Updater::handleMaxScriptNestLevel },
		{ false, "max_script_fetch_size", &Updater::handleMaxScriptFetchSize },
		{ false, "max_script_eval_size", &Updater::handleMaxScriptEvalSize },
		{ false, "save_traffic_threshold", &Updater::handleSaveTrafficThreshold },
		{ false, "report_client_ip", &Updater::handleReportClientIP },
		{ false, "allowed_tunnel_ports", &Updater::handleAllowedTunnelPorts }
	};
	ParamHandlerNode handlers[ARRAY_SIZE(handlers_proto)];
	memcpy(handlers, handlers_proto, sizeof(handlers));
	
	processParams(begin, end, handlers, handlers + ARRAY_SIZE(handlers));
}

void
ConfigFileStructure::Updater::handleUnknownSection(
	std::string const& section,
	std::list<Element>::iterator const& begin,
	std::list<Element>::iterator const& end)
{
	if (section == "forwarding") {
		// It's an obsolete section, but we don't want to
		// comment it out.
		return;
	}
	
	// comment it out
	list<Element>::iterator it = begin;
	for (; it != end; ++it) {
		commentOut(*it);
	}
}

void
ConfigFileStructure::Updater::processParams(
	std::list<Element>::iterator const& begin,
	std::list<Element>::iterator const& end,
	ParamHandlerNode* h_begin,
	ParamHandlerNode* h_end)
{
	list<Element>::iterator it = begin;
	for (; it != end; ++it) {
		if (it->getType() != Element::KEY_VALUE) {
			continue;
		}
		ParamHandlerNode* node = std::find(
			h_begin, h_end, it->getKey()
		);
		if (node == h_end) {
			// unknown parameter
			commentOut(*it);
			continue;
		}
		if (node->processed) {
			// duplicate parameter
			commentOut(*it);
			continue;
		}
		(this->*node->handler)(*it);
		node->processed = true;
	}
	
	// Now process parameters that weren't encountered in the config file.
	
	list<Element>::iterator insertion_pos = end;
	while (insertion_pos != begin) {
		--insertion_pos;
		if (insertion_pos->getType() != Element::BLANK_LINE) {
			++insertion_pos;
			break;
		}
	}
	
	for (ParamHandlerNode* node = h_begin; node != h_end; ++node) {
		if (node->processed) {
			continue;
		}
		m_rElements.insert(insertion_pos, Element()); // blank line
		list<Element>::iterator el_pos = m_rElements.insert(
			insertion_pos, Element(node->name, "")
		);
		(this->*node->handler)(*el_pos);
		node->processed = true;
	}
}

void
ConfigFileStructure::Updater::handleListenAddress(Element& el)
{
	list<SymbolicInetAddr> listen_addrs = m_rSource.getListenAddresses();
	el.setValue(StringUtils::join(", ", listen_addrs.begin(), listen_addrs.end()));
}

void
ConfigFileStructure::Updater::handleClientCompression(Element& el)
{
	string val = m_rSource.isClientCompressionEnabled() ? "yes" : "no";
	el.setValue(val);
}

void
ConfigFileStructure::Updater::handleAdBorder(Element& el)
{
	auto_ptr<Color> color = m_rSource.getBorderColor();
	if (!color.get()) {
		el.setValue("none");
	} else {
		ostringstream strm;
		strm << std::hex;
		strm << (unsigned)color->getRed();
		strm << (unsigned)color->getGreen();
		strm << (unsigned)color->getBlue();
		el.setValue(strm.str());
	}
}

void
ConfigFileStructure::Updater::handlePageCleanup(Element& el)
{
	char const* val = 0;
	switch (m_rSource.getPageCleanupLevel()) {
		case Config::CLEANUP_MAXIMUM:
		val = "maximum";
		break;
		case Config::CLEANUP_SAFE:
		val = "safe";
		break;
		default:
		val = "off";
	}
	el.setValue(val);
}

void
ConfigFileStructure::Updater::handleTrayIconAnimation(Element& el)
{
	string val = m_rSource.isTrayAnimationEnabled() ? "yes" : "no";
	el.setValue(val);
}

void
ConfigFileStructure::Updater::handleMaxScriptNestLevel(Element& el)
{
	el.setValue(StringUtils::fromNumber(m_rSource.getMaxScriptNestLevel()));
}

void
ConfigFileStructure::Updater::handleMaxScriptFetchSize(Element& el)
{
	el.setValue(StringUtils::fromNumber(m_rSource.getMaxScriptFetchSize()));
}

void
ConfigFileStructure::Updater::handleMaxScriptEvalSize(Element& el)
{
	el.setValue(StringUtils::fromNumber(m_rSource.getMaxScriptEvalSize()));
}

void
ConfigFileStructure::Updater::handleSaveTrafficThreshold(Element& el)
{
	el.setValue(StringUtils::fromNumber(m_rSource.getSaveTrafficThreshold()));
}

void
ConfigFileStructure::Updater::handleReportClientIP(Element& el)
{
	Config::ReportClientIP report = m_rSource.getReportClientIP();
	if (report == Config::REPORT_IP_ON) {
		el.setValue("yes");
	} else if (report == Config::REPORT_IP_OFF) {
		el.setValue("no");
	} else {
		el.setValue(m_rSource.getFixedClientIP());
	}
}

void
ConfigFileStructure::Updater::handleAllowedTunnelPorts(Element& el)
{
	ostringstream strm;
	m_rSource.getAllowedTunnelPorts().toStream(strm);
	el.setValue(strm.str());
}

void
ConfigFileStructure::Updater::commentOut(Element& el)
{
	if (el.getType() == Element::COMMENT) {
		return;
	}
	
	ostringstream strm;
	el.toStream(strm);
	string str = strm.str();
	
	if (str.find('\n') == string::npos) {
		el = Element(';', str);
	} else {
		el = Element(';', " <multi-line value was there>");
	}
}
