/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef CONF_H_
#define CONF_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "SymbolicInetAddr.h"
#include "ProxyDescriptor.h"
#include "PortSet.h"
#include <string>
#include <list>
#include <memory>
#include <stddef.h>

class Color;
namespace Forwarding
{
	class Config;
}

/**
 * \brief Represents the [forwarding] section of the "config" file.
 *
 * Starting from version 1.1, forwarding information is stored in the
 * "forwarding.xml" file.  Forwarding information from "config" is still
 * supported, and used when "forwarding.xml" is missing.
 */
class ObsoleteForwardingInfo
{
public:
	ObsoleteForwardingInfo();
	
	ObsoleteForwardingInfo(ObsoleteForwardingInfo const& other);
	
	~ObsoleteForwardingInfo();
	
	ObsoleteForwardingInfo& operator=(ObsoleteForwardingInfo const& rhs);
	
	void swap(ObsoleteForwardingInfo& other);
	
	ProxyDescriptor& nextHopProxy() { return m_nextHopProxy; }
	
	ProxyDescriptor const& nextHopProxy() const { return m_nextHopProxy; }
	
	bool isNextHopProxyEnabled() const { return m_isNextHopProxyEnabled; }
	
	void setNextHopProxyEnabled(bool val) { m_isNextHopProxyEnabled = val; }
	
	std::list<std::string>& proxyBypassList() { return m_proxyBypassList; }
	
	std::list<std::string> const& proxyBypassList() const { return m_proxyBypassList; }
	
	Forwarding::Config toNewFormat() const;
private:
	static bool isASCII(std::string const& str);
	
	ProxyDescriptor m_nextHopProxy;
	std::list<std::string> m_proxyBypassList;
	bool m_isNextHopProxyEnabled;
};


inline void swap(ObsoleteForwardingInfo& o1, ObsoleteForwardingInfo& o2)
{
	o1.swap(o2);
}


class Config
{
public:
	enum ReportClientIP { REPORT_IP_ON, REPORT_IP_OFF, REPORT_IP_FIXED };
	enum PageCleanupLevel { CLEANUP_OFF, CLEANUP_SAFE, CLEANUP_MAXIMUM };
	
	Config();
	
	Config(Config const& other);
	
	~Config();
	
	Config& operator=(Config const& other);
	
	void swap(Config& other);
	
	std::list<SymbolicInetAddr> getListenAddresses() const { return m_listenAddrs; }
	
	void setListenAddresses(std::list<SymbolicInetAddr> const& addrs) { m_listenAddrs = addrs; }
	
	ObsoleteForwardingInfo getObsoleteForwardingInfo() const { return m_forwardingInfo; }
	
	void setObsoleteForwardingInfo(ObsoleteForwardingInfo const& info) { m_forwardingInfo = info; }
	
	bool isClientCompressionEnabled() const { return m_isClientCompressionEnabled; }
	
	void setClientCompressionEnabled(bool val) { m_isClientCompressionEnabled = val; }
	
	std::auto_ptr<Color> getBorderColor() const;
	
	void setBorderColor(std::auto_ptr<Color> color);
	
	PageCleanupLevel getPageCleanupLevel() const { return m_pageCleanupLevel; }
	
	void setPageCleanupLevel(PageCleanupLevel level) { m_pageCleanupLevel = level; }
	
	bool isTrayAnimationEnabled() const { return m_isTrayAnimationEnabled; }
	
	void setTrayAnimationEnabled(bool val) { m_isTrayAnimationEnabled = val; }
	
	size_t getMaxScriptNestLevel() const { return m_maxScriptNestLevel; }
	
	void setMaxScriptNestLevel(size_t val) { m_maxScriptNestLevel = val; }
	
	size_t getMaxScriptFetchSize() const { return m_maxScriptFetchSize; }
	
	void setMaxScriptFetchSize(size_t val) { m_maxScriptFetchSize = val; }
	
	size_t getMaxScriptEvalSize() const { return m_maxScriptEvalSize; }
	
	void setMaxScriptEvalSize(size_t val) { m_maxScriptEvalSize = val; }
	
	size_t getSaveTrafficThreshold() const { return m_saveTrafficThreshold; }
	
	void setSaveTrafficThreshold(size_t val) { m_saveTrafficThreshold = val; }
	
	ReportClientIP getReportClientIP() const { return m_reportClientIP; }
	
	std::string const& getFixedClientIP() const { return m_fixedClientIP; }
	
	void setReportClientIP(ReportClientIP report, std::string const& fixed = std::string()) {
		m_reportClientIP = report;
		m_fixedClientIP = fixed;
	}
	
	PortSet getAllowedTunnelPorts() const { return m_allowedTunnelPorts; }
	
	void setAllowedTunnelPorts(PortSet const& ports) { m_allowedTunnelPorts = ports; }
	
	bool isTunnelPortAllowed(unsigned port) const;
	
	Forwarding::Config extractForwardingInfo() const;
private:
	template<typename T> static void auto_ptr_swap(
		std::auto_ptr<T>& o1, std::auto_ptr<T>& o2);
	
	std::list<SymbolicInetAddr> m_listenAddrs;
	ObsoleteForwardingInfo m_forwardingInfo;
	bool m_isClientCompressionEnabled;
	bool m_isTrayAnimationEnabled;
	std::auto_ptr<Color> m_ptrBorderColor;
	PageCleanupLevel m_pageCleanupLevel;
	size_t m_maxScriptNestLevel;
	size_t m_maxScriptFetchSize;
	size_t m_maxScriptEvalSize;
	size_t m_saveTrafficThreshold;
	ReportClientIP m_reportClientIP;
	std::string m_fixedClientIP;
	PortSet m_allowedTunnelPorts;
};


inline void swap(Config& lhs, Config& rhs)
{
	lhs.swap(rhs);
}

#endif
