/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "WorkerThreadPool.h"
#include "ServiceContext.h"
#include "Reactor.h"
#include "ReactorFactory.h"
#include "SynchFactory.h"
#include "TimeDelta.h"
#include "MonotonicTimer.h"
#include "Client.h"
#include "ScopedIncDec.h"
#include "Debug.h"
#include <ace/config-lite.h>
#include <ace/Thread_Manager.h>
#include <ace/Time_Value.h>
#include <ace/Reverse_Lock_T.h>
#include <ace/OS_NS_sys_time.h>
#include <memory>
#include <string>
#include <stdexcept>
#include <assert.h>

using namespace std;

class WorkerThreadPool::ReactorRegistration : public IntrusiveListNode
{
public:
	ReactorRegistration(Reactor* reactor, ReactorList& list);
	
	Reactor* reactor() { return m_pReactor; }
private:
	Reactor* m_pReactor;
};


WorkerThreadPool::WorkerThreadPool()
:	m_cond(m_mutex),
	m_numWaiters(0),
	m_status(INACTIVE),
#if defined(ACE_WIN32) && !ACE_HAS_WINNT4
	// CriticalSection-based mutexes don't support trylock() on win9x
	m_leaderMutex(USYNC_PROCESS),
#else
	m_leaderMutex(USYNC_THREAD),
#endif
	m_connAcceptor(ConnAcceptor::DONT_STOP_ON_SIGNALS)
{
}

WorkerThreadPool::~WorkerThreadPool()
{
	deactivate();
}

void
WorkerThreadPool::activate()
{
	ACE_Guard<Mutex> start_stop_guard(m_startStopMutex);
	ACE_Guard<Mutex> guard(m_mutex);
	
	if (m_status == INACTIVE) {
		prepareNewThreadLocked(); // may throw
		m_status = ACTIVE;
	} else {
		assert(m_status == ACTIVE);
	}
}

void
WorkerThreadPool::deactivate()
{
	ACE_Guard<Mutex> start_stop_guard(m_startStopMutex);
	
	{
		ACE_Guard<Mutex> guard(m_mutex);
		
		if (m_status == INACTIVE) {
			return;
		} else {
			assert(m_status == ACTIVE);
		}
		
		m_status = STOPPING;
		
		m_connAcceptor.abort();
		
		ReactorList::iterator it = m_runningReactors.begin();
		ReactorList::iterator const end = m_runningReactors.end();
		for (; it != end; ++it) {
			it->reactor()->stop();
		}
		
		m_cond.broadcast();
		// After this, worker threads will wake up and try
		// to lock m_mutex.
	}
	
	m_threadManager.wait();
	
	// Because all worker threads have already finished, we can
	// do this without locking m_mutex.  Note that m_startStopMutex
	// is still locked.
	m_status = INACTIVE;
}

void
WorkerThreadPool::addAcceptor(ConnAcceptor::AcceptorPtr const& acceptor)
{
	if (!m_connAcceptor.add(acceptor)) {
		throw std::runtime_error("acceptor could not be registered at a reactor");
	}
}

void
WorkerThreadPool::removeAcceptor(ConnAcceptor::AcceptorPtr const& acceptor)
{
	m_connAcceptor.remove(acceptor);
}

void
WorkerThreadPool::removeAllAcceptors()
{
	m_connAcceptor.removeAll();
}

void
WorkerThreadPool::prepareNewThreadLocked()
{
	// m_mutex is locked
	
	assert(m_status != STOPPING);
	
	if (m_numWaiters > 0) {
		if (m_cond.signal() == -1) {
			throw std::runtime_error("signalling a worker thread failed");
		}
	} else {
		int const res = m_threadManager.spawn(
			&WorkerThreadPool::runThread,
			this, THR_SCOPE_SYSTEM|THR_DETACHED
		);
		if (res == -1) {
			throw std::runtime_error("creating a worker thread failed");
		}
		m_lastThreadCreateTime = MonotonicTimer::getTimestamp();
	}
}

int
WorkerThreadPool::waitLocked()
{
	// m_mutex is locked
	
	ScopedIncDec<int> waiters_manager(m_numWaiters);
	
	ACE_Time_Value tv(WAIT_FOR_NEW_TASK);
	tv += ACE_OS::gettimeofday();
	return m_cond.wait(&tv);
}

void
WorkerThreadPool::runServiceUnlocked()
{
	// m_mutex is unlocked
	
	auto_ptr<Reactor> reactor;
	try {
		reactor = ReactorFactory::createBestReactor(SynchFactory<ACE_NULL_SYNCH>());
	} catch (Reactor::Exception& e) {
		DEBUGLOG2("Could not initialize the reactor: " << e.what());
		return;
	}
	ServiceContext service_context(*reactor);
	
	ACE_Guard<Mutex> guard(m_mutex);
	
	// m_mutex is locked
	
	ReactorRegistration reactor_registration(reactor.get(), m_runningReactors);
	
	while (m_status == ACTIVE) {
		if (tryLeadershipLocked(service_context)) {
			continue;
		}
		
		if (m_numWaiters >= WAITERS_KEEP_ALIVE && MonotonicTimer::getTimestamp()
		    > m_lastThreadCreateTime + TimeDelta::fromSec(WAIT_FOR_NEW_TASK)) {
			break;
		}
		
		if (waitLocked() == -1) {
			if (m_numWaiters >= WAITERS_KEEP_ALIVE) {
				break;
			}
		}
	}
}

/**
 * We use the "Leader / Followers" pattern.  Leader is
 * a thread that is currently accepting connections.
 * \return true if we succeeded becoming a leader, false otherwise.
 */
bool
WorkerThreadPool::tryLeadershipLocked(ServiceContext& service_context)
{
	// m_mutex is locked
	
	typedef ACE_Reverse_Lock<Mutex> ReverseLock;
	ReverseLock reverse_lock(m_mutex);
	ACE_Guard<ReverseLock> unguard(reverse_lock);
	
	return tryLeadershipUnlocked(service_context);
}

/**
 * We use the "Leader / Followers" pattern.  Leader is
 * a thread that is currently accepting connections.
 * \return true if we succeeded becoming a leader, false otherwise.
 */
bool
WorkerThreadPool::tryLeadershipUnlocked(ServiceContext& service_context)
{
	// m_mutex is unlocked
	
	SocketPtr client_sock(new RefCountableSAP<ACE_SOCK_Stream>);
	
	{ // leader lock scope
		ACE_Guard<LeaderMutex> leader_guard(m_leaderMutex, /* block = */0);
		if (!leader_guard.locked()) {
			return false;
		}
		
		ConnAcceptor::Status const conn_status = m_connAcceptor.accept(
			*client_sock, service_context.clientAddr()
		);
		if (conn_status != ConnAcceptor::ACCEPTED) {
			return true;
		}
	}
	
	// Now we need a new leader.
	
	{ // m_mutex lock scope
		ACE_Guard<Mutex> guard(m_mutex);
		if (m_status != ACTIVE) {
			assert(m_status == STOPPING);
			return true;
		}
		try {
			prepareNewThreadLocked();
		} catch (...) {
			DEBUGLOG2("unable to create a worker thread");
			// Actually it's not a big problem. The first worker
			// thread to finish its work will become a new leader.
			// If there are no other worker threads, that would be
			// us after we finish our work. So, accepting
			// connections can pause but won't stop.
		}
	}
	
	processConnectionUnlocked(service_context, client_sock);
	return true;
}

void
WorkerThreadPool::processConnectionUnlocked(
	ServiceContext& service_context, SocketPtr const& client_sock)
{
	Client client(service_context, client_sock);
	Reactor& reactor = service_context.reactor();
	Reactor::Status const reactor_status = reactor.runEventLoop();
	if (reactor_status == Reactor::STOPPED) {
		reactor.restart();
	}
	
	// Should be called after the event loop, not before it.
	// This way we release some resources before going to sleep
	// waiting for a new task, not after it.
	service_context.reset();
}

ACE_THR_FUNC_RETURN
WorkerThreadPool::runThread(void* arg)
{
	static_cast<WorkerThreadPool*>(arg)->runServiceUnlocked();
	return 0;
}


/*================ WorkerThreadPool::ReactorRegistration ===============*/

WorkerThreadPool::ReactorRegistration::ReactorRegistration(
	Reactor* reactor, ReactorList& list)
:	m_pReactor(reactor)
{
	list.push_back(*this);
}
