/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include "ReplacementFlash.h"
#include "Color.h"
#include "SplittableBuffer.h"
#include "DataChunk.h"
#include "CraftedResponse.h"
#include "Date.h"
#include "GlobalState.h"
#include "Conf.h"
#include <stddef.h>
#include <cstring>
#include <memory>
#include <cassert>

using namespace std;

class ReplacementFlash::Buffer
{
public:
	Buffer(size_t size);
	
	size_t getDataSize() const { return m_dataSize; }
	
	template<typename T> void write(T value, size_t pos);
	
	void advance(size_t size);
	
	void append(void const* data, size_t size);
	
	template<typename T> void append(T value);
	
	template<typename T> void appendBits(T value, int nbits);
	
	void appendRect(int left, int right, int top, int bottom);
	
	void resetBitOffset() { m_bitOffset = 0; }
	
	std::auto_ptr<DataChunk> chunk() { return m_ptrChunk; }
private:
	std::auto_ptr<DataChunk> m_ptrChunk;
	size_t m_dataSize;
	int m_bitOffset;
};


ReplacementFlash::Buffer::Buffer(size_t size)
:	m_ptrChunk(DataChunk::create(size)),
	m_dataSize(0),
	m_bitOffset(0)
{
}

template<typename T> void
ReplacementFlash::Buffer::write(T value, size_t pos)
{
	assert(pos + sizeof(T) <= m_dataSize);
	char* ptr = m_ptrChunk->getDataAddr() + pos;
	for (unsigned shift = 0; shift < sizeof(T)*8; shift += 8) {
		*ptr++ = (value>>shift);
	}
}

template<typename T> void
ReplacementFlash::Buffer::append(T value)
{
	size_t pos = m_dataSize;
	m_dataSize += sizeof(value);
	assert(m_dataSize <= m_ptrChunk->getDataSize());
	write(value, pos);
}

template<typename T> void
ReplacementFlash::Buffer::appendBits(T value, int nbits)
{
	char* ptr = m_ptrChunk->getDataAddr() + m_dataSize;
	if (m_bitOffset > 0) {
		int bits_avail = 8 - m_bitOffset;
		if (nbits <= bits_avail) {
			bits_avail -= nbits;
			*(ptr-1) |= ((value & ((1<<nbits)-1))<<bits_avail);
			m_bitOffset = ((m_bitOffset+nbits)&7);
			return;
		} else {
			nbits -= bits_avail;
			*(ptr-1) |= ((value>>nbits)&((1<<bits_avail)-1));
		}
	}
	m_dataSize += (nbits+7)>>3;
	assert(m_dataSize <= m_ptrChunk->getDataSize());
	while (true) {
		if (nbits <= 8) {
			*ptr = (value<<(8-nbits));
			m_bitOffset = nbits&7;
			return;
		} else {
			nbits -= 8;
			*ptr++ = (value>>nbits);
		}
	}
}

void
ReplacementFlash::Buffer::advance(size_t size)
{
	m_dataSize += size;
	assert(m_dataSize <= m_ptrChunk->getDataSize());
}

void
ReplacementFlash::Buffer::append(void const* data, size_t size)
{
	char* ptr = m_ptrChunk->getDataAddr() + m_dataSize;
	m_dataSize += size;
	assert(m_dataSize <= m_ptrChunk->getDataSize());
	memcpy(ptr, data, size);
}

// if called with m_bitOffset == 0, it always appends 17 bytes
void
ReplacementFlash::Buffer::appendRect(int left, int right, int top, int bottom)
{
	static const int nbits = (1<<5)-1;
	appendBits(nbits, 5);
	appendBits(left, nbits);
	appendBits(right, nbits);
	appendBits(top, nbits);
	appendBits(bottom, nbits);
	resetBitOffset();
}


ReplacementFlash::ReplacementFlash(int width, int height,
	char const* orig_url, Color const* border_color,
	unsigned int frame_rate)
{
	if (width <= 0) {
		width = 1;
	} else if (width > 1600) {
		width = 1600;
	}
	if (height <= 0) {
		height = 1;
	} else if (height > 1600) {
		height = 1600;
	}
	
	size_t const url_size = strlen(orig_url) + 1;
	size_t const total_size = 133 + url_size;
	Buffer buf(total_size);
	
	int width_twips = width * 20;
	int height_twips = height * 20;
	
	static uint8_t const hdr[] = {'F','W','S',4}; // flash version 4 
	buf.append(hdr, sizeof(hdr));
	buf.append<uint32_t>(total_size);
	
	buf.appendRect(0, width_twips, 0, height_twips); // frame dimentions
	buf.append<uint16_t>(frame_rate); // frame rate
	buf.append<uint16_t>(1); // frame count
	
	buf.append<uint16_t>((22<<6)|0x3f); // DefineShape2 tag
	size_t shape_length_pos = buf.getDataSize();
	buf.advance(4); // shape length (uint32_t) will be written later
	buf.append<uint16_t>(1); // char_id
	buf.appendRect(0, width_twips, 0, height_twips);
	
#if 1
	{
		static uint8_t const data[] = {
			0x01, 0x00, 0xff, 0xff, 0xff, 0x01, 20, 0
		};
		buf.append(data, sizeof(data));
	}
#else
	buf.append<uint8_t>(0x01); // count of fill styles
	buf.append<uint8_t>(0x00); // type of fill style: solid fill
	buf.append<uint8_t>(0xff); // fill color (r)
	buf.append<uint8_t>(0xff); // fill color (g)
	buf.append<uint8_t>(0xff); // fill color (b)
	buf.append<uint8_t>(0x01); // count of line styles
	buf.append<uint16_t>(20);  // line width in twips
#endif
	
	if (border_color && !(width < 3 && height < 3)) {
		buf.append(border_color->getRGB(), 3);
	} else {
		static uint8_t const data[] = {0xff, 0xff, 0xff};
		buf.append(data, sizeof(data));
	}
	
#if 1
	static uint8_t const data3[] = {
		0x88, 0x2e, 0x80, 0x00, 0x14, 0x00, 0x01, 0x40, 0x20, 0x3f
	};
	buf.append(data3, sizeof(data3));
	buf.appendBits(0x04, 3);
#else
	buf.append<uint8_t>(0x88); // number of fill style index bits (8), and line style index bits (8)
	buf.appendBits(11, 6); // flags: non-edge record, move, change fill 0 style, change line style
	buf.appendBits(20, 5); // move bit count: 20
	buf.appendBits(10, 20); // move to: originX+10
	buf.appendBits(10, 20); // move to: originY+10
	buf.appendBits(1, 8); // fill 0 style: 1
	buf.appendBits(1, 8); // line style: 1
	buf.appendBits(0xfc, 8); // 11111100, horizontal line, num bits: 15+2
#endif
	
	buf.appendBits(width_twips - 20, 17); // delta x
	buf.appendBits(0xfd, 8); // 11111101, vertical line, num bits: 15+2
	buf.appendBits(height_twips - 20, 17); // delta y
	buf.appendBits(0xfc, 8);
	buf.appendBits(-(width_twips - 20), 17);
	buf.appendBits(0xfd, 8);
	buf.appendBits(-(height_twips - 20), 17);
	buf.appendBits(0, 6); // EndShapeRecord
	buf.resetBitOffset();
	buf.write<uint32_t>(buf.getDataSize()-shape_length_pos-4, shape_length_pos);
	
	buf.append<uint16_t>((7<<6)|0x3f); // Button tag
	size_t button_length_pos = buf.getDataSize();
	
#if 1
	{
		static uint8_t const data[] = {
			0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x0f,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x83
		};
		buf.append(data, sizeof(data));
	}
#else
	buf.advance(4); // button length (uint32_t) will be written later
	buf.append<uint16_t>(2); // char_id
	buf.append<uint8_t>(0x0f); // flags: present in all states
	buf.append<uint16_t>(1); // char_id to place
	buf.append<uint16_t>(1); // place depth
	buf.appendBits(0, 2); // hasScale: no, hasRotate: no
	buf.appendBits(1, 5); // nTranslateBits: 1
	buf.appendBits(0, 2); // xTranslate = 0, yTranslate = 0
	buf.resetBitOffset();
	buf.append<uint8_t>(0); // CharacterEndFlag
	buf.append<uint8_t>(0x83); // ActionGetURL
#endif
	
	static char const target[] = "_level0";
	buf.append<uint16_t>(url_size + sizeof(target)); // ActionGetURL length
	buf.append(orig_url, url_size);
	buf.append(target, sizeof(target));
	buf.append<uint8_t>(0); // ActionEndFlag
	buf.write<uint32_t>(buf.getDataSize() - button_length_pos - 4, button_length_pos);
	
#if 1
	{
		static uint8_t const data[] = {
			0xbf, 0x06, 0x07, 0x00, 0x00, 0x00, 0x06, 0x01,
			0x00, 0x02, 0x00, 0x02, 0x00, 0x40, 0x00, 0x00, 0x00
		};
		buf.append(data, sizeof(data));
	}
#else
	buf.append<uint16_t>((26<<6)|0x3f); // PlaceObject2 tag
	size_t place_object_length_pos = buf.getDataSize();
	buf.advance(4); // length will be written later
	buf.append<uint8_t>(0x06); // place character, has matrix
	buf.append<uint16_t>(1); // depth
	buf.append<uint16_t>(2); // char_id to place
	buf.appendBits(0, 2); // hasScale: no, hasRotate: no
	buf.appendBits(1, 5); // nTranslateBits: 1
	buf.appendBits(0, 2); // xTranslate = 0, yTranslate = 0
	buf.resetBitOffset();
	buf.write<uint32_t>(buf.getDataSize() - place_object_length_pos - 4, place_object_length_pos);
	buf.append<uint16_t>(1<<6); // show frame
	buf.append<uint16_t>(0); // end tag
#endif
	
	assert(buf.getDataSize() == total_size);
	
	m_binaryData.append(DataChunk::resize(buf.chunk(), buf.getDataSize()));
}

auto_ptr<CraftedResponse>
ReplacementFlash::createHttpResponse(
	bool is_head_response, int width, int height,
	std::string orig_url, unsigned int frame_rate)
{
	auto_ptr<CraftedResponse> response(new CraftedResponse(HttpStatusLine::SC_OK));
	response->metadata().headers().setHeader(
		HttpHeader(BString("Content-Type"), getContentType())
	);
	response->metadata().headers().setHeader(
		HttpHeader(BString("Date"), Date::formatCurrentTime())
	);
	if (is_head_response) {
		response->metadata().setBodyStatus(HttpResponseMetadata::BODY_FORBIDDEN);
	} else {
		auto_ptr<Color> color(GlobalState::ReadAccessor()->config().getBorderColor());
		ReplacementFlash flash(width, height, orig_url.c_str(), color.get(), frame_rate);
		response->body().appendDestructive(flash.binaryData());
		response->metadata().setBodyStatus(HttpResponseMetadata::BODY_SIZE_KNOWN);
		response->metadata().setBodySize(response->body().size());
	}
	return response;
}
