/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ReactorHelpers.h"
#include "MtGuard.h"
#include "MonotonicTimer.h"
#include <ace/config-lite.h>
#include <ace/Lock.h>
#include <ace/OS_NS_unistd.h>
#include <ace/OS_NS_sys_time.h>
#ifdef ACE_WIN32
#include <mmsystem.h> // for timeGetTime()
#endif
#include <utility>
#include <cassert>
#include <time.h>

namespace ReactorHelpers {

using namespace std;


ReactorHandlerId
HandlerRepository::findByHandle(ACE_HANDLE handle) const
{
	HandleIdx const& idx = get<HandleTag>();
	HandleIter it = idx.find(handle);
	if (it == idx.end()) {
		return ReactorHandlerId();
	} else {
		return ReactorHandlerId(project<SequenceTag>(it));
	}
}

ReactorHandlerId
HandlerRepository::add(Handler const& new_handler)
{
	pair<iterator, bool> ins = insert(first_inactive, new_handler);
	if (!ins.second) {
		throw Reactor::DuplicateHandleException();
	}
	if (new_handler.events == 0) {
		first_inactive = ins.first;
	}
	return ReactorHandlerId(ins.first);
}

void
HandlerRepository::remove(ReactorHandlerId const& id)
{
	iterator const& it = iterFromId(id);
	if (it == first_inactive) {
		++first_inactive;
	}
	erase(it);
}

void
HandlerRepository::setEvents(iterator const& it, IOEvents events)
{
	IOEvents const old_events = it->events;
	if (old_events == events) {
		return;
	}
	
	if (old_events != 0 && events == 0) {
		// was active, becomes inactive
		relocate(first_inactive, it);
		first_inactive = it;
	} else if (old_events == 0 && events != 0) {
		// was inactive, becomes active
		if (first_inactive == it) {
			++first_inactive;
		} else {
			relocate(first_inactive, it);
		}
	}
	const_cast<Handler&>(*it).events = events;
}


/*============================= TimerQueue ==============================*/

TimerQueue::TimerQueue()
{
}

TimerQueue::~TimerQueue()
{
}

ReactorTimerId
TimerQueue::add(EventHandlerPtr const& handler, TimeDelta const* timeout)
{
	TimeStamp abs_time = timeoutToAbsTime(timeout);
	ensureTimeAfterThreshold(abs_time);
	
	pair<iterator, bool> ins = insert(Timer(handler, abs_time));
	assert(ins.second);
	return ReactorTimerId(ins.first);
}

void
TimerQueue::reschedule(ReactorTimerId const& id, TimeDelta const* timeout)
{
	TimeStamp abs_time = timeoutToAbsTime(timeout);
	ensureTimeAfterThreshold(abs_time);
	iterator const& it = iterFromId(id);
	replace(it, Timer(it->handler, abs_time));
}

TimeDelta
TimerQueue::getRemainingTime(ReactorTimerId const& id) const
{
	iterator const& it = iterFromId(id);
	if (it->abs_time == TimeStamp::max()) {
		return TimeDelta::max();
	} else {
		return it->abs_time - MonotonicTimer::getTimestamp();
	}
}

TimeStamp
TimerQueue::getEarliestTime() const
{
	if (empty()) {
		return TimeStamp::max();
	} else {
		return begin()->abs_time;
	}
}

void
TimerQueue::updateDispatchThreshold()
{
	m_dispatchThreshold = MonotonicTimer::getTimestamp();
	
	m_dispatchThreshold = TimeStamp::fromUsec(m_dispatchThreshold.toUsec() + 1);
	// Compensate for ensureTimeAfterThreshold().
	// Otherwise zero timers may require multiple demultiplexer calls,
	// because of the low resolution of the system timer.
}

void
TimerQueue::continueDispatching(ACE_Lock& mutex, bool& is_stopped)
{
	assert(m_dispatchThreshold < TimeStamp::max());
	while (!is_stopped && !empty()) {
		iterator it = begin();
		if (it->abs_time > m_dispatchThreshold) {
			// abs_time could also be TimeStamp::max()
			break;
		}
		
		replace(it, Timer(it->handler, TimeStamp::max()));
		
		ReactorTimerId timer_id(it);
		MtAntiGuard<ACE_Lock> anti_guard(mutex);
		it->handler->handleTimeout(timer_id);
	}
}

TimeStamp
TimerQueue::timeoutToAbsTime(TimeDelta const* timeout) const
{
	if (!timeout) {
		return TimeStamp::max();
	}
	
	TimeStamp abs_time = MonotonicTimer::getTimestamp();
	abs_time += *timeout;
	return abs_time;
}

void
TimerQueue::ensureTimeAfterThreshold(TimeStamp& abs_time) const
{
	if (abs_time <= m_dispatchThreshold) {
		abs_time = TimeStamp::fromUsec(m_dispatchThreshold.toUsec() + 1);
	}
}


/*========================= DemultiplexerGuard ========================*/

/*
A typical code for gaining exclusive access to the data structure
used by the demultiplexer function (select, poll, WaitForMultipleObjects)
is like this:
-------------------------------------------------
MtGuard<ACE_Lock> guard(*m_ptrMutex);
if (m_isInsideDemux) { // protected by m_ptrMutex
	wakeup();
}
MtGuard<ACE_Lock> demux_guard(*m_ptrDemuxMutex);
// here we have exclusive access
-------------------------------------------------
The DemultiplexerGuard class helps to simplify the code that wraps the
demultiplexer function. It's used like this:
{
	DemultiplexerGuard guard(*m_ptrMutex, *m_ptrDemuxMutex, m_isInsideDemux);
	poll(...);
}
*/

DemultiplexerGuard::DemultiplexerGuard(
	ACE_Lock& mutex, ACE_Lock& demux_mutex, bool& is_inside_demux)
:	m_rMutex(mutex),
	m_rDemuxMutex(demux_mutex),
	m_rInsideDemux(is_inside_demux)
{
	m_rDemuxMutex.acquire();
	m_rInsideDemux = true; // protected by m_rMutex
	m_rMutex.release();
	/*
	Here another thread may acquire m_rMutex,
	but if it tries to acquire m_rDemuxMutex as well,
	it will block until we release it.
	*/
}

// The demultiplexer function is called between the CTOR and DTOR.

DemultiplexerGuard::~DemultiplexerGuard()
{
	m_rDemuxMutex.release();
	/*
	Here another thread finally acquires m_rDemuxMutex
	(after having acquired m_ptrMutex earlier), and thus
	gets the exclusive access to the demultiplexer data structure.
	*/
	m_rMutex.acquire(); // will wait until another thread releases it
	m_rInsideDemux = false; // protected by m_rMutex
}

} // namespace ReactorHelpers
