/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ContentFiltersFile.h"
#include "ContentFilters.h"
#include "ConfError.h"
#include "ConfErrorHandler.h"
#include "RegexFilterDescriptor.h"
#include "FilterTag.h"
#include "FilterGroupTag.h"
#include "TextPattern.h"
#include "StringUtils.h"
#include "IntrusivePtr.h"
#include "ArraySize.h"
#include <ace/config-lite.h>
#include <boost/regex.hpp>
#include <boost/lexical_cast.hpp>
#include <set>
#include <string>
#include <memory>
#include <istream>
#include <ostream>
#include <algorithm>
#include <cassert>

using namespace std;

struct ContentFiltersFile::SectionBoundries
{
	typedef std::list<ConfIO::Element>::iterator ElementIterator;
	ElementIterator first;
	ElementIterator sectionElement;
	ElementIterator last;
	// Both first and last a are inclusive.
	// This was done to make moving sections easy.
	
	SectionBoundries(
		ElementIterator const& begin,
		ElementIterator const& section_element,
		ElementIterator const& end)
	: first(begin), sectionElement(section_element), last(end) {
		--last;
	}
	
	ElementIterator const& begin() const { return first; }
	
	ElementIterator end() const { ElementIterator end = last; return ++end; }
};


class ContentFiltersFile::ParamValue
{
public:
	ParamValue() : m_isNull(true) {}
	
	ParamValue(std::string const& val)
	: m_value(val), m_isNull(false) {}
	
	std::string const& getValue() const { return m_value; }
	
	bool isNull() const { return m_isNull; }
private:
	std::string m_value;
	bool m_isNull;
};


class ContentFiltersFile::Reader : public ConfIO
{
public:
	Reader(std::list<Element>& elements,
		ContentFilters& target, ConfErrorHandler& eh,
		FilterGroupTag const& group_tag);
	
	virtual ~Reader();
	
	bool read(std::string const& str);
private:
	typedef bool (Reader::*ParamHandlerPtr)(std::string const& value, int line);
	
	struct ParamHandlerNode
	{
		char const* name;
		ParamHandlerPtr handler;
		
		bool operator==(std::string const& rhs) const {
			return name == rhs;
		}
	};
	
	virtual bool processElement(Element const& element, int line);
	
	virtual bool processError(ConfError const& error);
	
	bool validateSection(std::string const& section, int line);
	
	bool processKeyValue(
		std::string const& key,
		std::string const& value, int line);
	
	bool handleParam(
		ParamHandlerNode const* begin, ParamHandlerNode const* end,
		std::string const& key, std::string const& value, int line);
	
	bool handleOrder(std::string const& value, int line);
	
	bool handleMatchCountLimit(std::string const& value, int line);
	
	bool handleUrl(std::string const& value, int line);
	
	bool handleContentType(std::string const& value, int line);
	
	bool handleSearch(std::string const& value, int line);
	
	bool handleReplace(std::string const& value, int line);
	
	bool handleReplacementType(std::string const& value, int line);
	
	bool handleIfFlag(std::string const& value, int line);
	
	bool handleSetFlag(std::string const& value, int line);
	
	bool handleClearFlag(std::string const& value, int line);
	
	bool makePattern(
		IntrusivePtr<TextPattern const>& target,
		std::string const& text, int line);
	
	void commitLastFilter();
	
	std::list<Element>& m_rElements;
	ContentFilters& m_rTarget;
	ConfErrorHandler& m_rErrorHandler;
	FilterGroupTag m_groupTag;
	IntrusivePtr<RegexFilterDescriptor> m_ptrFilter;
	std::set<std::string> m_sections;
	std::set<std::string> m_keys; // withing the current section
	int m_lastSectionLocation;
};


class ContentFiltersFile::Updater
{
public:
	typedef ConfIO::Element Element;
	
	Updater(std::list<Element>& elements, ContentFilters const& source);
	
	~Updater();
	
	void update();
private:
	typedef ConfIO::ElementIterator ElementIterator;
	typedef std::list<SectionBoundries>::iterator SectionIter;
	typedef ParamValue (Updater::*ParamHandlerPtr)(
		RegexFilterDescriptor const& filter);
	
	struct ParamHandlerNode
	{
		bool processed;
		char const* name;
		ParamHandlerPtr handler;
		
		bool operator==(std::string const& rhs) const {
			return name == rhs;
		}
	};
	
	void getSectionBoundries(std::list<SectionBoundries>& boundries);
	
	void reorderSections(std::list<SectionBoundries>& boundries);
	
	static SectionIter findSectionByName(SectionIter const& begin,
		SectionIter const& end, std::string const& name);
	
	void moveSectionBefore(
		std::list<SectionBoundries>& boundries,
		SectionIter const& sec, SectionIter const& move_pos);
	
	void createNewSectionBefore(
		std::list<SectionBoundries>& boundries,
		std::string const& name, SectionIter const& insert_pos);
	
	void processSections(std::list<SectionBoundries>& boundries);
	
	void processSection(
		RegexFilterDescriptor const& filter,
		SectionBoundries& section);
	
	void processParams(
		RegexFilterDescriptor const& filter,
		SectionBoundries& section,
		ParamHandlerNode* h_begin,
		ParamHandlerNode* h_end);
	
	ParamValue handleOrder(RegexFilterDescriptor const& filter);
	
	ParamValue handleMatchCountLimit(RegexFilterDescriptor const& filter);
	
	ParamValue handleUrl(RegexFilterDescriptor const& filter);
	
	ParamValue handleContentType(RegexFilterDescriptor const& filter);
	
	ParamValue handleSearch(RegexFilterDescriptor const& filter);
	
	ParamValue handleReplace(RegexFilterDescriptor const& filter);
	
	ParamValue handleReplacementType(RegexFilterDescriptor const& filter);
	
	ParamValue handleIfFlag(RegexFilterDescriptor const& filter);
	
	ParamValue handleSetFlag(RegexFilterDescriptor const& filter);
	
	ParamValue handleClearFlag(RegexFilterDescriptor const& filter);
	
	static bool rangeContainsComments(
		ElementIterator const& begin, ElementIterator const& end);
	
	static void commentOut(Element& el);
	
	std::list<Element>& m_rElements;
	ContentFilters const& m_rSource;
};


/*========================= ContentFiltersFile =========================*/

void
ContentFiltersFile::clear()
{
	m_elements.clear();
}

void
ContentFiltersFile::load(std::string const& text,
	ContentFilters& target, ConfErrorHandler& eh,
	FilterGroupTag const& group_tag)
{
	clear();
	Reader reader(m_elements, target, eh, group_tag);
	reader.read(text);
}

void
ContentFiltersFile::updateWith(ContentFilters const& filters)
{
	Updater updater(m_elements, filters);
	updater.update();
}

void
ContentFiltersFile::renameFilter(size_t pos, std::string const& new_name)
{
	list<ConfIO::Element>::iterator it = m_elements.begin();
	list<ConfIO::Element>::iterator const end = m_elements.end();
	for (; it != end; ++it) {
		if (it->getType() == ConfIO::Element::SECTION) {
			if (pos == 0) {
				break;
			}
			--pos;
		}
	}
	assert(it != end);
	it->setValue(new_name);
}

void
ContentFiltersFile::swap(ContentFiltersFile& other)
{
	m_elements.swap(other.m_elements);
}

void
ContentFiltersFile::toStream(std::ostream& strm) const
{
	ConfIO::write(strm, m_elements);
}


/*====================== ContentFiltersFile::Reader =======================*/

ContentFiltersFile::Reader::Reader(
	list<Element>& elements, ContentFilters& target,
	ConfErrorHandler& eh, FilterGroupTag const& group_tag)
:	m_rElements(elements),
	m_rTarget(target),
	m_rErrorHandler(eh),
	m_groupTag(group_tag),
	m_lastSectionLocation(0)
{
}

ContentFiltersFile::Reader::~Reader()
{
}

bool
ContentFiltersFile::Reader::read(std::string const& str)
{
	bool not_aborted = ConfIO::read(str);
	if (not_aborted) {
		commitLastFilter();
	}
	return not_aborted;
}

bool
ContentFiltersFile::Reader::processElement(Element const& element, int line)
{
	if (element.getType() == Element::SECTION) {
		commitLastFilter();
		assert(!m_ptrFilter.get());
		if (!validateSection(element.getText(), line)) {
			return false;
		}
		m_ptrFilter.reset(new RegexFilterDescriptor(
			FilterTag(), m_groupTag, element.getText()
		));
		m_sections.insert(element.getText());
		m_keys.clear();
		m_lastSectionLocation = line;
	} else if (element.getType() == Element::KEY_VALUE) {
		if (m_ptrFilter) {
			if (!processKeyValue(element.getKey(), element.getValue(), line)) {
				return false;
			}
			m_keys.insert(element.getKey());
		}
	}
	m_rElements.push_back(element);
	return true;
}

bool
ContentFiltersFile::Reader::processError(ConfError const& error)
{
	return m_rErrorHandler.handleError(error);
}

bool
ContentFiltersFile::Reader::validateSection(
	std::string const& section, int line)
{
	if (m_sections.find(section) != m_sections.end()) {
		ConfError err(
			ConfError::T_WARNING,
			"Duplicate filter name: "+section, line
		);
		return m_rErrorHandler.handleError(err);
	}
	return true;
}

bool
ContentFiltersFile::Reader::processKeyValue(
	string const& key,
	string const& value, int line)
{
	assert(m_ptrFilter);
	
	if (m_keys.find(key) != m_keys.end()) {
		ConfError err(
			ConfError::T_ERROR,
			"Duplicate parameter", line
		);
		return m_rErrorHandler.handleError(err);
	}
	
	static ParamHandlerNode const handlers[] = {
		{ "order", &Reader::handleOrder },
		{ "match_count_limit", &Reader::handleMatchCountLimit },
		{ "url", &Reader::handleUrl },
		{ "content_type", &Reader::handleContentType },
		{ "search", &Reader::handleSearch },
		{ "replace", &Reader::handleReplace },
		{ "replacement_type", &Reader::handleReplacementType },
		{ "if_flag", &Reader::handleIfFlag },
		{ "set_flag", &Reader::handleSetFlag },
		{ "clear_flag", &Reader::handleClearFlag }
	};
	return handleParam(
		handlers, handlers + ARRAY_SIZE(handlers),
		key, value, line
	);
}

bool
ContentFiltersFile::Reader::handleParam(
	ParamHandlerNode const* begin, ParamHandlerNode const* end,
	std::string const& key, std::string const& value, int line)
{
	assert(m_ptrFilter);
	
	ParamHandlerNode const* node = std::find(begin, end, key);
	if (node != end) {
		return (this->*node->handler)(value, line);
	} else {
		ConfError err(
			ConfError::T_WARNING,
			"Unknown paramenter: "+key, line
		);
		return m_rErrorHandler.handleError(err);
	}
}

bool
ContentFiltersFile::Reader::handleOrder(std::string const& value, int line)
{
	int val;
	try {
		val = boost::lexical_cast<int>(value);
	} catch (boost::bad_lexical_cast& e) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"order\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_ptrFilter->order() = val;
	return true;
}

bool
ContentFiltersFile::Reader::handleMatchCountLimit(
	std::string const& value, int line)
{
	int val = 0;
	try {
		val = boost::lexical_cast<int>(value);
	} catch (boost::bad_lexical_cast& e) {}
	if (val <= 0) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"match_count_limit\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_ptrFilter->matchCountLimit() = val;
	return true;
}

bool
ContentFiltersFile::Reader::handleUrl(std::string const& value, int line)
{
	return makePattern(m_ptrFilter->urlPattern(), value, line);
}

bool
ContentFiltersFile::Reader::handleContentType(std::string const& value, int line)
{
	return makePattern(m_ptrFilter->contentTypePattern(), value, line);
}

bool
ContentFiltersFile::Reader::handleSearch(std::string const& value, int line)
{
	return makePattern(m_ptrFilter->searchPattern(), value, line);
}

bool
ContentFiltersFile::Reader::handleReplace(std::string const& value, int line)
{
	m_ptrFilter->replacement().reset(new string(value));
	return true;
}

bool
ContentFiltersFile::Reader::handleReplacementType(
	std::string const& value, int line)
{
	typedef RegexFilterDescriptor RFD;
	RFD::ReplacementType type;
	if (value == "text") {
		type = RFD::TEXT;
	} else if (value == "expression") {
		type = RFD::EXPRESSION;
	} else if (value == "js") {
		type = RFD::JS;
	} else {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"replacement_type\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_ptrFilter->replacementType() = type;
	return true;
}

bool
ContentFiltersFile::Reader::handleIfFlag(std::string const& value, int line)
{
	if (value.empty()) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"if_flag\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_ptrFilter->ifFlag() = value;
	return true;
}

bool
ContentFiltersFile::Reader::handleSetFlag(std::string const& value, int line)
{
	if (value.empty()) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"set_flag\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_ptrFilter->setFlag() = value;
	return true;
}

bool
ContentFiltersFile::Reader::handleClearFlag(std::string const& value, int line)
{
	if (value.empty()) {
		ConfError err(
			ConfError::T_ERROR,
			"Illegal value for \"clear_flag\"", line
		);
		return m_rErrorHandler.handleError(err);
	}
	m_ptrFilter->clearFlag() = value;
	return true;
}

bool
ContentFiltersFile::Reader::makePattern(
	IntrusivePtr<TextPattern const>& target,
	std::string const& text, int line)
{
	namespace rc = boost::regex_constants;
	try {
		target.reset(new TextPattern(
			text, rc::normal|rc::icase|rc::optimize
		));
	} catch (boost::bad_expression& e) {
		target.reset(new TextPattern(
			text, TextPattern::MAKE_INVALID
		));
		ConfError err(
			ConfError::T_ERROR,
			string("Bad regex: ")+e.what(), line
		);
		return m_rErrorHandler.handleError(err);
	}
	return true;
}

void
ContentFiltersFile::Reader::commitLastFilter()
{
	if (!m_ptrFilter) {
		return;
	}
	
	/*
	We commit the filter unconditionally, to be able to edit it via the GUI.
	The filter won't be applied, as filter->isValid() will be false.
	*/
	IntrusivePtr<RegexFilterDescriptor> filter;
	filter.swap(m_ptrFilter);
	m_rTarget.filters().push_back(filter);
	
	int const line = m_lastSectionLocation;
	
	if (!filter->searchPattern().get()) {
		ConfError err(
			ConfError::T_ERROR,
			"'search' is not specified for filter \""
			+filter->name()+"\"", line
		);
		m_rErrorHandler.handleError(err);
	}
	
	if (!filter->replacement().get()) {
		ConfError err(
			ConfError::T_ERROR,
			"'replace' is not specified for filter \""
			+filter->name()+"\"", line
		);
		m_rErrorHandler.handleError(err);
	}
}


/*===================== ContentFiltersFile::Updater =======================*/

ContentFiltersFile::Updater::Updater(
	list<Element>& elements, ContentFilters const& source)
:	m_rElements(elements),
	m_rSource(source)
{
}

ContentFiltersFile::Updater::~Updater()
{
}

void
ContentFiltersFile::Updater::update()
{
	list<SectionBoundries> boundries;
	getSectionBoundries(boundries);
	reorderSections(boundries);
	processSections(boundries);
}

void
ContentFiltersFile::Updater::getSectionBoundries(
	std::list<SectionBoundries>& boundries)
{
	ElementIterator const end = m_rElements.end();
	
	/*
	We need to skip the comments at the beginning of a file, as they
	don't belong to a particular section. We do it by pretending
	that there is another section that ends at the beginning of the file.
	*/
	ElementIterator sec_items_end = m_rElements.begin();
	ElementIterator sec_next_el = ConfIO::findNextSection(
		sec_items_end, end
	);
	ElementIterator sec_end = ConfIO::distributeComments(
		sec_items_end, sec_next_el
	);
	
	while (sec_next_el != end) {
		ElementIterator sec_begin = sec_end;
		ElementIterator sec_el = sec_next_el;
		sec_items_end = ConfIO::getSectionItemsEnd(sec_el, end);
		sec_next_el = ConfIO::findNextSection(sec_items_end, end);
		if (sec_next_el == end) {
			sec_end = end;
		} else {
			sec_end = ConfIO::distributeComments(
				sec_items_end, sec_next_el
			);
		}
		boundries.push_back(SectionBoundries(sec_begin, sec_el, sec_end));
	}
}

/*
This method makes the list of sections match the list of filters in m_rSource.
It reorders sections, adds missing ones and removes those that don't have
a corresponding filter.
*/
void
ContentFiltersFile::Updater::reorderSections(
	std::list<SectionBoundries>& boundries)
{
	typedef ContentFilters::FilterList FilterList;
	typedef FilterList::const_iterator FilterIter;
	FilterList const& filters = m_rSource.filters();
	FilterIter it = filters.begin();
	FilterIter const end = filters.end();
	SectionIter sec_it = boundries.begin();
	SectionIter const sec_end = boundries.end();
	for (; it != end; ++it) {
		RegexFilterDescriptor const& filter = **it;
		if (sec_it != sec_end && filter.name() ==
		    sec_it->sectionElement->getValue()) {
			++sec_it;
			continue;
		}
		
		SectionIter sec = findSectionByName(
			sec_it, sec_end, filter.name()
		);
		if (sec == sec_end) {
			createNewSectionBefore(boundries, filter.name(), sec_it);
		} else {
			moveSectionBefore(boundries, sec, sec_it);
		}
	}
	
	// Now [sec_it, sec_end) contains sections that don't have
	// a corresponding filter.
	if (sec_it != sec_end) {
		m_rElements.erase(sec_it->begin(), m_rElements.end());
		boundries.erase(sec_it, sec_end);
	}
}

ContentFiltersFile::Updater::SectionIter
ContentFiltersFile::Updater::findSectionByName(SectionIter const& begin,
	SectionIter const& end, std::string const& name)
{
	SectionIter it = begin;
	for (; it != end; ++it) {
		if (it->sectionElement->getValue() == name) {
			break;
		}
	}
	return it;
}

void
ContentFiltersFile::Updater::moveSectionBefore(
	std::list<SectionBoundries>& boundries,
	SectionIter const& sec, SectionIter const& move_pos)
{
	ElementIterator el_move_pos = m_rElements.end();
	if (move_pos != boundries.end()) {
		el_move_pos = move_pos->first;	
	}
	list<Element> temp1;
	temp1.splice(temp1.end(), m_rElements, sec->begin(), sec->end());
	m_rElements.splice(el_move_pos, temp1);
	
	list<SectionBoundries> temp2;
	temp2.splice(temp2.end(), boundries, sec);
	boundries.splice(move_pos, temp2);
}

void
ContentFiltersFile::Updater::createNewSectionBefore(
	std::list<SectionBoundries>& boundries,
	std::string const& name, SectionIter const& insert_pos)
{
	ElementIterator el_insert_pos = m_rElements.end();
	if (insert_pos != boundries.end()) {
		el_insert_pos = insert_pos->first;
	}
	ElementIterator begin = m_rElements.insert(
		el_insert_pos, Element() // blank line
	);
	ElementIterator element = m_rElements.insert(
		el_insert_pos, Element(name) // section
	);
	ElementIterator end = element;
	++end;
	boundries.insert(insert_pos, SectionBoundries(begin, element, end));
}

void
ContentFiltersFile::Updater::processSections(
	std::list<SectionBoundries>& boundries)
{
	typedef ContentFilters::FilterList FilterList;
	typedef FilterList::const_iterator FilterIter;
	FilterList const& filters = m_rSource.filters();
	FilterIter it = filters.begin();
	FilterIter const end = filters.end();
	SectionIter sec_it = boundries.begin();
	SectionIter const sec_end = boundries.end();
	for (; it != end; ++it, ++sec_it) {
		RegexFilterDescriptor const& filter = **it;
		assert(sec_it != sec_end);
		assert(sec_it->sectionElement->getValue() == filter.name());
		processSection(filter, *sec_it);
	}
	assert(sec_it == sec_end);
}

void
ContentFiltersFile::Updater::processSection(
	RegexFilterDescriptor const& filter,
	SectionBoundries& section)
{
	static ParamHandlerNode const handlers_proto[] = {
		{ false, "order", &Updater::handleOrder },
		{ false, "match_count_limit", &Updater::handleMatchCountLimit  },
		{ false, "url", &Updater::handleUrl },
		{ false, "content_type", &Updater::handleContentType },
		{ false, "search", &Updater::handleSearch },
		{ false, "replace", &Updater::handleReplace },
		{ false, "replacement_type", &Updater::handleReplacementType },
		{ false, "if_flag", &Updater::handleIfFlag },
		{ false, "set_flag", &Updater::handleSetFlag },
		{ false, "clear_flag", &Updater::handleClearFlag }
	};
	ParamHandlerNode handlers[ARRAY_SIZE(handlers_proto)];
	memcpy(handlers, handlers_proto, sizeof(handlers));
	
	processParams(filter, section, handlers, handlers + ARRAY_SIZE(handlers));
}

void
ContentFiltersFile::Updater::processParams(
	RegexFilterDescriptor const& filter,
	SectionBoundries& section,
	ParamHandlerNode* h_begin, ParamHandlerNode* h_end)
{
	ElementIterator it = section.begin();
	ElementIterator next = it;
	ElementIterator const end = section.end();
	for (; it != end; it = next) {
		++next;
		if (it->getType() != Element::KEY_VALUE) {
			continue;
		}
		ParamHandlerNode* node = std::find(
			h_begin, h_end, it->getKey()
		);
		if (node == h_end) {
			// unknown parameter
			commentOut(*it);
			continue;
		}
		if (node->processed) {
			// duplicate parameter
			commentOut(*it);
			continue;
		}
		node->processed = true;
		
		ParamValue const val = (this->*node->handler)(filter);
		if (!val.isNull()) {
			it->setValue(val.getValue());
		} else {
			ElementIterator param_begin(
				ConfIO::getParamAreaBegin(section.begin(), it)
			);
			ElementIterator param_end(
				ConfIO::getParamAreaEnd(it, section.end())
			);
			if (rangeContainsComments(param_begin, param_end)) {
				commentOut(*it);
			} else {
				if (section.end() == param_end) {
					section.last = param_begin;
					--section.last;
				}
				m_rElements.erase(param_begin, param_end);
				next = param_end;
			}
		}
	}
	
	// Now process parameters that weren't encountered in the file.
	
	for (ParamHandlerNode* node = h_begin; node != h_end; ++node) {
		if (node->processed) {
			continue;
		}
		node->processed = true;
		
		ParamValue const val = (this->*node->handler)(filter);
		if (!val.isNull()) {
			m_rElements.insert(end, Element()); // blank line
			list<Element>::iterator el_pos = m_rElements.insert(
				end, Element(node->name, val.getValue())
			);
		}
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleOrder(RegexFilterDescriptor const& filter)
{
	if (filter.order() == 0) {
		return ParamValue();
	} else {
		return StringUtils::fromNumber(filter.order());
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleMatchCountLimit(
	RegexFilterDescriptor const& filter)
{
	if (filter.matchCountLimit() == -1) {
		return ParamValue();
	} else {
		return StringUtils::fromNumber(filter.matchCountLimit());
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleUrl(RegexFilterDescriptor const& filter)
{
	if (!filter.urlPattern().get()) {
		return ParamValue();
	} else {
		return filter.urlPattern()->source();
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleContentType(
	RegexFilterDescriptor const& filter)
{
	if (!filter.contentTypePattern().get()) {
		return ParamValue();
	} else {
		return filter.contentTypePattern()->source();
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleSearch(RegexFilterDescriptor const& filter)
{
	if (!filter.searchPattern().get()) {
		return ParamValue();
	} else {
		return filter.searchPattern()->source();
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleReplace(RegexFilterDescriptor const& filter)
{
	if (!filter.replacement().get()) {
		return ParamValue();
	} else {
		return *filter.replacement();
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleReplacementType(
	RegexFilterDescriptor const& filter)
{
	typedef RegexFilterDescriptor RFD;
	char const* type;
	switch (filter.replacementType()) {
		case RFD::EXPRESSION: {
			type = "expression";
			break;
		}
		case RFD::JS: {
			type = "js";
			break;
		}
		default: {
			type = "text";
			break;
		}
	}
	return ParamValue(type);
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleIfFlag(
	RegexFilterDescriptor const& filter)
{
	if (filter.ifFlag().empty()) {
		return ParamValue();
	} else {
		return filter.ifFlag();
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleSetFlag(
	RegexFilterDescriptor const& filter)
{
	if (filter.setFlag().empty()) {
		return ParamValue();
	} else {
		return filter.setFlag();
	}
}

ContentFiltersFile::ParamValue
ContentFiltersFile::Updater::handleClearFlag(
	RegexFilterDescriptor const& filter)
{
	if (filter.clearFlag().empty()) {
		return ParamValue();
	} else {
		return filter.clearFlag();
	}
}

bool
ContentFiltersFile::Updater::rangeContainsComments(
	ElementIterator const& begin, ElementIterator const& end)
{
	ElementIterator it = begin;
	for (; it != end; ++it) {
		if (it->getType() == Element::COMMENT) {
			return true;
		}
	}
	return false;
}

void
ContentFiltersFile::Updater::commentOut(Element& el)
{
	if (el.getType() == Element::COMMENT) {
		return;
	}
	
	ostringstream strm;
	el.toStream(strm);
	string str = strm.str();
	
	if (str.find('\n') == string::npos) {
		el = Element('#', str);
	} else {
		el = Element('#', " <multi-line value was there>");
	}
}
