/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AnimatedTrayIcon.h"
#include <glibmm/main.h>
#include <algorithm>
#include <cassert>

using namespace std;

AnimatedTrayIcon::AnimatedTrayIcon()
{
	Gtk::Window* wnd = m_statusDocklet.getWindow();
	assert(wnd);
	m_image.show();
	wnd->add(m_image);
}

AnimatedTrayIcon::~AnimatedTrayIcon()
{
}

void
AnimatedTrayIcon::setIcon(
	Glib::RefPtr<Gdk::Pixbuf> const& icon,
	Glib::ustring const& tooltip, int flags)
{
	m_nextAnimation.clear();
	m_nextAnimation.frames.assign(1, Frame(icon, 0));
	m_nextAnimation.tooltip = tooltip;
	m_nextAnimation.flags = flags;
	if (!(flags & FINISH_CURRENT_ITERATION)
	    || m_curAnimation.isFinalFrame()) {
		switchToNextAnimation();
	}
	m_statusDocklet.getWindow()->show();
}

void
AnimatedTrayIcon::removeIcon()
{
	m_timerConnection.disconnect();
	m_statusDocklet.getWindow()->hide();
	m_image.clear();
	m_curAnimation.clear();
	m_nextAnimation.clear();
}

void
AnimatedTrayIcon::setPixbuf(Glib::RefPtr<Gdk::Pixbuf> const& pixbuf)
{
	m_image.set(pixbuf);
	m_statusDocklet.getWindow()->show();
}

void
AnimatedTrayIcon::setTooltip(Glib::ustring const& tooltip)
{
	Gtk::Window* wnd = m_statusDocklet.getWindow();
	assert(wnd);
	m_tooltips.set_tip(*wnd, tooltip);
}

bool
AnimatedTrayIcon::onTimer()
{
	if (m_curAnimation.isFinalFrame()) {
		return false; // stop the timer
	}
	
	if (++m_curAnimation.curFrame < m_curAnimation.frames.size()) {
		switchToFrame(m_curAnimation.curFrame);
	} else if (!m_nextAnimation.frames.empty()) {
		switchToNextAnimation();
	} else if (m_curAnimation.flags & LOOP) {
		assert(!m_curAnimation.frames.empty());
		switchToFrame(0);
	}
	
	return false; // stop the timer
}

void
AnimatedTrayIcon::switchToFrame(int frame)
{
	m_curAnimation.curFrame = frame;
	Frame& frm = m_curAnimation.frames[frame];
	setPixbuf(frm.icon);
	if (m_curAnimation.isFinalFrame()) {
		m_timerConnection.disconnect();
	} else {
		m_timerConnection = Glib::signal_timeout().connect(
			sigc::mem_fun(*this, &AnimatedTrayIcon::onTimer),
			frm.delay_ms
		);
	}
}

void
AnimatedTrayIcon::switchToNextAnimation()
{
	m_timerConnection.disconnect();
	m_nextAnimation.swap(m_curAnimation);
	m_nextAnimation.clear();
	setTooltip(m_curAnimation.tooltip);
	if (m_curAnimation.frames.empty()) {
		m_statusDocklet.getWindow()->hide();
		m_image.clear();
	} else {
		switchToFrame(0);
	}
}


/*==================== AnimatedTrayIcon::Animation ======================*/

bool
AnimatedTrayIcon::Animation::isFinalFrame() const
{
	if (frames.size() < 2) {
		return true;
	}
	if (flags & LOOP) {
		return false;
	}
	if (curFrame < frames.size() - 1) {
		return false;
	}
	return true;
}

void
AnimatedTrayIcon::Animation::clear()
{
	frames.clear();
	curFrame = 0;
	tooltip.clear();
	flags = 0;
}

void
AnimatedTrayIcon::Animation::swap(Animation& other)
{
	frames.swap(other.frames);
	std::swap(curFrame, other.curFrame);
	std::swap(tooltip, other.tooltip);
	std::swap(flags, other.flags);
}
