/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Daemon.h"
#include "Conf.h"
#include "ConfError.h"
#include "ConfErrorHandler.h"
#include "ConfigFile.h"
#include "UrlPatterns.h"
#include "UrlPatternsFile.h"
#include "ContentFilters.h"
#include "FilterGroupTag.h"
#include "ContentFiltersFile.h"
#include "CombinedUrlPatterns.h"
#include "RegexFilterDescriptor.h"
#include "GlobalState.h"
#include "ConnAcceptor.h"
#include "InetAddr.h"
#include "SymbolicInetAddr.h"
#include "DnsResolver.h"
#include "IntrusivePtr.h"
#include "AutoClosingSAP.h"
#include "RefCountableSAP.h"
#include "StringUtils.h"
#include "Alarm.h"
#ifdef ENABLE_BINRELOC
#include "binreloc/prefix.h"
#endif
#ifdef ENABLE_PROXYWATCHER
#include "ProxyWatcherThread.h"
#endif
#include <ace/config-lite.h>
#include <ace/Dirent.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/Signal.h>
#include <ace/FILE_IO.h>
#include <ace/FILE_Addr.h>
#include <ace/FILE_Connector.h>
#include <boost/program_options.hpp>
#include <boost/tokenizer.hpp>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <deque>
#include <vector>
#include <list>
#include <cstdlib>
#include <sys/types.h>
#include <sys/file.h>
#include <unistd.h>
#include <netdb.h>
#include <pwd.h>
#include <grp.h>

namespace po = boost::program_options;
using namespace std;

class Daemon::ConfigErrorHandler : public ConfErrorHandler
{
public:
	ConfigErrorHandler(std::string const& fname)
	: m_fileName(fname), m_hasCriticalErrors(false) {}
	
	virtual bool handleError(ConfError const& err);
	
	bool hasCriticalErrors() const { return m_hasCriticalErrors; }
private:
	std::string m_fileName;
	bool m_hasCriticalErrors;
};


class Daemon::ConfigLoader
{
public:
	bool load(std::string const& confdir);
	
	Config const& getConfig() const { return m_config; }
private:
	Config m_config;
};


class Daemon::UrlPatternsLoader
{
public:
	bool load(std::string const& confdir);
	
	CombinedUrlPatterns const& getCombinedPatterns() const {
		return m_combinedPatterns;
	}
private:
	bool load(std::string const& confdir,
		std::string const& fname, UrlPatterns& target);
	
	CombinedUrlPatterns m_combinedPatterns;
};


class Daemon::ContentFiltersLoader
{
public:
	bool load(std::string const& confdir);
	
	ContentFilters const& getFilters() const { return m_filters; }
private:
	void loadEnabledFilterList(
		ContentFilters& filters, std::string const& abs_fname);
	
	void appendEnabledFilters(ContentFilters const& filters);
	
	ContentFilters m_filters;
};


/* ==================== Daemon::ConfigErrorHandler =======================*/

bool
Daemon::ConfigErrorHandler::handleError(ConfError const& err)
{
	cerr << '[' << m_fileName;
	if (err.getLine()) {
		cerr << ':' << err.getLine();
		if (err.getCol()) {
			cerr << ':' << err.getCol();
		}
	}
	cerr << "] ";
	if (err.getType() == ConfError::T_WARNING) {
		cerr << "Warning: ";
	} else {
		m_hasCriticalErrors = true;
	}
	cerr << err.getMessage();
	cerr << endl;
	return true;
}


/* ====================== Daemon::ConfigLoader =========================*/

bool
Daemon::ConfigLoader::load(std::string const& confdir)
{
	string const fname("config");
	string const abs_fname(confdir+"/"+fname);
	ifstream istrm(abs_fname.c_str());
	if (istrm.fail()) {
		cerr << "Could not open " << abs_fname << endl;
		return false;
	}
	ostringstream ostrm;
	ostrm << istrm.rdbuf();
	if (istrm.bad()) {
		cerr << "Error reading " << abs_fname << endl;
		return false;
	}
	
	ConfigFile cfile;
	ConfigErrorHandler eh(fname);
	cfile.load(ostrm.str(), m_config, eh);
	
	if (eh.hasCriticalErrors()) {
		return false;
	}
	
	return true;
}


/* ==================== Daemon::UrlPatternsLoader =======================*/

bool
Daemon::UrlPatternsLoader::load(std::string const& confdir)
{
	if (!load(confdir, "urls", m_combinedPatterns.standardPatterns())) {
		return false;
	}
	if (!load(confdir, "urls.local", m_combinedPatterns.localPatterns())) {
		return false;
	}
	return true;
}

bool
Daemon::UrlPatternsLoader::load(
	std::string const& confdir,
	std::string const& fname, UrlPatterns& target)
{
	string const abs_fname(confdir+"/"+fname);
	ifstream istrm(abs_fname.c_str());
	if (istrm.fail()) {
		cerr << "Could not open " << abs_fname << endl;
		return false;
	}
	ostringstream ostrm;
	ostrm << istrm.rdbuf();
	if (istrm.bad()) {
		cerr << "Error reading " << abs_fname << endl;
		return false;
	}
	
	UrlPatternsFile cfile;
	ConfigErrorHandler eh(fname);
	cfile.load(ostrm.str(), target, eh);
	
	if (eh.hasCriticalErrors()) {
		return false;
	}
	
	return true;
}


/* ====================== Daemon::ContentFilterLoader =========================*/

bool
Daemon::ContentFiltersLoader::load(std::string const& confdir)
{
	string const dirname(confdir+"/filters");
	ACE_Dirent dir;
	if (dir.open(dirname.c_str()) == -1) {
		cerr << "Could not open directory " << dirname << endl;
		return false;
	}
	
	deque<string> fnames;
	
	for (dirent* ent; (ent = dir.read()); ) {
		string const fname(ent->d_name);
		if (fname.find('.') != string::npos) {
			// skip files with extensions
			continue;
		}
		fnames.push_back(fname);
	}
	std::sort(fnames.begin(), fnames.end());
	
	bool critical_errors = false;	
	
	for (; !fnames.empty(); fnames.pop_front()) {
		string const fname = fnames.front();
		string const abs_fname(dirname+"/"+fname);
		ifstream istrm(abs_fname.c_str());
		if (istrm.fail()) {
			cerr << "Could not open " << abs_fname << endl;
			critical_errors = true;
			continue;
		}
		ostringstream ostrm;
		ostrm << istrm.rdbuf();
		if (istrm.bad()) {
			cerr << "Error reading " << abs_fname << endl;
			critical_errors = true;
			continue;
		}
		
		ContentFilters filters;
		ContentFiltersFile cfile;
		ConfigErrorHandler eh(fname);
		cfile.load(ostrm.str(), filters, eh, FilterGroupTag());
		
		if (eh.hasCriticalErrors()) {
			critical_errors = true;
			continue;
		}
		
		loadEnabledFilterList(filters, abs_fname+".enabled");
		appendEnabledFilters(filters);
	}
	
	m_filters.sortByOrder();
	return !critical_errors;
}

void
Daemon::ContentFiltersLoader::loadEnabledFilterList(
	ContentFilters& filters, std::string const& abs_fname)
{
	ifstream istrm(abs_fname.c_str());
	if (istrm.fail()) {
		return;
	}
	ostringstream ostrm;
	ostrm << istrm.rdbuf();
	if (istrm.bad()) {
		return;
	}
	
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("\r\n");
	Tokenizer tokens(ostrm.str(), sep);
	set<string> enabled_set(tokens.begin(), tokens.end());
	bool const all_enabled = (enabled_set.find("*") != enabled_set.end());
	
	typedef ContentFilters::FilterList FilterList;
	FilterList& list = filters.filters();
	FilterList::iterator it = list.begin();
	FilterList::iterator const end = list.end();
	for (; it != end; ++it) {
		RegexFilterDescriptor& filter = **it;
		bool const enabled = all_enabled ||
			(enabled_set.find(filter.name()) != enabled_set.end());
		filter.setEnabled(enabled);
	}
}

void
Daemon::ContentFiltersLoader::appendEnabledFilters(ContentFilters const& filters)
{
	typedef ContentFilters::FilterList FilterList;
	FilterList const& list = filters.filters();
	FilterList::const_iterator it = list.begin();
	FilterList::const_iterator const end = list.end();
	for (; it != end; ++it) {
		if ((*it)->isEnabled()) {
			m_filters.filters().push_back(*it);
		}
	}
}


/*=========================== Daemon ==================================*/

volatile bool Daemon::m_sSigtermReceived = false;
volatile bool Daemon::m_sSigintReceived = false;

bool
Daemon::init(std::string const& confdir)
{
	if (!loadConfiguration(confdir)) {
		return false;
	}
	
	if (!bind()) {
		return false;
	}
	return true;
}

bool
Daemon::run(bool nodaemon, ACE_FILE_IO& pid_file)
{
	if (!nodaemon) {
		if (ACE::daemonize("/", false) == -1) {
			cerr << "Could not enter the daemon mode" << endl;
			return false;
		}
	}
	
	{
		ACE_Sig_Action sigpipe_action(SIG_IGN, SIGPIPE);
		ACE_Sig_Action sigterm_action(&Daemon::sigtermHandler, SIGTERM);
		ACE_Sig_Action sigint_action(&Daemon::sigintHandler, SIGINT);
	}
	
	if (pid_file.get_handle() != ACE_INVALID_HANDLE) {
		ostringstream strm;
		strm << getpid() << '\n';
		string pid = strm.str();
		pid_file.truncate(0);
		pid_file.send_n(pid.c_str(), pid.size());
	}
	
#ifdef ENABLE_PROXYWATCHER
	ProxyWatcherThread proxy_wather;
#endif
	
	while (!m_sSigtermReceived && !m_sSigintReceived) {
		m_acceptor.acceptConnections();
	}
	
	if (pid_file.get_handle() != ACE_INVALID_HANDLE) {
		// It would be better to delete the pid file,
		// but we've already chrooted and dropped privileges.
		pid_file.truncate(0);
	}
	
	return true;
}

int
Daemon::killRunning(ACE_FILE_IO& pid_file)
{
	if (flock(pid_file.get_handle(), LOCK_EX|LOCK_NB) == 0) {
		std::cerr << "No instance is running" << std::endl;
		return EXIT_SUCCESS;
	}
	
	char buf[20];
	ssize_t size = pid_file.recv(buf, sizeof(buf));
	if (size == -1) {
		std::cerr << "Error reading pid file" << std::endl;
		return EXIT_FAILURE;
	}
	
	stringstream strm;
	strm.write(buf, size);
	pid_t pid = 0;
	strm >> pid;
	if (pid <= 0) {
		std::cerr << "Pid file contains an invalid process id" << std::endl;
		return EXIT_FAILURE;
	}
	if (kill(pid, 0) == -1) {
		switch (errno) {
		case EPERM:
			std::cerr << "No permission to signal the daemon process" << std::endl;
			break;
		case ESRCH:
			std::cerr << "Process referenced by pid not found" << std::endl;
			break;
		default:
			std::cerr << "Signalling the daemon process failed" << std::endl;
		}
		return EXIT_FAILURE;
	}
	
	kill(pid, SIGTERM);
	
	Alarm alrm;
	alrm(1);
	
	if (flock(pid_file.get_handle(), LOCK_EX) == 0) {
		return EXIT_SUCCESS;
	}
	
	kill(pid, SIGKILL);
	alrm(1);
	
	if (flock(pid_file.get_handle(), LOCK_EX) == 0) {
		pid_file.truncate(0);
		return EXIT_SUCCESS;
	}
	
	std::cerr << "Could not lock the pid file even after SIGKILL" << std::endl;
	return EXIT_FAILURE;
}

void
Daemon::initResolver()
{
	/*
	There is a problem with name resolution in a chroot'ed environment.
	On many Unix systems, the resolver will dlopen() some shared libraries
	and some config files before doing the actual name resolution.
	The bad news is that at least some of the files (/etc/hosts and
	/etc/resolv.conf in my case) are read on every name resolution,
	so we just have to have them in our chroot'ed environment.
	The good news is that shared libraries are only loaded on the first
	name resolution, which we do here.
	Note that resolving "localhost" doesn't cause loading of all the
	shared libraries needed by the resolver.
	*/
#if defined(__APPLE__)
	// Not necessary on OSX.
#else
	Alarm alrm;
	alrm(1); // Don't let it block for too long.
	gethostbyname("com.");
#endif
}

std::string
Daemon::normalizePath(std::string const& path)
{
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("/");
	Tokenizer tokenizer(path, sep);
	
	vector<string> elements;
	
	Tokenizer::iterator it = tokenizer.begin();
	Tokenizer::iterator const end = tokenizer.end();
	for (; it != end; ++it) {
		if (*it == "..") {
			if (elements.empty() || elements.back() == "..") {
				elements.push_back(*it);
			} else {
				elements.pop_back();
			}
		} else if (*it != ".") {
			elements.push_back(*it);
		}
	}
	
	string res;
	res.reserve(path.size());
	if (path[0] == '/') {
		res += '/';
	}
	res += StringUtils::join("/", elements.begin(), elements.end());
	
	return res;
}

bool
Daemon::loadConfiguration(std::string const& confdir)
{
	ConfigLoader config_loader;
	if (!config_loader.load(confdir)) {
		return false;
	}
	
	UrlPatternsLoader patterns_loader;
	if (!patterns_loader.load(confdir)) {
		return false;
	}
	
	ContentFiltersLoader cf_loader;
	if (!cf_loader.load(confdir)) {
		return false;
	}
	
	GlobalState::WriteAccessor global_state;
	global_state->config() = config_loader.getConfig();
	global_state->urlPatterns() = patterns_loader.getCombinedPatterns();
	global_state->contentFilters() = cf_loader.getFilters();
	
	return true;
}

bool
Daemon::bind()
{
	typedef RefCountableSAP<ACE_SOCK_Acceptor> Acceptor;
	typedef IntrusivePtr<Acceptor> AcceptorPtr;
	
	bool ok = true;
	deque<AcceptorPtr> acceptors;
	list<SymbolicInetAddr> addrs = GlobalState::ReadAccessor()->config().getListenAddresses();
	for (; !addrs.empty(); addrs.pop_front()) {
		SymbolicInetAddr const& addr = addrs.front();
		vector<InetAddr> resolved_addrs = DnsResolver::resolve(addr);
		if (resolved_addrs.empty()) {
			ok = false;
			cerr << "Could not resolve listen address \"" << addr << '"' << endl;
			continue;
		}
		AcceptorPtr acceptor(new Acceptor);
		if (acceptor->open(resolved_addrs[0], true) == -1) {
			ok = false;
			cerr << "Could not bind to \"" << addr << '"' << endl;
			continue;
		}
		acceptors.push_back(acceptor);
	}
	if (!ok) {
		return false;
	}
	if (acceptors.empty()) {
		cerr << "No addresses to listen on!" << endl;
		return false;
	}
	
	for (; !acceptors.empty(); acceptors.pop_front()) {
		m_acceptor.add(acceptors.front());
	}
	
	return true;
}

void
Daemon::sigtermHandler(int)
{
	m_sSigtermReceived = true;
}

void
Daemon::sigintHandler(int)
{
	m_sSigintReceived = true;
}


/* =============================== main ================================*/

// ACE #define's main() to include its own initialization
int main(int argc, char **argv)
{
	bool nodaemon = false;
	bool kill_running = false;
	string confdir;
	string chroot_dir;
	string effective_confdir; // relative to chroot_dir
#if defined(SYSCONFDIR)
	// binreloc may redefine SYSCONFDIR
	confdir = string(SYSCONFDIR) + "/bfilter";
#elif defined(__APPLE__)
	confdir = "/Library/Application Support/BFilter";
#endif
	string user;
	string group;
	string pidfile;
	try {
		po::options_description desc("Allowed options");
		desc.add_options()
			("help,h", "Print help message")
			("version,v", "Print version")
			("confdir,c", po::value<string>(&confdir), "Set custom config directory")
			("chroot,r", po::value<string>(&chroot_dir),
				"Set chroot directory. It must contain the config "
				"directory and may be specified relative to it."
			)
			("user,u", po::value<string>(&user), "Set unprivileged user")
			("group,g", po::value<string>(&group), "Set unprivileged group")
			("nodaemon,n", po::bool_switch(&nodaemon), "Disable background daemon mode")
			("pid,p", po::value<string>(&pidfile),
				"[without -k] Write process id to file\n"
				"[with -k]    Kill the running instance")
			("kill,k", po::bool_switch(&kill_running),
				"Kill the running instance. To be used with -p")
		;
		po::variables_map vm;
		po::store(po::parse_command_line(argc, argv, desc), vm);
		po::notify(vm);
		if (vm.count("help")) {
			std::cout << desc << std::endl;
			return EXIT_SUCCESS;
		}
		if (vm.count("version")) {
			std::cout << BFILTER_VERSION << std::endl;
			return EXIT_SUCCESS;
		}
		if (kill_running && pidfile.empty()) {
			std::cerr << "--kill requires --pid" << std::endl;
			return EXIT_FAILURE;
		}
	} catch (std::exception& e) {
		std::cerr << e.what() << std::endl;
		return EXIT_FAILURE;
	}
	
	Alarm::installNoOpHandler();
	
	AutoClosingSAP<ACE_FILE_IO> pid_file;
	if (!pidfile.empty()) {
		ACE_FILE_Addr addr(pidfile.c_str());
		ACE_FILE_Connector connector;
		if (connector.connect(pid_file, addr, 0, ACE_Addr::sap_any, 0, O_RDWR|O_CREAT) == -1) {
			std::cerr << "Could not open " << pidfile << " for writing" << std::endl;
			return EXIT_FAILURE;
		}
		if (kill_running) {
			return Daemon::killRunning(pid_file);
		}
		if (flock(pid_file.get_handle(), LOCK_EX|LOCK_NB) == -1) {
			std::cerr << "Another instance is running" << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	if (confdir[0] != '/') {
		std::cerr << "Config directory must be absolute" << std::endl;
		return EXIT_FAILURE;
	}
	
	confdir = Daemon::normalizePath(confdir);
	effective_confdir = confdir;
	
	if (!chroot_dir.empty()) {
		if (chroot_dir[0] == '/') {
			chroot_dir = Daemon::normalizePath(chroot_dir);
		} else {
			chroot_dir = Daemon::normalizePath(confdir+'/'+chroot_dir);
		}
		
		if (StringUtils::startsWith(confdir, chroot_dir)) {
			effective_confdir = confdir.substr(chroot_dir.size());
			// normalizePath() removes the trailing slash among other
			// things, so effective_confdir may now we empty, which is OK.
		} else {
			std::cerr << "Config directory must be within the chroot directory" << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	Daemon::initResolver();
	
	struct passwd* user_r = 0;
	struct group* group_r = 0;
	
	if (!user.empty()) {
		user_r = getpwnam(user.c_str());
		if (!user_r) {
			std::cerr << "Could not drop privileges: user "
				<< user << " doesn't exist" << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	if (!group.empty()) {
		group_r = getgrnam(group.c_str());
		if (!group_r) {
			std::cerr << "Could not drop privileges: group "
				<< group << " doesn't exist" << std::endl;
		}
	}
	
	if (!chroot_dir.empty()) {
		if (chdir(chroot_dir.c_str())) {
			std::cerr << "Could not chdir to " << chroot_dir << std::endl;
			return EXIT_FAILURE;
		}
		if (chroot(chroot_dir.c_str())) {
			std::cerr << "Could not chroot to " << chroot_dir << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	if (group_r) {
		if (setgid(group_r->gr_gid)) {
			std::cerr << "Could not drop priviliges: setgid() failed" << std::endl;
			return EXIT_FAILURE;
		}
	}
	if (user_r) {
		if (setuid(user_r->pw_uid)) {
			std::cerr << "Could not drop privileges: setuid() failed" << std::endl;
			return EXIT_FAILURE;
		}
	}
	
	
	Daemon daemon;
	if (!daemon.init(effective_confdir)) {
		return EXIT_FAILURE;
	}
	if (!daemon.run(nodaemon, pid_file)) {
		return EXIT_FAILURE;
	}
	
	return EXIT_SUCCESS;
}
