/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef CONFIO_H_
#define CONFIO_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ConfLexerCore.h"
#include "BString.h"
#include <iosfwd>
#include <string>
#include <list>
#include <stddef.h>

class ConfError;

class ConfIO : private ConfLexerCore
{
	friend class ConfLexerCore::Actions;
public:
	enum Flags { DISABLE_HEREDOC = 1 };
	
	class Element
	{
	public:
		enum Type { BLANK_LINE, COMMENT, SECTION, KEY_VALUE };
		
		Element(); // blank line
		
		Element(std::string const& section);
		
		Element(char comment_symbol, std::string const& comment);
		
		Element(std::string const& key, std::string const& value);
		
		~Element();
		
		Type getType() const { return m_type; }
		
		std::string const& getText() const { return m_value; }
		
		std::string const& getKey() const { return m_key; }
		
		std::string const& getValue() const { return m_value; }
		
		void setValue(std::string const& value) { m_value = value; }
		
		void toStream(std::ostream& strm, int flags = 0) const;
		
		bool operator==(Element const& other) const;
		
		bool operator!=(Element const& other) const { return !(*this == other); }
	private:
		Type m_type;
		char m_commentSymbol;
		std::string m_key;
		std::string m_value; // or comment
	};
	
	typedef std::list<Element>::iterator ElementIterator;
	
	ConfIO();
	
	virtual ~ConfIO();
	
	bool read(std::istream& strm);
	// returns false if reading was aborted
	bool read(std::string const& text);
	
	static void write(
		std::ostream& strm,
		std::list<Element> const& elements, int flags = 0);
	
	static ElementIterator findNextSection(
		ElementIterator const& begin, ElementIterator const& end);
	
	static ElementIterator getSectionItemsEnd(
		ElementIterator const& section_element,
		ElementIterator const& end);
	
	static ElementIterator distributeComments(
		ElementIterator const& begin, ElementIterator const& end);
	
	static ElementIterator getParamAreaBegin(
		ElementIterator const& begin, ElementIterator const& param);
	
	static ElementIterator getParamAreaEnd(
		ElementIterator const& param, ElementIterator const& end);
private:
	virtual bool processElement(Element const& element, int line) = 0;
	
	virtual bool processError(ConfError const& error) = 0;
	
	void newLine(Iterator const& pos);
	
	void setCommentSymbol(char symbol) { m_commentSymbol = symbol; }
	
	void processBlankLine();
	
	void processBrokenSection();
	
	void processUnexpectedCharacter(Iterator const& pos);
	
	void processComment(Iterator const& begin, Iterator const& end);
	
	void processSection(Iterator const& begin, Iterator const& end);
	
	void processKey(Iterator const& begin, Iterator const& end);
	
	void processValue();
	
	void processUnclosedQuote();
	
	void saveHeredocIdentifier(Iterator const& begin, Iterator const& end);
	
	bool isHeredocIdentifier(Iterator const& begin, Iterator const& end) const;
	
	bool isAtLineStart(Iterator const& pos);
	
	void valueAppend(Iterator const& begin, Iterator const& end);
	
	void valueAppendEscaped(char ch);
	
	static void writeValue(std::ostream& strm, std::string const& value, int flags = 0);
	
	static void writeHeredocValue(std::ostream& strm, std::string const& value);
	
	static void writeQuotedValue(std::ostream& strm, std::string const& value);
	
	static std::string generateHeredocIdentifier(std::string const& value);
	
	static void mutateHeredocIdentifier(std::string& id);
	
	static char escapeChar(char ch);
	
	static char unescapeChar(char ch);
	
	char m_commentSymbol;
	std::string m_key;
	std::string m_value;
	BString m_heredocIdentifier;
	int m_lineNumber;
	size_t m_lineStartPos;
};

inline std::ostream&
operator<<(std::ostream& strm, ConfIO::Element const& el)
{
	el.toStream(strm);
	return strm;
}

#endif
