/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ImageResponseFilter.h"
#include "AbstractImageInspector.h"
#include "SplittableBuffer.h"
#include "GlobalState.h"
#include "HttpStatusLine.h"
#include "HttpResponseMetadata.h"
#include "HttpHeadersCollection.h"
#include "HeuristicScore.h"
#include "CraftedResponse.h"
#include "ReplacementImage.h"
#include "DownloadProgress.h"
#include "ErrorDescriptor.h"
#include "ErrorCodes.h"
#include "IntrusivePtr.h"
#include "RequestTag.h"
#include <stddef.h>

using namespace std;

ImageResponseFilter::ImageResponseFilter(
	ResponseFilterChain& chain, bool ignore_size,
 	std::auto_ptr<AbstractImageInspector> inspector)
:	ResponseFilterBase(chain),
	m_state(INSPECTING),
	m_ignoreSize(ignore_size),
	m_ptrInspector(inspector)
{
}

ImageResponseFilter::~ImageResponseFilter()
{
}

void
ImageResponseFilter::processMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	m_ptrMetadata = metadata;
}

void
ImageResponseFilter::processBodyData(SplittableBuffer& data, bool eof)
{
	switch (m_state) {
		case FORWARDING: {
			outputBodyData(data, eof);
			break;
		}
		case SUBSTITUTING: {
			data.clear();
			break;
		}
		case INSPECTING: {
			m_ptrInspector->consumeDataChunk(data, eof);
			switch (m_ptrInspector->getStatus()) {
				case AbstractImageInspector::COMPLETE: {
					if (isAd()) {
						m_state = SUBSTITUTING;
 						outputAdSubstitution();
						considerRequestCancel();
						break;
					}
					// fall through
				}
				case AbstractImageInspector::FAILED: {
					m_state = FORWARDING;
					outputMetadata(m_ptrMetadata);
					outputBodyData(m_ptrInspector->accumulatedData(), eof);
					break;
				}
				default: break;
			}
			break;
		}
	}
}

bool
ImageResponseFilter::isAd()
{
	return m_ptrInspector->isAnimated() || (
		!m_ignoreSize &&
		HeuristicScore::isCommonAdSize(
			m_ptrInspector->getWidth(),
			m_ptrInspector->getHeight()
		)
	);
}

void
ImageResponseFilter::outputAdSubstitution()
{
	auto_ptr<CraftedResponse> response = ReplacementImage::createHttpResponse(
		false, m_ptrInspector->getWidth(), m_ptrInspector->getHeight(),
		HttpStatusLine::SC_OK
	);
	
	getRequestTag()->flags().set(RequestTag::RESPONSE_CRAFTED);
	response->metadata().setClientRequestId(m_ptrMetadata->getClientRequestId());
	outputMetadata(response->retrieveMetadata());
	outputBodyData(response->body(), true);
}

void
ImageResponseFilter::considerRequestCancel()
{
	DownloadProgress const& dp = getFilterChain().getDownloadProgress();
	size_t threshold = GlobalState::ReadAccessor()->config().getSaveTrafficThreshold();
	if (dp.total() - dp.received() > threshold*1024) { // or if dp.total() is undefined
		raiseError(IntrusivePtr<ErrorDescriptor>(
			new ErrorDescriptor(
				ErrorCodes::REQUEST_CANCEL_TO_SAVE_TRAFFIC,
				"request cancelled to save traffic"
			)
		));
	}
}
