/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpStateRequestHeaders.h"
#include "HttpRequestParser.h"
#include "HttpRequestMetadata.h"
#include "HttpRequestLine.h"
#include "HttpHeader.h"
#include "HttpHeadersCollection.h"
#include "HttpHeaderElement.h"
#include "HttpHeaderStructure.h"
#include "HttpMethodRegistry.h"
#include "HttpVersion.h"
#include "BString.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "StringUtils.h"
#include "InsensitiveEqual.h"
#include "ErrorDescriptor.h"
#include "ErrorResponse.h"
#include "ErrorCodes.h"
#include <memory>
#include <string>

using namespace std;

HttpStateRequestHeaders::HttpStateRequestHeaders(HttpRequestParser& parser)
:	HttpStateHeaders(parser),
	m_rParser(parser)
{
}

HttpStateRequestHeaders::~HttpStateRequestHeaders()
{
}

void
HttpStateRequestHeaders::activate(HttpRequestLine const& request_line)
{
	m_ptrMetadata.reset(new HttpRequestMetadata(request_line));
}

void
HttpStateRequestHeaders::cleanup()
{
	reset();
	m_ptrMetadata.reset(0);
}

void
HttpStateRequestHeaders::addHeader(BString const& name, BString const& value)
{
	if (isSingularHeader(name)) {
		/*
		Believe it or not, but some pages output two Content-Length or
		Content-Type headers, and manage to get away with that.
		We only leave the last one of them, as otherwise they would
		cause parse errors.
		*/
		m_ptrMetadata->headers().setHeader(name, value);
	} else {
		m_ptrMetadata->headers().addHeader(name, value);
	}
}

HttpState*
HttpStateRequestHeaders::headersReceived(bool& body_eof)
{
	BString const& method = m_ptrMetadata->requestLine().getMethod();
	if (!HttpMethodRegistry::find(method)) {
		IntrusivePtr<ErrorDescriptor> edesc(new ErrorDescriptor(
			ErrorCodes::UNSUPPORTED_METHOD,
			"unsupported method: " + method.toStdString(),
			sigc::bind(
				sigc::ptr_fun(&ErrorResponse::errUnsupportedMethod),
				m_ptrMetadata->requestLine().getURI().toString(),
				method.toStdString()
			)
		));
		return m_rParser.activateStateError(edesc);
	}
	
	bool const is_persistent = isPersistentConnection(
		m_ptrMetadata->headers(),
		m_ptrMetadata->requestLine().getHttpVersion()
	);
	
	BString const chunked("chunked");
	BString const identity("identity");
	InsensitiveEqual const icomp;
	
	HttpHeader* tenc = m_ptrMetadata->headers().getHeaderPtr(BString("Transfer-Encoding"));
	if (tenc && !icomp(tenc->getValue(), identity)) {
		HttpHeaderStructure structure(*tenc);
		HttpHeaderElement const* last = structure.getLastElement();
		if (last && icomp(last->getName(), chunked)) {
			structure.elements().pop_back();
			if (structure.empty()) {
				m_ptrMetadata->headers().removeHeader(BString("Transfer-Encoding"));
			} else {
				structure.commitChanges(*tenc);
			}
			m_ptrMetadata->setBodyStatus(HttpRequestMetadata::BODY_SIZE_UNKNOWN);
			m_rParser.eventGotMetadata(m_ptrMetadata, is_persistent);
			return m_rParser.activateStateChunkHeader();
		}
	}
	
	HttpHeader* clen = m_ptrMetadata->headers().getHeaderPtr(BString("Content-Length"));
	if (clen) {
		BString const cl(clen->getValue());
		char const* cl_end = cl.end();
		uintmax_t body_size = StringUtils::parseUnsigned<uintmax_t>(cl.begin(), cl_end);
		if (cl_end != cl.end()) {
			// the header's value is already trimmed, so the problem is elsewhere
			return m_rParser.activateStateError("Content-Length header is broken");
		}
		m_ptrMetadata->setBodyStatus(HttpRequestMetadata::BODY_SIZE_KNOWN);
		m_ptrMetadata->setBodySize(body_size);
		if (body_size != 0) {
			m_rParser.eventGotMetadata(m_ptrMetadata, is_persistent);
			return m_rParser.activateStateSizedFlatBody(body_size);
		} else {
			m_rParser.eventGotMetadata(m_ptrMetadata, is_persistent);
			body_eof = true;
			return m_rParser.activateStateInactive();
		}
	}
	
	m_ptrMetadata->setBodyStatus(HttpRequestMetadata::BODY_FORBIDDEN);
	m_rParser.eventGotMetadata(m_ptrMetadata, is_persistent);
	body_eof = true;
	return m_rParser.activateStateInactive();
}
