/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DecompressorResponseFilter.h"
#include "GzipDecompressor.h"
#include "DeflateDecompressor.h"
#include "ResponseFilterChain.h"
#include "ResponseErrorInitiator.h"
#include "SplittableBuffer.h"
#include "BString.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestMetadata.h"
#include "HttpStatusLine.h"
#include "HttpHeader.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "ErrorResponse.h"
#include "ErrorDescriptor.h"
#include "ErrorCodes.h"
#include "StringUtils.h"
#include "IntrusivePtr.h"
#include <memory>
#include <cstring>

using namespace std;

DecompressorResponseFilter::DecompressorDescriptor const
DecompressorResponseFilter::m_sAvailableDecompressors[] = {
	{ "gzip", &DecompressorResponseFilter::createGzipDecompressor },
	{ "deflate", &DecompressorResponseFilter::createDeflateDecompressor }
};

DecompressorResponseFilter::DecompressorResponseFilter(ResponseFilterChain& chain)
:	ResponseFilterBase(chain),
	m_isEOF(false)
{
}

DecompressorResponseFilter::~DecompressorResponseFilter()
{
}

void
DecompressorResponseFilter::processMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	if (HttpHeader* teh = metadata->headers().getHeaderPtr(BString("Transfer-Encoding"))) {
		setupDecompressor(*metadata, teh);
	} else if (HttpHeader* ceh = metadata->headers().getHeaderPtr(BString("Content-Encoding"))) {
		setupDecompressor(*metadata, ceh);
	}
	if (m_ptrDecompressor.get()) {
		metadata->setBodyStatus(HttpResponseMetadata::BODY_SIZE_UNKNOWN);
	}
	outputMetadata(metadata);
}

void
DecompressorResponseFilter::processBodyData(SplittableBuffer& data, bool eof)
{
	if (!m_ptrDecompressor.get()) {
		outputBodyData(data, eof);
		return;
	}
	
	m_isEOF |= eof;
	m_ptrDecompressor->consume(data, m_isEOF);
	SplittableBuffer output;
	size_t written = 0;
	
	do {
		written = m_ptrDecompressor->retrieveOutput(output, OUTPUT_LIMIT);
		if (written != 0) {
			outputBodyData(output, false);
		}
		if (m_ptrDecompressor->isError() && !isErrorRaised()) {
			string problem("error decompressing response");
			raiseError(IntrusivePtr<ErrorDescriptor>(
				new ErrorDescriptor(
					ErrorCodes::ERROR_DECOMPRESSING_RESPONSE, problem,
					sigc::bind(
						sigc::ptr_fun(&ErrorResponse::errGenericError),
						getRequest().requestLine().getURI().toString(),
						problem, int(HttpStatusLine::SC_INTERNAL_SERVER_ERROR)
					)
				)
			));
			return;
		}
	} while (written != 0);
	
	if (m_isEOF) {
		outputBodyData(output, m_isEOF);
	}
}

bool
DecompressorResponseFilter::isSupported(BString const& compression)
{
	return findDescriptor(compression);
}

DecompressorResponseFilter::DecompressorDescriptor const*
DecompressorResponseFilter::findDescriptor(BString const& name)
{
	static DecompressorDescriptor const* const end = m_sAvailableDecompressors +
		sizeof(m_sAvailableDecompressors)/sizeof(m_sAvailableDecompressors[0]);
	DecompressorDescriptor const* desc = m_sAvailableDecompressors;
	for (; desc != end; ++desc) {
		char const* begin = desc->name;
		char const* end = begin + strlen(begin);
		if (StringUtils::ciEqual(begin, end, name.begin(), name.end())) {
			return desc;
		}
	}
	return 0;
}

AbstractDecompressor*
DecompressorResponseFilter::createGzipDecompressor()
{
	return new GzipDecompressor;
}

AbstractDecompressor*
DecompressorResponseFilter::createDeflateDecompressor()
{
	return new DeflateDecompressor(DeflateDecompressor::AUTODETECT);
}

void
DecompressorResponseFilter::setupDecompressor(HttpResponseMetadata& metadata, HttpHeader* hdr)
{
	HttpHeaderStructure structure(*hdr);
	HttpHeaderElement const* last = structure.getLastElement();
	DecompressorDescriptor const* desc = 0;
	if (last) {
		desc = findDescriptor(last->getName());
	}
	if (desc) {
		m_ptrDecompressor.reset(desc->constructor());
		structure.elements().pop_back();
		if (structure.empty()) {
			metadata.headers().removeHeader(hdr->getName());
		} else {
			structure.commitChanges(*hdr);
		}
	}
}
