// MessageStack.cpp: implementation of the MessageStack class. 
// 
////////////////////////////////////////////////////////////////////// 
         
#include "../../Include/Comm/MessageStack.h" 
#include "../../Include/Comm/Message.h" 
#include <stddef.h> 
         
 
////////////////////////////////////////////////////////////////////// 
// Construction/Destruction 
////////////////////////////////////////////////////////////////////// 
         
	CMessageStack::CMessageStack() : CObject ("MessageStack") 
        { 
	      	m_StackTop = NULL; 
      		m_StackBottom = NULL; 
        	m_ulMessageCount = 0; 
 
		PASS_INITLOCK (&lock); 
        } 
         
        CMessageStack::~CMessageStack() 
        { 
		PASS_DESTROYLOCK (&lock); 
	 
        	// Clean up the stack 
        	__MessageStackElement *temp, *temp2; 
        	temp = m_StackTop; 
         
        	while (temp != NULL) 
        	{ 
        		if (temp->message != NULL) 
        		{ 
        			temp->message->Release(); 
        		} 
        		temp2 = temp; 
        		temp = temp->next; 
        		delete temp2; 
        	} 
        } 
         
        void CMessageStack::PutMessage (CMessage *aMessage) 
        { 
			PASS_LOCK (&lock); 
 
			if (aMessage == NULL) 
			{ 
				PASS_UNLOCK (&lock); 
				return; 
			} 
 
        	// New message to stack 
         
        	__MessageStackElement *tempStack = new __MessageStackElement; 
        	tempStack->message = aMessage; 
         
        	if (m_StackTop == NULL) 
        	{ 
	     		tempStack->next = NULL; 
        		m_StackBottom = tempStack; 
        	} 
        	else 
        	{ 
        		tempStack->next = m_StackTop; 
        	} 
         
        	// New head 
         
        	m_StackTop = tempStack; 
         
        	m_ulMessageCount++; 
 
			PASS_UNLOCK (&lock); 
        } 
         
        BOOL CMessageStack::IsMessageAvailable (void) 
        { 
			PASS_LOCK (&lock); 
         
        	if (m_StackTop == NULL) 
        	{ 
				PASS_UNLOCK (&lock); 
        		return FALSE; 
        	} 
 
			PASS_UNLOCK (&lock); 
        	return TRUE; 
        } 
         
        CMessage *CMessageStack::GetMessage (void) 
        { 
			PASS_LOCK (&lock); 
  
	     	CMessage *returnMessage = NULL; 
#ifdef PASS_EXCEPTION          
        	try 
#endif // PASS_EXCEPTION 
        	{ 
         
        	// No message 
        	if (m_StackBottom == NULL || m_StackTop == NULL) 
        	{ 
				PASS_UNLOCK (&lock); 
     			return NULL; 
        	} 
         
        	// Last message 
        	if (m_StackTop == m_StackBottom) 
        	{ 
        		CMessage *temp = m_StackBottom->message; 
        		delete m_StackBottom; 
	     		m_StackTop = NULL; 
     			m_StackBottom = NULL; 
        		m_ulMessageCount--; 
 
				PASS_UNLOCK (&lock); 
        		return temp; 
        	} 
         
        	// Two message left 
         
        	if (m_StackTop->next == m_StackBottom) 
        	{ 
        		CMessage *temp = m_StackBottom->message; 
        		delete m_StackBottom; 
	     		m_StackBottom = NULL; 
        		m_StackBottom = m_StackTop; 
	     		m_StackTop->next = NULL; 
        		m_ulMessageCount--; 
 
				PASS_UNLOCK (&lock); 
        		return temp; 
         
        	} 
         
        	// Remove a message at the bottom of the stack 
         
	     	__MessageStackElement *temp = NULL; 
        	temp = m_StackTop; 
         
        	while (temp->next != m_StackBottom) 
        	{ 
        		temp = temp->next; 
        	} 
         
        	// Found the last - 1 message 
         
        	returnMessage = m_StackBottom->message; 
        	delete m_StackBottom; 
	     	m_StackBottom = NULL; 
         
	     	temp->next = NULL; 
        	m_StackBottom = temp; 
         
        	m_ulMessageCount--; 
        	}
#ifdef PASS_EXCEPTION 			
        	catch (...) 
        	{ 
				PASS_UNLOCK (&lock); 
	     		return NULL; 
        	} 
#endif // PASS_EXCEPTION          
			PASS_UNLOCK (&lock); 
        	return returnMessage; 
        } 
         
        CMessage *CMessageStack::PeekMessage (void) 
        { 
			PASS_LOCK (&lock); 
			 
        	// No message 
        	if (m_StackBottom == NULL || m_StackTop == NULL) 
        	{ 
				PASS_UNLOCK (&lock); 
	    		return NULL; 
        	} 
         
        	// Found the last - 1 message 
         
        	CMessage *tempMessage = m_StackBottom->message; 
         
			PASS_UNLOCK (&lock); 
        	return tempMessage; 
        } 
         
        CMessage *CMessageStack::GetMessageForID (unsigned long messageID) 
        { 
			PASS_LOCK (&lock); 
         
#ifdef PASS_EXCEPTION 
        	try  
#endif // PASS_EXCEPTION 
        	{ 
	     		__MessageStackElement *temp = NULL; 
        		temp = m_StackTop; 
	     		__MessageStackElement *previous = NULL; 
         
        		BOOL found = FALSE; 
         
        		while (temp != NULL && (unsigned int)temp != 0xdddddddd && found == FALSE) 
        		{ 
        			CMessage *localMessage = temp->message; 
         
        			if (localMessage != NULL && (unsigned int)localMessage != 0xdddddddd) 
        			{ 
        				if (localMessage->GetMessageID() == messageID)  
        				{ 
        					found = TRUE; 
        				} 
        				else 
        				{ 
        					previous = temp; 
        					temp = temp->next; 
        				} 
        			} 
        			else 
        			{ 
        				previous = temp; 
        				temp = temp->next; 
        			} 
        		} 
        		if (found == TRUE) 
        		{ 
	     			CMessage *returnMessage = NULL; 
         
        			// Found the message, now remove it from the list; 
        			if (previous == NULL && temp->next == NULL) 
        			{ 
        				// This was the only one in the list 
 	     				m_StackTop = NULL; 
 	     				m_StackBottom = NULL; 
        				returnMessage = temp->message; 
        				delete temp; 
 	     				temp = NULL; 
        			}  
        			else if (previous == NULL && temp->next != NULL) 
        			{ 
        				// This is the first of the list 
        				m_StackTop = temp->next; 
        				returnMessage = temp->message; 
        				delete temp; 
 	     				temp = NULL; 
        			} 
        			else if (previous != NULL && temp->next == NULL) 
        			{ 
        				// This is the last one. 
 	     				previous->next = NULL; 
        				m_StackBottom = previous; 
        				returnMessage = temp->message; 
        				delete temp; 
 	     				temp = NULL; 
        			} 
        			else if (previous != NULL && temp->next != NULL) 
        			{ 
        				// Somewhere in the list 
        				previous->next = temp->next; 
        				returnMessage = temp->message; 
        				delete temp; 
 	     				temp = NULL; 
        			} 
         
	   			m_ulMessageCount--; 
    
					PASS_UNLOCK (&lock); 
        			return returnMessage; 
        		} 
         
        	} 
#ifdef PASS_EXCEPTION 
        	catch (...) 
        	{ 
				PASS_UNLOCK (&lock); 
	     		return NULL; 
        	} 
#endif // PASS_EXCEPTION 
         
			PASS_UNLOCK (&lock); 
	     	return NULL; 
     }
 
