// ConnectionManager.cpp: implementation of the CConnectionManager class. 
// 
////////////////////////////////////////////////////////////////////// 
 
#ifdef linux 
#include <stddef.h> 
#include <string.h> 
#include <unistd.h> 
#include <stdio.h> 
#endif //linux 
 
#ifdef __SCO__ 
#include <stddef.h> 
#include <string.h> 
#include <unistd.h> 
#include <stdio.h> 
#endif // __SCO__ 
 
#ifdef __HPUX__ 
#include <stddef.h> 
#include <string.h> 
#include <unistd.h> 
#include <stdio.h> 
#endif // __HPUX__ 
 
#include "../../Include/Comm/ConnectionManager.h" 
#include "../../Include/Base/LongToObjectCollection.h" 
#include "../../Include/Base/StringToObjectCollection.h" 
#include "../../Include/Comm/Connection.h" 
#include "../../Include/Comm/Connection_Dual.h" 
#include "../../Include/Comm/Connection_Mono.h" 
#include "../../Include/Comm/Connection_Local.h" 
#include "../../Include/Comm/ConnectionListener.h" 
#include "../../Include/Base/ObjectManager.h" 
#include "../../Include/Base/LoggerManager.h" 
#include "../../Include/Comm/ConnectionMessage_Close.h" 
#include "../../Include/Comm/ConnectionMessage_OpenBack.h" 
#include "../../Include/Comm/GenericMessage.h" 
#include "../../Include/Comm/DispatchMessage.h" 
#include "../../Include/Comm/ServerSessionDispatcher.h" 
#include "../../Include/Comm/ServerSessionDefault.h" 
 
#ifdef PASS_SSL 
#include <openssl/ssl.h> 
#include <openssl/rand.h> 
#endif //PASS_SSL 
 
////////////////////////////////////////////////////////////////////// 
// Construction/Destruction 
////////////////////////////////////////////////////////////////////// 
         
        CConnectionManager::CConnectionManager() : CManager("Connection") 
        { 
		PASS_INITLOCK (&lock);            
            m_pServerSessionDispatcher = new CServerSessionDispatcher;
            m_pServerSessionDefault = new CServerSessionDefault;
            m_pMsgClose = new CConnectionMessage_Close;
            m_pMsgOpenBack = new CConnectionMessage_OpenBack;
            m_pMsgGeneric = new CGenericMessage;
            m_pMsgDispatch = new CDispatchMessage;
 	     	m_ConnectionListByName = new CStringToObjectCollection; 
 	     	m_ConnectionListByID = new CLongToObjectCollection; 
        	m_IsServerStarted = FALSE; 
        	m_ServerPort = 0; 
			m_SessionName = NULL;
			m_ClientReceivePort = 0; 
        	m_ConnectionCount = 0; 
			m_ulMaxServerConnection = 0; 
        	m_Listener = NULL; 
			m_ulConnectionIDCounter = 0; 
#ifdef PASS_SSL 
        	m_bSecuredConnection = FALSE; 
        	m_szCertificateFile = NULL; 
        	m_szPrivateKeyFile = NULL; 
			memset (m_szPrivateKeyPassword,0,sizeof(m_szPrivateKeyPassword)); 
 
#ifdef _WIN32 
			// Initialize random generator 
			RAND_screen(); 
#endif // _WIN32 
 
#endif //PASS_SSL 
	   	SetTickTime (10); 
         
#ifdef _WIN32 
        	// Initialise WinSock 
         
        	WORD wVersionRequested;WSADATA wsaData;int err;  
        	wVersionRequested = MAKEWORD( 1, 1 );  
        	err = WSAStartup( wVersionRequested, &wsaData ); 
        	if ( err == 0 )  
        	{ 
        		/* Confirm that the WinSock DLL supports 2.2.*/ 
        		/* Note that if the DLL supports versions greater    */ 
        		/* than 2.2 in addition to 2.2, it will still return */ 
        		/* 2.2 in wVersion since that is the version we      */ 
        		/* requested.                                        */  
        		if ( LOBYTE( wsaData.wVersion ) != 1 || 
        				HIBYTE( wsaData.wVersion ) != 1 ) 
        		{ 
        			/* Tell the user that we could not find a usable */ 
        			/* WinSock DLL.                                  */ 
        			WSACleanup( ); 
        		} 
        	} 
#endif //_WIN32 
        	 
        	// Register CConnection Message 
         
        	CObjectManager *manager = CObjectManager::Instance(); 
         
        	manager->RegisterObjectByNameAndType (m_pMsgClose,"ConnectionMessage_Close","Message/Connection"); 
        	manager->RegisterObjectByNameAndType (m_pMsgOpenBack,"ConnectionMessage_OpenBack","Message/Connection"); 
        	manager->RegisterObjectByNameAndType (m_pMsgGeneric,"GenericMessage","Message"); 
        	manager->RegisterObjectByNameAndType (m_pMsgDispatch,"DispatchMessage","Message");

			manager->RegisterObject (m_pServerSessionDispatcher);
			manager->RegisterObject (m_pServerSessionDefault);
 
			// By default set connection type as 'Dual' 
			memset (m_szConnectionType,0,20); 
			strcpy (m_szConnectionType,"Dual"); 

			// Set by default ServerSessionDefault
			SetSessionObjectName ("ServerSessionDefault");
        } 
         
        CConnectionManager::~CConnectionManager() 
        { 
		    PASS_DESTROYLOCK (&lock);
        	delete m_ConnectionListByName; 
            delete m_ConnectionListByID;    
        	if (m_SessionName != NULL) 
        	{ 
        		delete m_SessionName; 
        	}

            if(m_Listener != NULL)
            {
           		unsigned long timeout = 0;
   				m_Listener->Stop();
                m_Listener->Close();
  		        while(m_Listener->CanBeDeleted() == FALSE && timeout < 50) 
  		        {                 
                    PASS_MILLISLEEP(100);
           			timeout++;
                }
           		if (timeout == 50)
		        {
			        m_Listener->Kill();
                }
                delete m_Listener;
            }
            delete m_pMsgClose;
            delete m_pMsgOpenBack;
            delete m_pMsgGeneric;
            delete m_pMsgDispatch;
            delete m_pServerSessionDispatcher;
            delete m_pServerSessionDefault;
        } 
         
        void CConnectionManager::SetServerBasePort (unsigned short port) 
        { 
        	m_ServerPort = port; 
        	m_ClientReceivePort = port; 
        } 
         
        void CConnectionManager::SetSessionObjectName (char *name) 
        { 
        	if (name != NULL) 
        	{ 
        		if (m_SessionName != NULL) 
        		{ 
        			delete [] m_SessionName; 
        		} 
        		m_SessionName = new char [(strlen (name)) + 1]; 
        		strcpy (m_SessionName,name); 
        	} 
        } 
         
BOOL CConnectionManager::StartServer (void) 
{ 
	// check if the port is set 
	if (m_ServerPort == 0) 
	{ 
		return FALSE; 
	} 
 
	// First start the listener 
	if (m_Listener != NULL) 
	{ 
		// Allow only one listener at a time 
		return FALSE; 
	} 
 
	m_Listener = new CConnectionListener; 
 
	m_Listener->SetPort (m_ServerPort); 
	m_Listener->SetSessionObjectName (m_SessionName); 
	m_Listener->SetConnectionManager (this); 
 
#ifdef PASS_SSL 
	if (m_bSecuredConnection == TRUE) 
	{ 
		m_Listener->SetCertificateFile (m_szCertificateFile); 
		m_Listener->SetPrivateKeyFile (m_szPrivateKeyFile); 
		m_Listener->SetSecuredConnection (TRUE); 
	} 
#endif //PASS_SSL 
 
	if (!(m_Listener->Initialize ())) 
	{ 
		delete m_Listener; 
		m_Listener = NULL; 
		return FALSE; 
	} 
	m_Listener->SetTickTime (2); 
	m_Listener->Start(); 
 
	return TRUE; 
} 
         
        CConnection *CConnectionManager::CreateConnection (void) 
        { 
		PASS_LOCK (&lock);
 
			CConnection *localConnection = NULL; 
 
			if (strcmp (m_szConnectionType,"Dual") == 0) 
			{ 
				CConnection_Dual *localConnection_Dual = NULL; 
	        	localConnection_Dual = new CConnection_Dual; 
	        	// Set the connection listenner port 
	        	localConnection_Dual->SetReceivePort (++m_ClientReceivePort); 
#ifdef PASS_SSL 
	        	if (m_bSecuredConnection == TRUE) 
		     	{ 
		     		localConnection_Dual->SetCertificateFile (m_szCertificateFile); 
		     		localConnection_Dual->SetPrivateKeyFile (m_szPrivateKeyFile); 
		     		localConnection_Dual->SetSecuredConnection (m_bSecuredConnection); 
		     	} 
#endif //PASS_SSL 
 
	        	AddConnection(localConnection_Dual); 
				localConnection = (CConnection *)localConnection_Dual; 
			} 
			if (strcmp (m_szConnectionType,"Mono") == 0) 
			{ 
				CConnection_Mono *localConnection_Mono = NULL; 
	        	localConnection_Mono = new CConnection_Mono; 
#ifdef PASS_SSL 
	        	if (m_bSecuredConnection == TRUE) 
		     	{ 
		     		localConnection_Mono->SetCertificateFile (m_szCertificateFile); 
		     		localConnection_Mono->SetPrivateKeyFile (m_szPrivateKeyFile); 
		     		localConnection_Mono->SetSecuredConnection (m_bSecuredConnection); 
		     	} 
#endif //PASS_SSL 
 
	        	AddConnection(localConnection_Mono); 
				localConnection = (CConnection *)localConnection_Mono; 
			} 
 
			if (strcmp (m_szConnectionType,"Local") == 0) 
			{ 
				CConnection_Local *localConnection_Local = NULL; 
	        	localConnection_Local = new CConnection_Local; 
 
	        	AddConnection(localConnection_Local); 
				localConnection = (CConnection *)localConnection_Local; 
			} 

			// Set the local ID 
			localConnection->SetConnectionID (GetNewConnectionID()); 
 		PASS_UNLOCK (&lock);

     		return localConnection; 
      } 
      
     CConnection *CConnectionManager::CreateConnection (char *name) 
     { 
     		CConnection *localConnection = CreateConnection (); 
      
    	 	if (localConnection != NULL) 
     		{ 
     			localConnection->SetConnectionName (name); 
     		} 
      
        	return localConnection; 
     } 
         
        void CConnectionManager::AddConnection (CConnection *connection) 
        { 
	     	char buffer[20]; 
	     	char connectionName[256]; 
	      
	     	if (connection != NULL) 
	     	{ 
				m_ConnectionCount++; 
 
	     		connection->GetConnectionName (connectionName,256); 
 
				unsigned long newID = GetNewConnectionID(); 
	      
	     		if (strlen (connectionName) == 0) 
	     		{ 
	     			_snprintf (buffer,20,"%ld",newID); 

					m_ConnectionListByName->Add (buffer,(CObject *)connection); 
	     		} 
	     		else 
	     		{ 
	     			m_ConnectionListByName->Add (connectionName,(CObject *)connection); 
	     		} 
  
				if (connection->GetConnectionID() == 0) 
				{ 
					// Set a new connection ID 
					connection->SetConnectionID (newID); 
				} 
 
				m_ConnectionListByID->Add (connection->GetConnectionID(),connection); 
     		} 
        } 
         
        void CConnectionManager::Tick(void) 
        { 
        	// Check for released connections 
		PASS_LOCK (&lock);
  
  	// Check to restart server listener 
  	if (m_Listener != NULL) 
  	{ 
  		if (m_Listener->CanBeDeleted() == TRUE) 
  		{ 
  			Trace (__FILE__,__LINE__,LOGMASK_COMM,"The listener is down, try to restart it"); 
 
  			delete m_Listener; 
  			m_Listener = NULL; 
  			// The listener is down, try to restart it 
  			m_Listener = new CConnectionListener; 
   
#ifdef PASS_SSL 
  			if (m_bSecuredConnection == TRUE) 
  			{ 
  				m_Listener->SetCertificateFile (m_szCertificateFile); 
  				m_Listener->SetPrivateKeyFile (m_szPrivateKeyFile); 
  				m_Listener->SetSecuredConnection (TRUE); 
  			} 
#endif //PASS_SSL 
   
  			m_Listener->SetPort (m_ServerPort); 
  			m_Listener->SetSessionObjectName (m_SessionName); 
  			m_Listener->SetConnectionManager (this); 
  			if (!(m_Listener->Initialize ())) 
  			{ 
  				Trace (__FILE__,__LINE__,LOGMASK_COMM,"Can't restart listener"); 
  				m_Listener->Stop();			 
  			} 
  			else 
  			{ 
  				m_Listener->SetTickTime (2); 
  				m_Listener->Start(); 
  			} 
  		} 
  	} 
   
      	CConnection *deletedCon; // SCO ? OK !
      	BOOL bConDeleted = FALSE; 
         
      		ULONG32 position = 0; 
      		CConnection *con = NULL; 
      		char key[256]; 
       
      		con = (CConnection *)m_ConnectionListByName->GetFirst (&position,key,256); 
       
      		while (con != NULL && bConDeleted == FALSE) 
        	{ 
        		if (con->CanBeDeleted()) 
        		{ 
					Trace (__FILE__,__LINE__,LOGMASK_COMM,"Remove connection : %s with ID : %ld",key,con->GetConnectionID()); 
 
					// Remove from connection list by name 
					m_ConnectionListByName->Remove (key); 
 
					// Remove from connection list by ID 
					m_ConnectionListByID->Remove (con->GetConnectionID()); 
 
					m_ConnectionCount--; 
					// Delete connection object 
	      			deletedCon = con; 
        			delete con; 
        			bConDeleted = TRUE; 
	  			} 
				else 
				{			 
 		     		con = (CConnection *)m_ConnectionListByName->GetNext (&position,key,256); 
				} 
        	} 
  		
		PASS_UNLOCK (&lock);               
        } 
         
#ifdef PASS_SSL 
        void CConnectionManager::SetSecuredConnection (BOOL flag) 
        { 
        	m_bSecuredConnection = flag; 
         
        } 
         
        void CConnectionManager::SetCertificateFile (char *file) 
        { 
        	if (file != NULL) 
        	{ 
        		if (m_szCertificateFile != NULL) 
        		{ 
        			delete [] m_szCertificateFile; 
        		} 
        		m_szCertificateFile = new char [(strlen (file)) + 1]; 
        		strcpy (m_szCertificateFile,file); 
        	} 
        } 
         
        void CConnectionManager::SetPrivateKeyFile (char *file) 
        { 
        	if (file != NULL) 
        	{ 
        		if (m_szPrivateKeyFile != NULL) 
        		{ 
        			delete [] m_szPrivateKeyFile; 
        		} 
        		m_szPrivateKeyFile = new char [(strlen (file)) + 1]; 
        		strcpy (m_szPrivateKeyFile,file); 
        	} 
        } 
         
		void CConnectionManager::SetPrivateKeyPassword (char *pass) 
		{ 
			strncpy (m_szPrivateKeyPassword,pass,sizeof(m_szPrivateKeyPassword)); 
		} 
 
		char *CConnectionManager::GetPrivateKeyPassword (void) 
		{ 
			return m_szPrivateKeyPassword; 
		} 
 
#endif //PASS_SSL 
      CConnection *CConnectionManager::GetConnection (char *name) 
      { 
      	return (CConnection *)m_ConnectionListByName->Lookup (name); 
      } 
       
    CStringToObjectCollection *CConnectionManager::GetConnectionList (void) 
    { 
    	return m_ConnectionListByName; 
    } 
     
	CConnection *CConnectionManager::GetConnectionForID (unsigned long id) 
	{ 
		return (CConnection *)m_ConnectionListByID->Lookup (id); 
	} 
 
      void CConnectionManager::Lock (void) 
      { 
	PASS_LOCK (&lock);
      } 
       
      void CConnectionManager::Unlock (void) 
      {
	PASS_UNLOCK (&lock); 
      } 
 
void CConnectionManager::SetMaxServerConnection (unsigned long number) 
{ 
	m_ulMaxServerConnection = number; 
} 
 
unsigned long CConnectionManager::GetMaxServerConnection (void) 
{ 
	return m_ulMaxServerConnection; 
} 
 
unsigned long CConnectionManager::GetConnectionCount (void) 
{ 
	return m_ConnectionCount; 
} 
 
void CConnectionManager::SetConnectionType (char *type) 
{ 
	strncpy (m_szConnectionType,type,20); 
} 
 
char *CConnectionManager::GetConnectionType (void) 
{ 
	return m_szConnectionType; 
} 
 
unsigned long CConnectionManager::GetNewConnectionID(void) 
{ 
	return ++m_ulConnectionIDCounter; 
}
 
/*
void CConnectionManager::DeleteObjects(char *pType)
{
    ULONG32 position = 0; 
    CObject *pObject = NULL;

    CObjectManager *manager = CObjectManager::Instance(); 
    CLongToObjectCollection *pTemp = manager->GetObjectsForType (pType);

    if(pTemp != NULL)
    {
	    pObject = pTemp->GetFirst (&position); 

	    while (pObject != NULL) 
	    { 
            delete pObject;
		    pObject = pTemp->GetNext (&position); 
	    } 

        pTemp->RemoveAll();
        delete pTemp;
        pTemp = NULL;
    }
}
*/
