/* ------------------------------------------------------------------------
 * $Id: Vector2D.cc,v 1.2 2001/06/18 21:55:14 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-15 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <string.h>
#include <math.h>
#include <iostream>

// -- Local Includes
#include "Math.hh"
#include "Vector2D.hh"

// -- Code Segment

Vector2D::Vector2D(float x, float y)
{
    _coord[0] = x; _coord[1] = y;
}

Vector2D::Vector2D(const float c[2])
{
    clear();
    memcpy(_coord, c, sizeof(float) * 2);
}

Vector2D::Vector2D(const Vector2D &v)
{
    *this = v;
}

Vector2D &Vector2D::operator += (const Vector2D &v)
{
    for (int i = 0; i < 2; i++)
	_coord[i] += v._coord[i];
    return *this;
}

Vector2D &Vector2D::operator -= (const Vector2D &v)
{    
    for (int i = 0; i < 2; i++)
        _coord[i] -= v._coord[i];
    return *this;
}

Vector2D Vector2D::operator + (const Vector2D &v) const
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
        r._coord[i] = _coord[i] + v._coord[i];
    return r;
}

Vector2D Vector2D::operator - (const Vector2D &v) const
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
	r._coord[i] = _coord[i] - v._coord[i];
    return r;
}

Vector2D Vector2D::operator / (const Vector2D &v) const
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
	r._coord[i] = _coord[i] / v._coord[i];
    return r;
}

Vector2D &Vector2D::operator += (float f)
{
    for (int i = 0; i < 2; i++)
        _coord[i] += f;
    return *this;
}

Vector2D &Vector2D::operator -= (float f)
{
    for (int i = 0; i < 2; i++)
	_coord[i] -= f;
    return *this;
}

Vector2D &Vector2D::operator /= (float f)
{
    for (int i = 0; i < 2; i++)
	_coord[i] /= f;
    return *this;
}

Vector2D &Vector2D::operator *= (float f)
{
    for (int i = 0; i < 2; i++)
	_coord[i] *= f;
    return *this;
}

Vector2D Vector2D::operator + (float f) const
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
	r._coord[i] = _coord[i] + f;
    return r;
}

Vector2D Vector2D::operator - (float f) const
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
	r._coord[i] = _coord[i] - f;
    return r;
}

Vector2D Vector2D::operator / (float f) const
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
	r._coord[i] = _coord[i] / f;
    return r;
}

Vector2D Vector2D::operator * (float f) const
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
	r._coord[i] = _coord[i] * f;
    return r;
}

Vector2D Vector2D::operator - () const
{
    return Vector2D(-_coord[0], -_coord[1]);
}

Vector2D &Vector2D::operator = (const Vector2D &v)
{
    memcpy(_coord, v._coord, sizeof(_coord));
    for (int i = 0; i < 2; i++)
	_coord[i] = v._coord[i];
    return *this;
}

float Vector2D::magnitude()
{
    return (float) sqrt(_coord[0] * _coord[0] + 
			_coord[1] * _coord[1]);
}

void Vector2D::normalize()
{
    // Compute magnitude of vector
    float mag = magnitude();
    
    // Sanity check
    if (mag == 0.0f) return;
    
    // Normalize all components with magnitude
    for (int i = 0; i < 2; i++)
	_coord[i] /= mag;
}

void Vector2D::clear()
{
    for (int i = 0; i < 2; i++) _coord[i] = 0.0f;
    _coord[2] = 1.0f;
}

void Vector2D::zero() 
{
    clear();
}


float Vector2D::dot(const Vector2D &a, const Vector2D &b)
{
    float r = 0.0f;
    for (int i = 0; i < 2; i++)
	r += a(i) * b(i);
    return r;
}

Vector2D operator * (float f, const Vector2D &v)
{
    Vector2D r;
    for (int i = 0; i < 2; i++)
	r(i) = f * v(i);
    return r;
}

std::ostream & operator << (std::ostream &f, const Vector2D &v)
{
    f << "[" << v.x() << ", " << v.y() << "]'";
    return f;
}

bool operator == (const Vector2D &a, const Vector2D &b)
{
    for (int i = 0; i < 2; i++) 
	if (Math::equal(a(i), b(i)) == false)
	    return false;
    return true;
}
