/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/**
 * If you hack around, make sure all pathes are returned with a trailing
 * file separator and all values are initialized to something other than
 * null.
 *
 * Most important, keep all preferences sorted alphabetically everywhere.
 */
package xnap.util;

import xnap.*;
import xnap.plugin.PluginManager;
import xnap.net.*;
import xnap.util.prefs.IntValidator;
import xnap.util.prefs.StringValidator;

import java.awt.*;
import java.beans.*;
import java.io.*;
import java.net.*;
import java.util.*;

public class Preferences extends PreferencesSupport 
    implements TablePreferencesSupport {

    //--- Constant(s) ---

    public static final String XNAP1_FILENAME = "xnap.properties";
    public static final String XNAP2_FILENAME = "xnap2.properties";

    public static final int PREFS_VERSION = 10;

    //--- Data field(s) ---

    private static Preferences singleton = null;

    public static String nativeLauncherCmd = "";
    public static String nativeLauncherType = "other";

    public static final boolean isMacOSX;
    public static final boolean isWindowsOS;
    static {
	String os = System.getProperty("os.name").toLowerCase();
	isWindowsOS = (os.indexOf("windows") != -1);
	if (isWindowsOS) {
	    nativeLauncherType = "ziga.dll";
	}
	String version = System.getProperty("os.version");
        isMacOSX = (version.startsWith("10.0"));
    }

    private long lastLaunchReleaseNr = -1;

    //--- Constructor(s) ---

    private Preferences(String filename)
    {
	super(filename, PREFS_VERSION, "xnap");
    }

    //--- Method(s) ---

    /**
     * There should only ever be one copy of Preferences instantiated.
     */
    public static synchronized Preferences getInstance() 
    {
	if (singleton == null) {
	    String f = FileHelper.getHomeDir() + XNAP2_FILENAME;
	    singleton = new Preferences(f);
	}

	return singleton;
    }

    public void read() 
    {
	if (prefsFile.exists()) {
	    super.read();
	}
	else {
	    File f = new File(FileHelper.getHomeDir() + XNAP1_FILENAME);
	    if (f.exists()) {
		super.read(f);
	    }
	}

	defaults();

	if (lastLaunchReleaseNr == -1) {
	    // remember version for this session
	    lastLaunchReleaseNr = getLong("lastLaunchReleaseNr");
	    // set to new value
	    set("lastLaunchReleaseNr", XNap.RELEASE_NR);
	}
    }

    public void convert(int oldVersion) 
    {
	if (oldVersion <= 2) {
	    renameProperty("xnap.autofetchnaptigator", 
			   "xnap.plugin.nap.autoFetchNaptigator");
	    renameProperty("xnap.autoversioncheck", "xnap.autoVersionCheck");
	    renameProperty("xnap.clientinfo", "xnap.plugin.nap.clientInfo");
	    renameProperty("xnap.delincompletefiles", 
			   "xnap.delIncompleteFiles");
	    renameProperty("xnap.downloaddir", "xnap.downloadDir");
	    renameProperty("xnap.filterresults", "xnap.filterResults");
	    renameProperty("xnap.http.proxy.host", "xnap.httpProxyHost");
	    renameProperty("xnap.http.proxy.port", "xnap.httpProxyPort");
	    renameProperty("xnap.incompletedir", "xnap.incompleteDir");
	    renameProperty("xnap.lastupdatecheck", "xnap.lastUpdateCheck");
	    renameProperty("xnap.linktype", "xnap.linkType");
	    renameProperty("xnap.localport", "xnap.plugin.nap.localPort");
	    renameProperty("xnap.lookandfeel", "xnap.lookAndFeel");
	    renameProperty("xnap.maxautoconnectservers", 
			   "xnap.plugin.nap.maxAutoconnectServers");
	    renameProperty("xnap.maxconsolelines", "xnap.maxConsoleLines");
	    renameProperty("xnap.maxdownloads", "xnap.maxDownloads");
	    renameProperty("xnap.maxsearchresults", "xnap.maxSearchResults");
	    renameProperty("xnap.maxsearchservers", 
			   "xnap.plugin.nap.maxSearchServers");
	    renameProperty("xnap.maxuploads", "xnap.maxUploads");
	    renameProperty("xnap.mp3playercmd", "xnap.mp3PlayerCmd");
	    renameProperty("xnap.mp3playertype", "xnap.mp3PlayerType");
	    renameProperty("xnap.gift.daemon", "xnap.plugin.gift.daemon");
	    renameProperty("xnap.gift.host", "xnap.plugin.gift.host");
	    renameProperty("xnap.gift.port", "xnap.plugin.gift.port");
	    renameProperty("xnap.removeduplicateresults", 
			   "xnap.removeDuplicateResults");
	    renameProperty("xnap.search.bitrate", "xnap.searchBitrate");
	    renameProperty("xnap.search.compare", "xnap.searchCompare");
	    renameProperty("xnap.search.mediatype", "xnap.searchMediaType");
	    renameProperty("xnap.showclosedialog", "xnap.showCloseDialog");
	    renameProperty("xnap.showicons", "xnap.showIcons");
	    renameProperty("xnap.showsplash", "xnap.showSplash");
	    renameProperty("xnap.socksproxyhost", "xnap.socksProxyHost");
	    renameProperty("xnap.socksproxyport", "xnap.socksProxyPort");
	    renameProperty("xnap.uploaddirs", "xnap.uploadDirs");
	    renameProperty("xnap.useautoconnector", 
			   "xnap.plugin.nap.useAutoconnector");
	    renameProperty("xnap.usehttpproxy", "xnap.useHttpProxy");
	    removeProperty("xnap.uselimewire");
	    renameProperty("xnap.useopenfileaction", "xnap.useOpenFileAction");
	    renameProperty("xnap.usesingleport", 
			   "xnap.plugin.nap.useSinglePort");
	    renameProperty("xnap.useSocksProxy", "xnap.useSocksProxy");
	    renameProperty("xnap.window.height", "xnap.windowHeight");
	    renameProperty("xnap.window.width", "xnap.windowWidth");
	    renameProperty("xnap.window.x", "xnap.windowX");
	    renameProperty("xnap.window.y", "xnap.windowY");

	    setProperty("plugin.nap.useSinglePort", "true");
	    if (getInt("plugin.nap.localPort") == 0) {
		setProperty("firewalled", "true");
	    }
	}

	if (oldVersion <= 3) {
	    if (getDownloadThrottle() > 0) {
		setThrottleDownloads(true);
	    }
	    if (getUploadThrottle() > 0) {
		setThrottleUploads(true);
	    }
	}

	if (oldVersion <= 4) {
	    renameProperty("xnap.useOpenFileAction", "xnap.fileLauncherType");
	    if (!isWindowsOS && getBoolean("fileLauncherType")) {
		setFileLauncherCmd("kfmclient");
		setFileLauncherType("kfmclient");
	    }
	    else {
		setFileLauncherCmd(nativeLauncherCmd);
		setFileLauncherType(nativeLauncherType);
	    }
	}

	if (oldVersion <= 5) {
	    removeProperty("maintainSortOrder");
	    setTableColumns("channel", "0;1;2");
	    setTableColumns("download", "0;1;2;4;5;6");
	    setTableColumns("library", "0;3;4;6");
	    setTableColumns("search", "0;2;3;5;6;10");
	    setTableColumns("upload", "0;1;2;4;5;6");
	}

	if (oldVersion <= 6) {
	    setColor("progressRunning", 11513855);
	    setColor("progressFinished", 65280);
	}

	if (oldVersion <= 7) {
  	    setLimitDownloads(getMaxDownloads() > 0);
  	    setLimitUploads(getMaxUploads() > 0);
	}

	if (oldVersion <= 8) {
	    String lastLaunchVer = get("lastLaunchVer");
	    removeProperty("lastLaunchVer");
	    if (VersionParser.compare(lastLaunchVer, "2.0-pre1") > 0) {
		// make sure the user is not bothered with the wizard
		set("lastLaunchReleaseNr", "1");
	    }
	}

	if (oldVersion <= 9) {
	    setDownloadRetryInterval(120);
	}
    }

    public void defaults()
    {
	setDefault("lastLaunchReleaseNr", "-1");

	setDefault("AudioDownloadDir", "");
	setDefault("appendServerNameToChatUser", "false");
	setDefault("autoComplete", "false");
	setDefault("autoJoinChannels", "#xnap");
	setDefault("autoResumeDownloads", "false");
	setDefault("autoVersionCheck", "true");
	setDefault("autoDownlodMaxSearches", "12");
	setDefault("autoDownlodSearchInterval", "900", 
		   new IntValidator(600));
	setDefault("beepOnChatMessage", "false");
	setDefault("blinkOnChannelJoin", "true");
	setDefault("browseTableColumns", "0;2;3;5;10");
	setDefault("buildSearchTreeOnStartup", "false");
	setDefault("captureLauncherOutput", "false");
	setDefault("channelTableColumns", "0;1;2");
	setDefault("chatBackgroundColor", "16777215");
	setDefault("chatErrorColor", "255");
	setDefault("chatFont", "Monospaced;0;12");
	setDefault("chatIgnoreMessage", "I'm ignoring you, please stop sending"
		   + " messages.");
	setDefault("chatInfoColor", "51455");
	setDefault("chatMessageColor", "0");
	setDefault("chatUserColor", "16711680");
	setDefault("chatVerticalDividerLocation", "400");
	setDefault("chatGlobalVerticalDividerLocation", "400");
	setDefault("consoleFont", "Monospaced;1;12");
	setDefault("consoleForegroundColor", "51455");
	setDefault("consoleBackgroundColor", "0");
	setDefault("consoleBackgroundColor", "0");
	setDefault("correctivePixels", "0");
	setDefault("debugLogFile", FileHelper.getHomeDir() + "debug.log");
	setDefault("delIncompleteFiles", "true");
	setDefault("DocumentsDownloadDir", "");
	setDefault("downloadDir", 
		   FileHelper.directory(System.getProperty("user.home")));
	setDefault("downloadMaxTries", "60");
	setDefault("downloadRetryInterval", "300",
		   new IntValidator(60));
	setDefault("downloadTableColumns", "0;1;2;4;5;6");
	setDefault("downloadThrottle", "0", new IntValidator(0));
	setDefault("doubleFilterMediaType", "true");
	setDefault("email", "anonymous@universe", StringValidator.EMAIL);
	setDefault("enabledPlugins", "xnap.plugin.nap.Plugin");
	setDefault("enabledGUIPlugins",
		   "xnap.plugin.nap.GUIPlugin;xnap.plugin.viewer.mp3.Plugin;"
		   + "xnap.plugin.viewer.image.Plugin");
	setDefault("faqURL", "http://xnap.sourceforge.net/faq.php");
	setDefault("feedbackEmail", "");
	setDefault("feedbackName", "");
	setDefault("feedbackURL", "http://xnap.sourceforge.net/feedback.php");
	setDefault("fileLauncherCmd", nativeLauncherCmd);
	setDefault("fileLauncherType", nativeLauncherType);
	setDefault("filterResults", "false");
	setDefault("firewalled", "false");
	setDefault("focusOnAllEvents", "true");
	setDefault("hotlistDividerLocation", "200");
	setDefault("hotlistTableColumns", "0;3;4;5");
	setDefault("httpProxyHost", "");
	setDefault("httpProxyPort", "0");
	setDefault("ImagesDownloadDir", "");
	setDefault("incompleteDir", "");
	setDefault("language", "");
	setDefault("lastUpdateCheck", "0");
	setDefault("libraryHorizontalDividerLocation", "220");
	setDefault("libraryTableColumns", "0;3;4;6");
	setDefault("libraryVerticalDividerLocation", "200");
	setDefault("libraryTreeNodes", 
		   "downloads;incompletes;home;root;shares");
	setDefault("limitDownloads", "false");
	setDefault("limitDownloadAttempts", "false");
	setDefault("limitUploads", "true");
	setDefault("limitDownloadsPerUser", "true");
	setDefault("limitUploadsPerUser", "true");
	setDefault("linkType", "0");
	setDefault("logTransfersToFile", "false");
	setDefault("lookAndFeel", 
		   "com.incors.plaf.kunststoff.KunststoffLookAndFeel");
	setDefault("maxDownloads", "20", new IntValidator(0));
	setDefault("maxDownloadsPerUser", "3", new IntValidator(0));
	setDefault("maxConsoleLines", "100", new IntValidator(1));
	setDefault("maxOpenWithCommands", "5");
	setDefault("maxSwarmedConnects", "1");
	setDefault("maxUploads", "5", new IntValidator(0));
	setDefault("maxUploadsPerUser", "1", new IntValidator(0));
	setDefault("maxSearchResults", "200");
	setDefault("mp3PlayerCmd", "");
	setDefault("mp3PlayerType", "other");
	setDefault("openWithCommands", "");
	setDefault("password", StringHelper.randomString(8),
		   StringValidator.REGULAR_STRING);
	setDefault("prefsWindowHeight", "520");
	setDefault("prefsWindowWidth", "640");
	setDefault("progressRunningColor", "11513855");
	setDefault("progressFinishedColor", "65280");
	setDefault("rememberFeedback", "true");
	setDefault("removeDuplicateResults", "true");
	setDefault("searchBitrate", "0");
	setDefault("searchCompare", "0");
	setDefault("searchDividerLocation", "200");
	setDefault("searchHistorySize", "30");
	setDefault("searchMediaType", "0");
	setDefault("searchTableColumns", "0;2;3;5;6;10");
	setDefault("searchTableSortedColumn", "-11");
	setDefault("sendChatIgnoreMessage", "true");
	setDefault("PrintServerNotificationsInChatWindow", "false");
	setDefault("SoftwareDownloadDir", "");
	setDefault("shareFullPath", "true");
	setDefault("showChatMsgTime", "true");
	setDefault("showCloseDialog", "true");
	setDefault("showCmdlToolbar", "false");
	setDefault("showCopyDialog", "true");
	setDefault("showDeleteDownloadsDialog", "true");
	setDefault("showDeleteFilesDialog", "true");
	setDefault("showIcons", "true");
	setDefault("showMainToolbar", "true");
	setDefault("showMoveDialog", "true");
	setDefault("showSearchOptions", "true");
	setDefault("showSplash", "true");
	setDefault("showToolTips", !Updater.hasMacOSXToolTipsBug() + "");
	setDefault("socksProxyHost", "");
	setDefault("socksProxyPort", "0");
	setDefault("theme", "xnap.gui.theme.PlainTheme");
	setDefault("transferDividerLocation", "220");
	setDefault("transferLogFile", FileHelper.getHomeDir() + "transfer.log");
	setDefault("throttleDownloads", "false");
	setDefault("throttleUploads", "false");
	setDefault("updaterInterval", "600", new IntValidator(60));
	setDefault("updateURL",
		   "http://xnap.sourceforge.net/updates/LATEST_VERSION.txt");
	setDefault("uploadDirs", "");
	setDefault("uploadTableColumns", "0;1;2;4;5;6");
	setDefault("uploadThrottle", "0", new IntValidator(0));
	setDefault("useEmacsKeyBindings", "false");
	setDefault("username", StringHelper.randomString(8),
		   StringValidator.REGULAR_STRING);
	setDefault("useHttpProxy", "false");
	setDefault("userTableColumns", "0;1");
	setDefault("useSocksProxy", "false");
	setDefault("useTabbedPane", "false");
	setDefault("useTabbedTransferPane", "false");
	setDefault("VideoDownloadDir", "");
	setDefault("windowHeight", "480");
	setDefault("windowWidth", "640");
	setDefault("windowX", "0");
	setDefault("windowY", "0");
	setDefault("xnapJarURL", "http://xnap.sourceforge.net/redirect.php");
    }

    public synchronized
	void addColorListener(String color, PropertyChangeListener l)
    {
	addPropertyChangeListener(color + "Color", l);
    }

    public synchronized
	void addFontListener(String font, PropertyChangeListener l)
    {
	addPropertyChangeListener(font + "Font", l);
    }

    public synchronized 
	void addTableListener(String table, PropertyChangeListener l)
    {
	addPropertyChangeListener(table + "TableColumns", l);
	addPropertyChangeListener(table + "TableMaintainSortOrder", l);
    }

    public boolean getAppendServerNameToChatUser(){
	return getBoolean("appendServerNameToChatUser");
    }

    public void setAppendServerNameToChatUser(boolean newValue) {
	set("appendServerNameToChatUser", newValue);
    }

    public boolean getAutoComplete() {
	return getBoolean("autoComplete");
    }

    public void setAutoComplete(boolean newValue) {
	set("autoComplete", newValue);
    }

    public int getAutoDownloadMaxSearches() {
	return getInt("autoDownlodMaxSearches");
    }

    public void setAutoDownloadMaxSearches(int newValue) {
	set("autoDownlodMaxSearches", newValue);
    }

    public int getAutoDownloadSearchInterval() {
	return getInt("autoDownlodSearchInterval");
    }

    public void setAutoDownloadSearchInterval(int newValue) {
	set("autoDownlodSearchInterval", newValue);
    }

    public String getAutoJoinChannels() {
        return get("autoJoinChannels");
    }

    public String[] getAutoJoinChannelsArray() {
	return StringHelper.toArray(getAutoJoinChannels(), ARRAY_SEPARATOR);
    }

    public void setAutoJoinChannels(String newValue) {
	set("autoJoinChannels", newValue.toLowerCase());
    }

    public boolean getAutoResumeDownloads() {
	return getBoolean("autoResumeDownloads");
    }

    public void setAutoResumeDownloads(boolean newValue) {
	set("autoResumeDownloads", newValue);
    }

    public boolean getAutoVersionCheck() {
	return getBoolean("autoVersionCheck");
    }

    public void setAutoVersionCheck(boolean newValue) {
	set("autoVersionCheck", newValue);
    }

    public boolean getBeepOnChatMessage() {
	return getBoolean("beepOnChatMessage");
    }

    public void setBeepOnChatMessage(boolean newValue) {
	set("beepOnChatMessage", newValue);
    }

    public boolean getBlinkOnChannelJoin() {
	return getBoolean("blinkOnChannelJoin");
    }

    public void setBlinkOnChannelJoin(boolean newValue) {
	set("blinkOnChannelJoin", newValue);
    }

    public boolean getBuildSearchTreeOnStartup() {
	return getBoolean("buildSearchTreeOnStartup");
    }

    public void setBuildSearchTreeOnStartup(boolean newValue) {
	set("buildSearchTreeOnStartup", newValue);
    }

    public boolean getCaptureLauncherOutput() {
	return getBoolean("captureLauncherOutput");
    }

    public void setCaptureLauncherOutput(boolean newValue) {
	set("captureLauncherOutput", newValue);
    }

    public String getChatIgnoreMessage() {
	return get("chatIgnoreMessage");
    }

    public void setChatIgnoreMessage(String newValue) {
	set("chatIgnoreMessage", newValue);
    }
    
    public int getChatGlobalVerticalDividerLocation() {
        return getInt("chatGlobalVerticalDividerLocation");
    }

    public void setChatGlobalVerticalDividerLocation(int newValue) {
	set("chatGlobalVerticalDividerLocation", newValue);
    }

    public int getChatVerticalDividerLocation() {
        return getInt("chatVerticalDividerLocation");
    }

    public void setChatVerticalDividerLocation(int newValue) {
	set("chatVerticalDividerLocation", newValue);
    }

    public int getColor(String key) {
	return getInt(key + "Color");
    }

    public void setColor(String key, int newValue) {
	set(key + "Color", newValue);
    }

    public int getCorrectivePixels() {
	return getInt("correctivePixels");
    }

    public void setCorrectivePixels(int newValue) {
	set("correctivePixels", newValue);
    }

    public String getDebugLogFile() {
	return get("debugLogFile");
    }

    public void setDebugLogFile(String newValue) {
	set("debugLogFile", newValue);
    }

    public boolean getDelIncompleteFiles() {
	return getBoolean("delIncompleteFiles");
    }

    public void setDelIncompleteFiles(boolean newValue) {
	set("delIncompleteFiles", newValue);
    }
    
    public String getDownloadDir() {
	return FileHelper.directory(get("downloadDir"));
    }
    
    public void setDownloadDir(String newValue) {
	set("downloadDir", FileHelper.directory(newValue));
    }

    public String getMediaTypeDownloadDir(String type) {
	String dir = get(type + "DownloadDir");
	if (dir.length() > 0) {
	    return FileHelper.directory(dir);
	}
	else {
	    return dir;
	}
    }

    public void setMediaTypeDownloadDir(String type, String newValue) {
	set(type + "DownloadDir", newValue);
    }

    public int getDownloadMaxTries() {
        return getInt("downloadMaxTries");
    }    

    public int getDownloadRetryInterval() {
        return getInt("downloadRetryInterval");
    }

    public void setDownloadRetryInterval(int newValue) {
        set("downloadRetryInterval", newValue);
    }

    public int getDownloadThrottle() {
        return getInt("downloadThrottle");
    }

    public void setDownloadThrottle(int newValue) {
	set("downloadThrottle", newValue);
    }

    public String getEmail() {
	return get("email");
    }

    public void setEmail(String newValue) {
	set("email", newValue);
    }

    public String getEnabledPlugins() {
	return get("enabledPlugins");
    }

    public void setEnabledPlugins(String newValue) {
	set("enabledPlugins", newValue);
    }

    public String getEnabledGUIPlugins() {
	return get("enabledGUIPlugins");
    }

    public void setEnabledGUIPlugins(String newValue) {
	set("enabledGUIPlugins", newValue);
    }

    public String getFaqURL() {
	return get("faqURL");
    }

    public String getFeedbackEmail() {
	return get("feedbackEmail");
    }

    public void setFeedbackEmail(String newValue) {
	set("feedbackEmail", newValue);
    }

    public String getFeedbackName() {
	return get("feedbackName");
    }

    public void setFeedbackName(String newValue) {
	set("feedbackName", newValue);
    }

    public String getFeeedbackURL() {
	return get("feedbackURL");
    }

    public String getFileLauncherCmd() {
	return get("fileLauncherCmd");
    }

    public void setFileLauncherCmd(String newValue) {
	set("fileLauncherCmd", newValue);
    }

    public String getFileLauncherType() {
	return get("fileLauncherType");
    }

    public void setFileLauncherType(String newValue) {
	set("fileLauncherType", newValue);
    }

    public String getFont(String key) {
	return get(key + "Font");
    }

    public void setFont(String key, String newValue) {
	set(key + "Font", newValue);
    }

    public boolean isFirewalled() {
	return getBoolean("firewalled");
    }

    public void setFirewalled(boolean newValue) {
	set("firewalled", newValue);
    }

    public boolean getFilterResults() {
	return getBoolean("filterResults");
    }

    public void setFilterResults(boolean newValue) {
	set("filterResults", newValue);
    }

    public boolean getFocusOnAllEvents() {
	return getBoolean("focusOnAllEvents");
    }

    public void setFocusOnAllEvents(boolean newValue) {
	set("focusOnAllEvents", newValue);
    }

    public boolean getNotShowDialog(String dialogName) {
	dialogName = StringHelper.toFirstUpper(dialogName);
	return getBoolean("notShowDialog" + dialogName);
    }

    public void setNotShowDialog(String dialogName, boolean newValue) {
	dialogName = StringHelper.toFirstUpper(dialogName);
	set("notShowDialog" + dialogName, newValue);
    }

    public int getHotlistDividerLocation() {
        return getInt("hotlistDividerLocation");
    }

    public void setHotlistDividerLocation(int newValue) {
	set("hotlistDividerLocation", newValue);
    }

    public String getHttpProxyHost() {
        return get("httpProxyHost");
    }

    public void setHttpProxyHost(String newValue) {
	set("httpProxyHost", newValue);
    }

    public int getHttpProxyPort() {
	return getInt("httpProxyPort");
    }

    public void setHttpProxyPort(int newValue) {
	set("httpProxyPort", newValue);
    }

    public String getIncompleteDir() {
	return FileHelper.directory(get("incompleteDir"));
    }

    public void setIncompleteDir(String newValue) {
	set("incompleteDir", newValue);
    }

    public String getLanguage() {
	return get("language");
    }

    public void setLanguage(String newValue) {
	set("language", newValue);
    }

    public long getLastLaunchReleaseNr() {
	return lastLaunchReleaseNr;
    }

    public boolean getLimitDownloads() {
	return getBoolean("limitDownloads");
    }

    public void setLimitDownloads(boolean newValue) {
	set("limitDownloads", newValue);
    }

    public boolean getLimitUploads() {
	return getBoolean("limitUploads");
    }

    public void setLimitUploads(boolean newValue) {
	set("limitUploads", newValue);
    }

    public boolean getLimitDownloadsPerUser() {
	return getBoolean("limitDownloadsPerUser");
    }

    public void setLimitDownloadsPerUser(boolean newValue) {
	set("limitDownloadsPerUser", newValue);
    }

    public boolean getLimitUploadsPerUser() {
	return getBoolean("limitUploadsPerUser");
    }

    public void setLimitUploadsPerUser(boolean newValue) {
	set("limitUploadsPerUser", newValue);
    }

    public int getLinkType() {
	return getInt("linkType");
    }

    public void setLinkType(int newValue) {
	set("linkType", newValue);
    }

    public boolean getLogTransfersToFile() {
	return getBoolean("logTransfersToFile");
    }

    public void setLogTransfersToFile(boolean newValue) {
	set("logTransfersToFile", newValue);
    }

    public String getPassword() {
        return get("password");
    }

    public void setPassword(String newValue) {
	set("password", newValue);
    }

    public int getProgressColor() {
	return getInt("progressColor");
    }

    public void setProgressColor(int newValue) {
	set("progressColor", newValue);
    }

    public int getProgressColorFinished() {
	return getInt("progressColorFinished");
    }

    public void setProgressColorFinished(int newValue) {
	set("progressColorFinished", newValue);
    }

    public long getLastUpdateCheck() {
        return getLong("lastUpdateCheck");
    }

    public void setLastUpdateCheck(long newValue) {
	set("lastUpdateCheck", newValue);
    }

    public int getLibraryHorizontalDividerLocation() {
        return getInt("libraryHorizontalDividerLocation");
    }

    public void setLibraryHorizontalDividerLocation(int newValue) {
	set("libraryHorizontalDividerLocation", newValue);
    }

    public String getLibraryTreeNodes() {
        return get("libraryTreeNodes");
    }

    public String[] getLibraryTreeNodesArray() {
	return StringHelper.toArray(getLibraryTreeNodes(), ARRAY_SEPARATOR);
    }

    public void setLibraryTreeNodes(String newValue) {
	set("libraryTreeNodes", newValue);
    }

    public int getLibraryVerticalDividerLocation() {
        return getInt("libraryVerticalDividerLocation");
    }

    public void setLibraryVerticalDividerLocation(int newValue) {
	set("libraryVerticalDividerLocation", newValue);
    }

    public String getLookAndFeel() {
        return get("lookAndFeel");
    }

    public void setLookAndFeel(String newValue) {
	set("lookAndFeel", newValue);
    }


    public boolean getSendChatIgnoreMessage() {
	return getBoolean("sendChatIgnoreMessage");
    }

    public void setSendChatIgnoreMessage(boolean newValue) {
	set("sendChatIgnoreMessage", newValue);
    }
    
    public boolean getPrintServerNotificationsInChatWindow() {
	return getBoolean("PrintServerNotificationsInChatWindow");
    }

    public void setPrintServerNotificationsInChatWindow(boolean newValue) {
	set("PrintServerNotificationsInChatWindow", newValue);
    }

    public boolean getShareFullPath() {
	return getBoolean("shareFullPath");
    }

    public void setShareFullPath(boolean newValue) {
	set("shareFullPath", newValue);
    }

    public boolean shouldCheckForUpdate(int days) {
        return (System.currentTimeMillis()
		- getLastUpdateCheck() > days * 24 * 60 * 60 * 1000);
    }

    public boolean getShowSearchOptions() {
	return getBoolean("showSearchOptions");
    }

    public void setShowSearchOptions(boolean newValue) {
	  set("showSearchOptions", newValue);
    }

    public boolean getShowCloseDialog() {
	return getBoolean("showCloseDialog");
    }

    public void setShowCloseDialog(boolean newValue) {
	set("showCloseDialog", newValue);
    }

    public boolean getShowChatMsgTime() {
	return getBoolean("showChatMsgTime");
    }

    public void setShowChatMsgTime(boolean newValue) {
	set("showChatMsgTime", newValue);
    }

    public boolean getShowCmdlToolbar() {
	return getBoolean("showCmdlToolbar");
    }

    public void setShowCmdlToolbar(boolean newValue) {
	set("showCmdlToolbar", newValue);
    }
    
    public boolean getShowCopyDialog() {
	return getBoolean("showCopyDialog");
    }
    
    public void setShowCopyDialog(boolean newValue) {
	set("showCopyDialog", newValue);
    }


    public boolean getShowDeleteDownloadsDialog() {
	return getBoolean("showDeleteDownloadsDialog");
    }

    public void setShowDeleteDownloadsDialog(boolean newValue) {
	set("showDeleteDownloadsDialog", newValue);
    }

    public boolean getShowDeleteFilesDialog() {
	return getBoolean("showDeleteFilesDialog");
    }

    public void setShowDeleteFilesDialog(boolean newValue) {
	set("showDeleteFilesDialog", newValue);
    }

    public boolean getShowIcons() {
	return getBoolean("showIcons");
    }

    public void setShowIcons(boolean newValue) {
	   set("showIcons", newValue);
    }

    public boolean getShowMainToolbar() {
	return getBoolean("showMainToolbar");
    }

    public void setShowMainToolbar(boolean newValue) {
	set("showMainToolbar", newValue);
    }

    public boolean getShowSplash() {
	return getBoolean("showSplash");
    }

    public void setShowSplash(boolean newValue) {
	  set("showSplash", newValue);
    }

    public boolean getShowToolTips() {
	return getBoolean("showToolTips");
    }

    public void setShowToolTips(boolean newValue) {
	   set("showToolTips", newValue);
    }

    public int getMaxDownloads() {
        return getInt("maxDownloads");
    }

    public void setMaxDownloads(int newValue) {
	set("maxDownloads", newValue);
    }

    public int getMaxDownloadsPerUser() {
        return getInt("maxDownloadsPerUser");
    }

    public void setMaxDownloadsPerUser(int newValue) {
	set("maxDownloadsPerUser", newValue);
    }

    public int getMaxConsoleLines() {
        return getInt("maxConsoleLines");
    }

    public void setMaxConsoleLines(int newValue) {
	set("maxConsoleLines", newValue);
    }

    public int getMaxOpenWithCommands() {
        return getInt("maxOpenWithCommands");
    }

    public void setMaxOpenWithCommands(int newValue) {
	set("maxOpenWithCommands", newValue);
    }

    public int getMaxUploads() {
        return getInt("maxUploads");
    }

    public void setMaxUploads(int newValue) {
	set("maxUploads", newValue);
    }

    public int getMaxUploadsPerUser() {
        return getInt("maxUploadsPerUser");
    }

    public void setMaxUploadsPerUser(int newValue) {
	set("maxUploadsPerUser", newValue);
    }

    public int getMaxSearchResults() {
        return getInt("maxSearchResults");
    }

    public void setMaxSearchResults(int newValue) {
	set("maxSearchResults", newValue);
    }

    public int getMaxSwarmedConnects() {
	return getInt("maxSwarmedConnects");
    }

    public boolean getShowMoveDialog() {
	return getBoolean("showMoveDialog");
    }
    
    public void setShowMoveDialog(boolean newValue) {
	set("showMoveDialog", newValue);
    }

    public String getMP3PlayerCmd() {
        return get("mp3PlayerCmd");
    }

    public void setMP3PlayerCmd(String newValue) {
	set("mp3PlayerCmd", newValue);
    }

    public String getMP3PlayerType() {
        return get("mp3PlayerType");
    }

    public void setMP3PlayerType(String newValue) {
	set("mp3PlayerType", newValue);
    }

    public void setLimitDownloadAttempts(boolean newValue) {
	set("limitDownloadAttempts", newValue);
    }

    public boolean getLimitDownloadAttempts() {
        return getBoolean("limitDownloadAttempts");
    }

    public String getOpenWithCommands() {
        return get("openWithCommands");
    }
    
    public String[] getOpenWithCommandsArray() {
	return StringHelper.toArray(getOpenWithCommands(), ARRAY_SEPARATOR);
    }

    public LinkedList getOpenWithCommandsList() {
	return StringHelper.toList(getOpenWithCommands(), ARRAY_SEPARATOR);
    }

    public void setOpenWithCommands(String newValue) {
	set("openWithCommands", newValue);
    }

    public void setOpenWithCommands(Collection c) {
	setOpenWithCommands(StringHelper.toString(c, ARRAY_SEPARATOR));
    }

    public void setOpenWithCommands(String[] commands) {
	setOpenWithCommands(StringHelper.toString(commands, ARRAY_SEPARATOR));
    }
    
    public int getPrefsWindowHeight() {
      return getInt("prefsWindowHeight");
    }

    public void setPrefsWindowHeight(int newValue) {
        set("prefsWindowHeight", newValue);
    }

    public int getPrefsWindowWidth() {
      return getInt("prefsWindowWidth");
    }

    public void setPrefsWindowWidth(int newValue) {
        set("prefsWindowWidth", newValue);
    }

    public boolean getRememberFeedback() {
        return getBoolean("rememberFeedback");
    }

    public boolean getRemoveDuplicateResults() {
        return getBoolean("removeDuplicateResults");
    }

    public void setRemoveDuplicateResults(boolean newValue) {
	set("removeDuplicateResults", newValue);
    }

    public int getSearchBitrate() {
        return getInt("searchBitrate");
    }

    public void setSearchBitrate(int newValue) {
	set("searchBitrate", newValue);
    }

    public int getSearchCompare() {
        return getInt("searchCompare");
    }

    public void setSearchCompare(int newValue) {
	set("searchCompare", newValue);
    }

    public int getSearchDividerLocation() {
        return getInt("searchDividerLocation");
    }

    public void setSearchDividerLocation(int newValue) {
	set("searchDividerLocation", newValue);
    }

    public int getSearchHistorySize() {
        return getInt("searchHistorySize");
    }

    public void setSearchHistorySize(int newValue) {
        set("searchHistorySize", newValue);
    }

    public int getSearchMediaType() {
        return getInt("searchMediaType");
    }

    public void setSearchMediaType(int newValue) {
	set("searchMediaType", newValue);
    }

    public String getSearchResultOpenAction() {
        return get("searchResultOpenAction");
    }

    public String getSocksProxyHost() {
        return get("socksProxyHost");
    }

    public void setSocksProxyHost(String newValue) {
	set("socksProxyHost", newValue);
    }

    public int getSocksProxyPort() {
        return getInt("socksProxyPort");
    }

    public void setSocksProxyPort(int newValue) {
	set("socksProxyPort", newValue);
    }

    public String getTheme() {
        return get("theme");
    }

    public void setTheme(String newValue) {
	set("theme", newValue);
    }

    public int getTransferDividerLocation() {
        return getInt("transferDividerLocation");
    }

    public void setTransferDividerLocation(int newValue) {
	set("transferDividerLocation", newValue);
    }

    public String getTransferLogFile() {
	return get("transferLogFile");
    }

    public void setTransferLogFile(String newValue) {
	set("transferLogFile", newValue);
    }

    public int getUpdaterInterval()
    {
	return getInt("updaterInterval");
    }

    public String getUpdateURL() {
	return get("updateURL");
    }
    
    public synchronized String getUploadDirs() {
        return get("uploadDirs");
    }
    
    public String[] getUploadDirsArray() 
    {
	return StringHelper.toArray(getUploadDirs(), ARRAY_SEPARATOR);
    }

    public synchronized void setUploadDirs(String newValue) 
    {
	newValue = FileHelper.directories(newValue);
	set("uploadDirs", newValue);
    }

    public synchronized void setUploadDirs(String[] array) 
    {
	set("uploadDirs", StringHelper.toString(array, ARRAY_SEPARATOR));
    }

    public synchronized void appendToUploadDirs(String path)
    {
	path = (new File(path)).getAbsolutePath();
	if (getUploadDirs().indexOf(path) == -1) {
	    set("uploadDirs", getUploadDirs() + ";" + path);
	}
    }
    
    public synchronized void removeFromUploadDirs(String path)
    {
	path = (new File(path)).getAbsolutePath();
	String ulDirs = getUploadDirs();
	StringBuffer sb = new StringBuffer(ulDirs.length());
	StringTokenizer st = new StringTokenizer(ulDirs, ";");
	while (st.hasMoreTokens()) {
	    String dir = st.nextToken();
	    if (!dir.equals(path)) {
		sb.append(dir);
		sb.append(";");
	    }	    
	}
	setUploadDirs(sb.toString());
    }

    public int getUploadThrottle() {
        return getInt("uploadThrottle");
    }

    public void setUploadThrottle(int newValue) {
	set("uploadThrottle", newValue);
    }
    
    public boolean getUseEmacsKeyBindings() {
        return getBoolean("useEmacsKeyBindings");
    }

    public void setUseEmacsKeyBindings(boolean newValue) {
	set("useEmacsKeyBindings", newValue);
    }

    public String getUsername() {
        return get("username");
    }

    public void setUsername(String newValue) {
	set("username", newValue);
    }

    public boolean getUseHttpProxy() {
        return getBoolean("useHttpProxy");
    }

    public void setUseHttpProxy(boolean newValue) {
	set("useHttpProxy", newValue);
    }

    public boolean getUseSocksProxy() {
        return getBoolean("useSocksProxy");
    }

    public void setUseSocksProxy(boolean newValue) {
	set("useSocksProxy", newValue);
    }

    public boolean getUseTabbedPane() {
        return getBoolean("useTabbedPane");
    }

    public void setUseTabbedPane(boolean newValue) {
	set("useTabbedPane", newValue);
    }

    public boolean getUseTabbedTransferPane() {
        return getBoolean("useTabbedTransferPane");
    }

    public void setUseTabbedTransferPane(boolean newValue) {
	set("useTabbedTransferPane", newValue);
    }

    public String getTableColumns(String table) {
        return get(table + "TableColumns");
    }

    public int[] getTableColumnsArray(String table) 
    {
	return StringHelper.toIntArray(getTableColumns(table), ARRAY_SEPARATOR);
    }

    public void setTableColumns(String table, String newValue) {
	set(table + "TableColumns", newValue);
    }

    public void setTableColumns(String table, int[] newValue) {
	setTableColumns(table, StringHelper.toString(newValue, ARRAY_SEPARATOR));
    }

    public int[] getTableColumnWidths(String table) {
	return StringHelper.toIntArray
	    (get(table + "TableColumnWidths"), ARRAY_SEPARATOR);
    }

    public void setTableColumnWidths(String table, int[] widths) {
	set(table + "TableColumnWidths", 
	    StringHelper.toString(widths, ARRAY_SEPARATOR));
    }

    public boolean getTableMaintainSortOrder(String table) {
        return getBoolean(table + "TableMaintainSortOrder");
    }

    public void setTableMaintainSortOrder(String table, boolean newValue) {
	set(table + "TableMaintainSortOrder", newValue);
    }

    public int getTableSortedColumn(String table) {
        return getInt(table + "TableSortedColumn");
    }

    public void setTableSortedColumn(String table, int newValue) {
	set(table + "TableSortedColumn", newValue);
    }

    public boolean getThrottleDownloads() {
        return getBoolean("throttleDownloads");
    }

    public void setThrottleDownloads(boolean newValue) {
	set("throttleDownloads", newValue);
    }

    public boolean getThrottleUploads() {
        return getBoolean("throttleUploads");
    }

    public void setThrottleUploads(boolean newValue) {
	set("throttleUploads", newValue);
    }

    public int getWindowHeight() {
      return getInt("windowHeight");
    }

    public void setWindowHeight(int newValue) {
        set("windowHeight", newValue);
    }

    public int getWindowWidth() {
      return getInt("windowWidth");
    }

    public void setWindowWidth(int newValue) {
        set("windowWidth", newValue);
    }

    public int getWindowX() {
      return getInt("windowX");
    }

    public void setWindowX(int newValue) {
        set("windowX", newValue);
    }

    public int getWindowY() {
      return getInt("windowY");
    }

    public void setWindowY(int newValue) {
        set("windowY", newValue);
    }

    public String getXNapJarURL() {
	return get("xnapJarURL");
    }

}
