/*
 *  Java Napster version x.yz (for current version number as well as for
 *  additional information see version.txt)
 *
 *  Previous versions of this program were written by Florian Student
 *  and Michael Ransburg available at www.weblicity.de/jnapster and
 *  http://www.tux.org/~daneel/content/projects/10.shtml respectively.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.gui.table;

import xnap.gui.table.*;
import xnap.gui.util.SwingSynchronizedCache;
import xnap.plugin.nap.*;
import xnap.plugin.nap.net.*;
import xnap.plugin.nap.util.NapPreferences;
import xnap.net.event.*;
import xnap.util.*;
import xnap.util.event.*;

import java.beans.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class ServerTableModel extends AbstractDynamicTableModel
    implements ListListener, StatusChangeListener {

    //--- Constant(s) ---

    public static final int HOST = 0;
    public static final int NETWORK = 1;
    public static final int LOCAL_PORT = 2;
    public static final int VERSION = 3;
    public static final int FILE_COUNT = 4;
    public static final int FILE_SIZE = 5;
    public static final int USER_COUNT = 6;
    public static final int STATUS = 7;

    protected Column columns[] = new Column[] {
	new Column(Plugin.tr("Server"),String.class, new StringCellRenderer()),
	new Column(Plugin.tr("Network"), String.class),
	new Column(Plugin.tr("Local Port"), Integer.class),
	new Column(Plugin.tr("Version"),String.class,new StringCellRenderer()),
	new Column(Plugin.tr("Files"),Integer.class, new NumberCellRenderer()),
	new Column(Plugin.tr("Total (GB)"), Integer.class,
		   new NumberCellRenderer()),
	new Column(Plugin.tr("Users"), Integer.class, new NumberCellRenderer()),
	new Column(Plugin.tr("Status"), String.class, new StringCellRenderer()),
    };


    //--- Data Field(s) ---

    private ArrayList rows = new ArrayList();

    //--- Constructor(s) ---

    public ServerTableModel(EventVector data)
    {
	super("server", NapPreferences.getInstance());

	SwingSynchronizedCache cache = new SwingSynchronizedCache(this);
	data.addListListener(cache);

	setColumns(columns);
    }

    // --- Method(s) ---

    public void elementAdded(ListEvent e)
    {
	Server s = (Server)e.getElement();
	s.addStatusChangeListener(this);
	rows.add(s);
	fireTableRowsInserted(rows.size() - 1, rows.size() - 1);
    }

    public void elementRemoved(ListEvent e)
    {
	if (e.getID() == ListEvent.LIST_CLEARED) {
	    int size = rows.size();
	    if (size > 0) {
		for (Iterator i = rows.iterator(); i.hasNext();) {
		    Server s = (Server)i.next();
		    s.removeStatusChangeListener(this);
		}
		rows.clear();
		fireTableRowsDeleted(0, size - 1);
	    }
	}
	else {
	    int i = rows.indexOf(e.getElement());
	    if (i != -1) {
		Server s = (Server)e.getElement();
		s.removeStatusChangeListener(this);
		rows.remove(i);
		fireTableRowsDeleted(i, i);
	    }
	}
    }

    public boolean isCellEditable(int row, int column)
    {
	return false;
    }

    public int getRowCount() 
    { 
	return rows.size(); 
    }

    public Server get(int row)
    {
	return (Server)rows.get(mapToIndex(row));
    }

    public Object get(int row, int col) 
    { 
	Server s = (Server)rows.get(row);

	switch (col) {
	case HOST:
	    return s.getHost() + ":" + s.getPort();
	case NETWORK:
	    return s.getNetwork();
	case LOCAL_PORT:
	    return new Integer(s.getLocalPort());
	case VERSION:
	    return s.getVersion().toString();
	case FILE_COUNT:
	    return new Integer(s.getFileCount());
	case FILE_SIZE:
	    return new Integer(s.getFileSize());
	case USER_COUNT:
	    return new Integer(s.getUserCount());
	case STATUS:
	    return s.getStatusText();
	default:
	    return null;
	}
    }

    public String getTableName()
    {
	return Plugin.tr("Server Table");
    }

    public void statusChange(StatusChangeEvent e)
    {
	SwingUtilities.invokeLater(new StatusChangeEventHandler(e.getSource()));
    }

    private class StatusChangeEventHandler implements Runnable
    {

	Object item;

	public StatusChangeEventHandler(Object item)
	{
	    this.item = item;
	}

	public void run()
	{
	    int i = rows.indexOf(item);
	    if (i != -1 && i < rows.size()) {
		fireTableRowsUpdated(i, i);
	    }
	}

    }

}
