/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.gnutella.util;

import xnap.util.*;

import java.util.*;

/**
 * IPHelper contains some static methods handling ip conversions and
 * comparisons.
 */
public class IPHelper
{
    /**
     * Converts ip given as long to a human readable string.
     */
    public static String longToIPAddress(long longip) 
    {
	StringBuffer sb = new StringBuffer();
	sb.append(longip & 0xFF);
	sb.append('.');
	sb.append((longip >> 8) & 0xFF);
	sb.append('.');
	sb.append((longip >> 16) & 0xFF);
	sb.append('.');
	sb.append((longip >> 24) & 0xFF);
	return sb.toString();
    }
    
    /**
     * Converts ip given as byte array to a human readable string.
     */
    public static String bytearrayToIPAddressString(byte[] ip) 
    {
	if (ip.length != 4) {
	    return null;
	}
	
	StringBuffer sb = new StringBuffer();
	
	sb.append(ip[0] & 0xFF);
	sb.append('.');
	sb.append(ip[1] & 0xFF);
	sb.append('.');
	sb.append(ip[2] & 0xFF);
	sb.append('.');
	sb.append(ip[3] & 0xFF);
	
	return sb.toString();
    }
    
    /**
     * Converts ip given as String to a byte array.
     */
    public static byte[] ipStringToBytearray(String ipString) 
    {
	byte[] ip = new byte[4];
	StringTokenizer st = new StringTokenizer(ipString, ".");
	
	if (st.countTokens() != 4) {
	    return null;
	}
	
	try {
	    for (int i = 0; i < 4; i++) {
		ip[i] = (byte) Integer.parseInt(st.nextToken());
	    }
	}
	catch(NumberFormatException e) {
	    return null;
	}
	
	return ip;
    }
    
    /**
     * Converts ip given as String to an int array.
     */
    public static int[] ipStringToIntarray(String ipString) 
    {
	int[] ip = new int[4];
	StringTokenizer st = new StringTokenizer(ipString, ".");
	
	if (st.countTokens() != 4) {
	    return null;
	}
	
	try {
	    for (int i = 0; i < 4; i++) {
		ip[i] = Integer.parseInt(st.nextToken());
	    }
	}
	catch(NumberFormatException e) {
	    return null;
	}
	
	return ip;
    }

    /**
     * Checks if given ip belongs to a firewalled host.
     */
    public static boolean isPrivateIP(String ip)
    {
	return isPrivateIP(ipStringToBytearray(ip));
    }

    public static boolean isPrivateIP(byte[] ip)
    {

	if (ip == null)
	    return true;
	
	if (ip[0] == (byte) 10 || ip[0] == (byte) 127)
	    return true;

	if (ip[0] == (byte) 192 && ip[1] == (byte) 168)
	    return true;

	if (ip[0] == (byte) 172 && ip[1] > (byte) 15 && ip[1] < (byte) 32)
	    return true;
	
	if (ip[0] == (byte) 255 && ip[1] == (byte) 255 && ip[2] == (byte) 255)
	    return true;

	if (ip[0] == (byte) 0 && ip[1] == (byte) 0 && ip[2] == (byte) 0
	    && ip[3] == (byte) 0)
	    return true;

	return false;
    }
}
