/*
   Copyright (C) 1997-2001 Id Software, Inc.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

 */

#include "tvm_local.h"

#include "tvm_misc.h"

//=============
//TVM_FindLocal
//
//Searches all active entities for the next one that holds
//the matching string at fieldofs (use the FOFS() macro) in the structure.
//
//Searches beginning at the edict after from, or the beginning if NULL
//NULL will be returned if the end of the list is reached.
//=============
edict_t *TVM_FindLocal( tvm_relay_t *relay, const edict_t *start, size_t fieldofs, const char *match )
{
	char *s;
	edict_t	*from;

	if( !start )
		from = &relay->local_edicts[0];
	else
		from = &relay->local_edicts[ENTNUM( start ) + 1];

	for(; from < &relay->local_edicts[relay->local_numentities]; from++ )
	{
		if( !from->r.inuse )
			continue;
		s = *(char **) ( (qbyte *)from + fieldofs );
		if( !s )
			continue;
		if( !Q_stricmp( s, match ) )
			return from;
	}

	return NULL;
}

//=================
//TVM_AllowDownload
//=================
qboolean TVM_AllowDownload( tvm_relay_t *relay, edict_t *ent, const char *requestname, const char *uploadname )
{
	return qfalse;
}

//=================
//TVM_UpdateServerSettings
//=================
static void TVM_UpdateServerSettings( tvm_relay_t *relay )
{
	char *settings;

	settings = relay->configStrings[CS_SERVERSETTINGS];
	relay->tv = ( atoi( COM_Parse( &settings ) ) != 0 );
	relay->has_challengers = ( atoi( COM_Parse( &settings ) ) != 0 );
	relay->maxteams = atoi( COM_Parse( &settings ) );
	relay->gametype = atoi( COM_Parse( &settings ) );
	relay->instagib = ( atoi( COM_Parse( &settings ) ) != 0 );
	relay->falldamage = ( atoi( COM_Parse( &settings ) ) != 0 );

	relay->configStringsOverwritten[CS_SERVERSETTINGS] = qtrue;
	trap_ConfigString( relay, CS_SERVERSETTINGS,
	                  va( "1 %i %i %i %i %i", relay->has_challengers, relay->maxteams, relay->gametype, relay->instagib, relay->falldamage ) );
}

//============
//TVM_PrintMsg_Template
//
//NULL sends to all the message to all clients
//============
static void TVM_PrintMsg_Template( tvm_relay_t *relay, edict_t *ent, const char *tmplt, char *msg )
{
	char *p;

	assert( !ent || ( ent->local && ent->r.client ) );
	assert( msg );

	// double quotes are bad
	while( ( p = strchr( msg, '\"' ) ) != NULL )
		*p = '\'';

	trap_GameCmd( relay, ent ? PLAYERNUM( ent ) : -1, va( tmplt, msg ) );
}

//============
//TVM_PrintMsg
//
//NULL sends to all the message to all clients
//============
void TVM_PrintMsg( tvm_relay_t *relay, edict_t *ent, const char *format, ... )
{
	char msg[1024];
	va_list	argptr;

	assert( !ent || ( ent->local && ent->r.client ) );
	assert( format );

	va_start( argptr, format );
	Q_vsnprintfz( msg, sizeof( msg ), format, argptr );
	va_end( argptr );

	TVM_PrintMsg_Template( relay, ent, "pr \"%s\"", msg );
}

//============
//TVM_CenterPrintMsg
//
//NULL sends to all the message to all clients
//============
// MOVEME
void TVM_CenterPrintMsg( tvm_relay_t *relay, edict_t *ent, const char *format, ... )
{
	char msg[1024];
	va_list	argptr;

	assert( !ent || ( ent->local && ent->r.client ) );
	assert( format );

	va_start( argptr, format );
	Q_vsnprintfz( msg, sizeof( msg ), format, argptr );
	va_end( argptr );

	TVM_PrintMsg_Template( relay, ent, "cp \"%s\"", msg );
}

//=================
//TVM_ConfigString
//=================
qboolean TVM_ConfigString( tvm_relay_t *relay, int number, const char *value )
{
	assert( number >= 0 && number < MAX_CONFIGSTRINGS );
	assert( value && strlen( value ) < MAX_CONFIGSTRING_CHARS );

	if( number < 0 || number >= MAX_CONFIGSTRINGS )
		TVM_RelayError( relay, "TVM_ConfigString: Invalid number" );

	Q_strncpyz( relay->configStrings[number], value, sizeof( relay->configStrings[number] ) );

	if( number == CS_SERVERSETTINGS )
		TVM_UpdateServerSettings( relay );

	return ( !relay->configStringsOverwritten[number] );
}
