/*  smplayer, GUI front-end for mplayer.
    Copyright (C) 2007 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "core.h"
#include <qdir.h>
#include <qfileinfo.h>
#include <qregexp.h>

#include <cmath>

#include "mplayerprocess.h"
#include "mplayerwindow.h"
#include "desktopinfo.h"
#include "constants.h"
#include "helper.h"
#include "keys.h"
#include "global.h"

Core::Core( MplayerWindow *mpw, QWidget* parent, const char* name ) 
	: QObject( parent, name ) 
{
	mplayerwindow = mpw;

	we_are_restarting = FALSE;
	state = STOPPED;

    proc = new MplayerProcess(this);
    //proc->setCommunication( Q3Process::Stdin | Q3Process::Stdout | Q3Process::Stderr );
    
	connect( proc, SIGNAL(receivedCurrentSec(double)),
             this, SLOT(changeCurrentSec(double)) );

	connect( proc, SIGNAL(receivedCurrentFrame(int)),
             this, SLOT(changeCurrentFrame(int)) );

	connect( proc, SIGNAL(receivedPause()),
			 this, SLOT(changePause()) );

    connect( proc, SIGNAL(processExited()),
	         this, SLOT(processFinished()) );

	connect( proc, SIGNAL(mplayerFullyLoaded()),
			 this, SLOT(finishRestart()) );

	connect( proc, SIGNAL(lineAvailable(QString)),
             this, SLOT(updateLog(QString)) );

	connect( proc, SIGNAL(receivedCacheMessage(QString)),
			 this, SLOT(displayMessage(QString)) );

	connect( proc, SIGNAL(receivedCreatingIndex(QString)),
			 this, SLOT(displayMessage(QString)) );

	connect( proc, SIGNAL(receivedConnectingToMessage(QString)),
			 this, SLOT(displayMessage(QString)) );

	connect( proc, SIGNAL(receivedResolvingMessage(QString)),
			 this, SLOT(displayMessage(QString)) );

	connect( proc, SIGNAL(receivedScreenshot(QString)),
             this, SLOT(displayScreenshotName(QString)) );
	
	connect( proc, SIGNAL(receivedWindowResolution(int,int)),
             this, SLOT(gotWindowResolution(int,int)) );

	connect( proc, SIGNAL(receivedNoVideo()),
             this, SLOT(gotNoVideo()) );

	connect( proc, SIGNAL(receivedVO(QString)),
             this, SLOT(gotVO(QString)) );

	connect( proc, SIGNAL(receivedAO(QString)),
             this, SLOT(gotAO(QString)) );

	connect( proc, SIGNAL(receivedEndOfFile()),
             this, SLOT(fileReachedEnd()) );

	//pref.load();
	mset.reset(pref.initial_volume);

	mplayerwindow->setMonitorAspect( pref.monitor_aspect_double() );
}


Core::~Core() {
	saveMediaInfo();

    if (proc->isRunning()) stopMplayer();
    proc->tryTerminate();
    delete proc;
}


// Public restart
void Core::restart() {
	qDebug("Core::restart");
	if (proc->isRunning()) {
		restartPlay();
	} else {
		qDebug("Core::restart: mplayer is not running");
	}
}

bool Core::checkHaveSettingsSaved(QString group_name) {
	qDebug("Core::checkHaveSettingsSaved: group_name: '%s'", group_name.utf8().data());

	settings->beginGroup( group_name );
	bool saved = settings->readBoolEntry( "saved", FALSE );
	settings->endGroup();

	return saved;
}

void Core::saveMediaInfo() {
	qDebug("Core::saveMediaInfo");

	QString group_name;

	/*
	if ( (mdat.type == TYPE_DVD) && (!mdat.dvd_id.isEmpty()) ) {
		group_name = dvdForPref( mdat.dvd_id, mset.current_title_id );
	}
	else
	*/
	if ( (mdat.type == TYPE_FILE) && (!mdat.filename.isEmpty()) ) {
		group_name = Helper::filenameForPref( mdat.filename );
	}

	if (!group_name.isEmpty()) {
		settings->beginGroup( group_name );
		settings->writeEntry( "saved", TRUE);

		/*mdat.save(*settings);*/
		mset.save();

		settings->endGroup();
	}
}

void Core::loadMediaInfo(QString group_name) {
	qDebug("Core::loadMediaInfo: '%s'", group_name.utf8().data() );

	settings->beginGroup( group_name );

	/*mdat.load(*settings);*/
	mset.load();

	settings->endGroup();
}


void Core::updateLog(QString line) {
	if ( (line.find("A:")==-1) && (line.find("V:")==-1) ) {
		mplayer_log += line + "\n";
	}
}

void Core::initializeMenus() {
	qDebug("Core::initializeMenus");

	emit menusNeedInitialize();
}


void Core::updateWidgets() {
	qDebug("Core::updateWidgets");

	emit widgetsNeedUpdate();
}


void Core::tellmp(const QString & command) {
	qDebug("Core::tellmp: '%s'", command.utf8().data());

    //qDebug("Command: '%s'", command.utf8().data());
    if (proc->isRunning()) {
		proc->writeToStdin( command + "\n" );
    } else {
		qWarning(" tellmp: no process running: %s", command.utf8().data());
    }
}

// Generic open, autodetect type
void Core::open(QString file, int seek) {
	qDebug("Core::open: '%s'", file.utf8().data());

	QFileInfo fi(file);

	if ( (fi.exists()) && (!fi.isDir()) ) {
		qDebug(" * identified as local file");
		// Local file
		file = QFileInfo(file).absFilePath();
		openFile(file, seek);
	} 
	else
	if ( (fi.exists()) && (fi.isDir()) ) {
		// Directory
		qDebug(" * identified as a directory");
		qDebug("   checking if contains a dvd");
		file = QFileInfo(file).absFilePath();
		if (Helper::directoryContainsDVD(file)) {
			qDebug(" * directory contains a dvd");
			openDVD("dvd://1:"+ file);
		} else {
			qDebug(" * directory doesn't contain a dvd");
			qDebug("   opening nothing");
		}
	}
	else 
	if (file.lower().startsWith("dvd:")) {
		qDebug(" * identified as dvd");
		openDVD(file);
		/*
		QString f = file.lower();
		QRegExp s("^dvd://(\\d+)");
		if (s.search(f) != -1) {
			int title = s.cap(1).toInt();
			openDVD(title);
		} else {
			qWarning("Core::open: couldn't parse dvd title, playing first one");
			openDVD();
		}
		*/
	}
	else {
		qDebug(" * no identified, playing as stream");
		openStream(file);
	}
}

void Core::openFile(QString filename, int seek) {
	qDebug("Core::openFile: '%s'", filename.utf8().data());

	QFileInfo fi(filename);
	if (fi.exists()) {
		playNewFile(fi.absFilePath(), seek);
	} else {
		//File doesn't exists
		//TODO: error message
	}
}


void Core::loadSub(const QString & sub ) {
    if ( !sub.isEmpty() ) {
		//tellmp( "sub_load " + sub );
		mset.external_subtitles = sub;
		restartPlay();
	}
}

/*
void Core::openDVD( bool from_folder, QString directory) {
	qDebug("Core::openDVD");

	if (from_folder) {
		if (!directory.isEmpty()) {
			QFileInfo fi(directory);
			if ( (fi.exists()) && (fi.isDir()) ) {
				pref.dvd_directory = directory;
				pref.play_dvd_from_hd = TRUE;
				openDVD();
			} else {
				qDebug("Core::openDVD: directory '%s' is not valid", directory.utf8().data());
			}
		} else {
			qDebug("Core::openDVD: directory is empty");
		}
	} else {
		pref.play_dvd_from_hd = FALSE;
		openDVD();
	}
}

void Core::openDVD() {
	openDVD(1);
}

void Core::openDVD(int title) {
	qDebug("Core::openDVD: %d", title);

	if (proc->isRunning()) {
		stopMplayer();
	}

	// Save data of previous file:
	saveMediaInfo();

	mdat.reset();
	mdat.filename = "dvd://" + QString::number(title);
	mdat.type = TYPE_DVD;

	mset.reset(pref.initial_volume);

	mset.current_title_id = title;
	mset.current_chapter_id = 1;
	mset.current_angle_id = 1;

	initializeMenus();

	initPlaying();
}
*/

void Core::openDVD(QString dvd_url) {
	qDebug("Core::openDVD: '%s'", dvd_url.utf8().data());

	//Checks
	QString folder = Helper::dvdSplitFolder(dvd_url);
	int title = Helper::dvdSplitTitle(dvd_url);

	if (title == -1) {
		qWarning("Core::openDVD: title invalid, not playing dvd");
		return;
	}

	if (folder.isEmpty()) {
		qDebug("Core::openDVD: not folder");
	} else {
		QFileInfo fi(folder);
		if ( (!fi.exists()) || (!fi.isDir()) ) {
			qWarning("Core::openDVD: folder invalid, not playing dvd");
			return;
		}
	}

	if (proc->isRunning()) {
		stopMplayer();
	}

	// Save data of previous file:
	saveMediaInfo();

	mdat.reset();
	mdat.filename = dvd_url;
	mdat.type = TYPE_DVD;

	mset.reset(pref.initial_volume);

	mset.current_title_id = title;
	mset.current_chapter_id = 1;
	mset.current_angle_id = 1;

	/* initializeMenus(); */

	initPlaying();
}

void Core::openStream(QString name) {
	qDebug("Core::openStream: '%s'", name.utf8().data());

	if (proc->isRunning()) {
		stopMplayer();
	}

	// Save data of previous file:
	saveMediaInfo();

	mdat.reset();
	mdat.filename = name;
	mdat.type = TYPE_STREAM;

	mset.reset(pref.initial_volume);

	/* initializeMenus(); */

	initPlaying();
}


void Core::playNewFile(QString file, int seek) {
	qDebug("Core::playNewFile: '%s'", file.utf8().data());

	if (proc->isRunning()) {
		stopMplayer();
	}

	// Save data of previous file:
	saveMediaInfo();

	mdat.reset();
	mdat.filename = file;
	mdat.type = TYPE_FILE;

	int old_volume = mset.volume;
	mset.reset(pref.initial_volume);

	// Check if we already have info about this file
	if (checkHaveSettingsSaved( Helper::filenameForPref(file) )) {
		qDebug("We have settings for this file!!!");

		// In this case we read info from config
		if (!pref.dont_remember_media_settings) {
			loadMediaInfo( Helper::filenameForPref(file) );
			qDebug("Media settings read");
			if (pref.dont_remember_time_pos) {
				mset.current_sec = 0;
				qDebug("Time pos reset to 0");
			}
		} else {
			qDebug("Media settings have not read because of preferences setting");
		}
	} else {
		// Recover volume
		mset.volume = old_volume;
	}

	/* initializeMenus(); */

	initPlaying(seek);
}


void Core::restartPlay() {
	we_are_restarting = true;
	initPlaying();
}

void Core::initPlaying(int seek) {
	qDebug("Core::initPlaying");

	/*
	mdat.list();
	mset.list();
	*/

	/* updateWidgets(); */

	mplayerwindow->showLogo(FALSE);

	if (proc->isRunning()) {
		stopMplayer();
	}

	int start_sec = (int) mset.current_sec;
	if (seek > -1) start_sec = seek;

	startMplayer( mdat.filename, start_sec );
}

// This is reached when a new video has just started playing
// and maybe we need to give some defaults
void Core::newMediaPlaying() {
	QString file = mdat.filename;
	int type = mdat.type;
	mdat = proc->mediaData();
	mdat.filename = file;
	mdat.type = type;

	initializeMenus(); // Old

	// First audio if none selected
	if ( (mset.current_audio_id == NONE_SELECTED) && (mdat.audios.numItems() > 0) ) {
		// Don't set mset.current_audio_id here! changeAudio will do. 
		// Otherwise changeAudio will do nothing.
		int audio = mdat.audios.itemAt(0).ID();
		changeAudio( audio );
	}

	// Subtitles
	if (mset.external_subtitles.isEmpty()) {
		if (pref.autoload_sub) {
			//Select first subtitle if none selected
			if (mset.current_sub_id == NONE_SELECTED) {
				int sub = SUB_ID_NONE; // In case of no subtitle available
				if (mdat.subtitles.numItems() > 0) {
					sub = mdat.subtitles.itemAt(0).ID();
				} 
				changeSubtitle( sub );
			}
		} else {
			changeSubtitle( SUB_ID_NONE );
		}
	}

	// mkv chapters
	if (mdat.mkv_chapters > 0) {
		// Just to show the first chapter checked in the menu
		mset.current_chapter_id = 0;  // 0 is the first chapter in mkv
	}

	mdat.initialized = TRUE;

	/* updateWidgets(); */

	mdat.list();
	mset.list();
}

void Core::finishRestart() {
	qDebug("Core::finishRestart");

	if (!we_are_restarting) {
		newMediaPlaying();
	} 

	we_are_restarting = false;

	if (mset.external_subtitles.isEmpty()) {
		changeSubtitle( mset.current_sub_id );
	}

	if (mset.aspect_ratio_id < ASPECT_43_LETTERBOX) {
		changeAspectRatio(mset.aspect_ratio_id);
	}

	bool isMuted = mset.mute;
	setVolume( mset.volume, TRUE );
	if (isMuted) mute(TRUE);

	setGamma( mset.gamma );

	updateWidgets(); // New

	emit mediaLoaded();
}


void Core::stop()
{
	qDebug("Core::stop");
	qDebug("   state: %d", state);
	
	if (state==STOPPED) {
		// if pressed stop twice, reset video to the beginning
		qDebug("   mset.current_sec: %f", mset.current_sec);
		mset.current_sec = 0;
		updateWidgets();
	}

	stopMplayer();
	emit mediaStoppedByUser();
}


void Core::play()
{
	qDebug("Core::play");
    
	if ((proc->isRunning()) && (state==PAUSED)) {
		tellmp("pause"); // Unpauses
    } 
	else
	if ((proc->isRunning()) && (state==PLAYING)) {
		// nothing to do, continue playing
	}
	else {
		// if we're stopped, play it again
		if ( !mdat.filename.isEmpty() ) {
			/*
			qDebug( "current_sec: %f, duration: %f", mset.current_sec, mdat.duration);
			if ( (floor(mset.current_sec)) >= (floor(mdat.duration)) ) {
				mset.current_sec = 0;
			}
			*/
			restartPlay();
		}
    }
}

void Core::pause_and_frame_step() {
	qDebug("Core::pause_and_frame_step");
	
	if (proc->isRunning()) {
		if (state == PAUSED) {
			tellmp("frame_step");
		}
		else {
			tellmp("pause");
		}
	}
}

void Core::pause() {
	qDebug("Core::pause");
	qDebug("Current state: %d", state);

	if (proc->isRunning()) {
		// Pauses and unpauses
		tellmp("pause");
	}
}

void Core::play_or_pause() {
	if (proc->isRunning()) {
		pause();
	} else {
		play();
	}
}

void Core::frameStep() {
	qDebug("Core::franeStep");

	if (proc->isRunning()) {
		tellmp("frame_step");
	}
}

void Core::screenshot() {
	qDebug("Core::screenshot");

	if ( (!pref.screenshot_directory.isEmpty()) && 
         (QFileInfo(pref.screenshot_directory).isDir()) ) 
	{
		tellmp("screenshot 0");
		qDebug(" taken screenshot");
	} else {
		qDebug(" error: directory for screenshots not valid");
	}
}

void Core::processFinished()
{
    qDebug("Core::processFinished");

	// Enable screensaver (in windows)
	if (pref.disable_screensaver) {
		Helper::setScreensaverEnabled(TRUE);
	}

	qDebug("Core::processFinished: we_are_restarting: %d", we_are_restarting);

	//mset.current_sec = 0;

	if (!we_are_restarting) {
		qDebug("Core::processFinished: play has finished!");
		state = STOPPED;
		emit stateChanged(state);
	}

	int exit_status = proc->exitStatus();
	qDebug(" exit_status: %d", exit_status);
	if (exit_status != 0) {
		emit mplayerFinishedWithError(exit_status);
	}
}

void Core::fileReachedEnd() {
	// If we're at the end of the movie, reset to 0
	mset.current_sec = 0;
	updateWidgets();

	emit mediaFinished();
}

void Core::goToPos(int perc)
{
    qDebug("Core::goToPos: per: %d", perc);

    tellmp ( "seek " + QString::number( perc) + " 1");
}



void Core::startMplayer( QString file, double seek )
{
	qDebug("Core::startMplayer");

	if (proc->isRunning()) {
		qWarning("Core::startMplayer: MPlayer still running!");
		return;
    } 

	// Disable screensaver (in windows)
	if (pref.disable_screensaver) {
		Helper::setScreensaverEnabled(FALSE);
	}

	mplayer_log = "";
	bool is_mkv = (QFileInfo(file).extension(FALSE).lower() == "mkv");

	// DVD
	QString dvd_folder;
	int dvd_title = -1;
	if (mdat.type==TYPE_DVD) {
		dvd_folder = Helper::dvdSplitFolder(file);
		if (dvd_folder.isEmpty()) dvd_folder = pref.dvd_device;
		if (dvd_folder.endsWith("/")) dvd_folder = dvd_folder.remove( dvd_folder.length()-1,1);
		dvd_title = Helper::dvdSplitTitle(file);
		file = "dvd://" + QString::number(dvd_title);
	}


	proc->clearArguments();

	// Set working directory to screenshot directory
	if ( (!pref.screenshot_directory.isEmpty()) && 
         (QFileInfo(pref.screenshot_directory).isDir()) ) 
	{
		qDebug("Core::startMplayer: setting working directory to '%s'", pref.screenshot_directory.utf8().data());
		//proc->setWorkingDirectory( QDir( pref.screenshot_directory ) );
		proc->setWorkingDirectory( pref.screenshot_directory );
	}


	proc->addArgument( pref.mplayer_bin );

	proc->addArgument("-noquiet");

	// Demuxer and audio and video codecs:
	if (!mset.forced_demuxer.isEmpty()) {
		proc->addArgument("-demuxer");
		proc->addArgument(mset.forced_demuxer);
	}
	if (!mset.forced_audio_codec.isEmpty()) {
		proc->addArgument("-ac");
		proc->addArgument(mset.forced_audio_codec);
	}
	if (!mset.forced_video_codec.isEmpty()) {
		proc->addArgument("-vc");
		proc->addArgument(mset.forced_video_codec);
	}

	proc->addArgument("-sub-fuzziness");
	if (mset.external_subtitles.isEmpty()) {
		proc->addArgument( QString::number(pref.subfuzziness) );
	} else {
		proc->addArgument("0");
	}

	/*
	if (!pref.mplayer_verbose.isEmpty()) {
		proc->addArgument("-msglevel");
		proc->addArgument( pref.mplayer_verbose );
	}
	*/
	
	proc->addArgument("-identify");

	// We need this to get info about mkv chapters
	if (is_mkv) {
		proc->addArgument("-msglevel");
		proc->addArgument("demux=6");

		// **** Reset chapter *** 
		// Select first chapter, otherwise we cannot
		// resume playback at the same point
		// (time would be relative to chapter
		mset.current_chapter_id = 0;
	}
	
	proc->addArgument("-slave");

	if (!pref.vo.isEmpty()) {
		proc->addArgument( "-vo");
		proc->addArgument( pref.vo );
	}

	if (!pref.ao.isEmpty()) {
		proc->addArgument( "-ao");
		proc->addArgument( pref.ao );
	}

	proc->addArgument( "-zoom");
	proc->addArgument("-nokeepaspect");

	// Performance options
	#ifdef Q_OS_WIN
	proc->addArgument("-priority");
	QString p;
	switch (pref.priority) {
		case PRIORITY_REALTIME: p = "realtime"; break;
		case PRIORITY_HIGH: p = "high"; break;
		case PRIORITY_ABOVENORMAL: p = "abovenormal"; break;
		case PRIORITY_BELOWNORMAL: p = "belownormal"; break;
		case PRIORITY_IDLE: p = "idle"; break;
		default: p = "normal";
	}
	proc->addArgument( p );
	#endif

	if (pref.frame_drop) {
		proc->addArgument("-framedrop");
	}

	if (pref.hard_frame_drop) {
		proc->addArgument("-hardframedrop");
	}

	if (pref.autosync) {
		proc->addArgument("-autosync");
		proc->addArgument( QString::number( pref.autosync_factor ) );
	}


#ifndef Q_OS_WIN
	if (!pref.use_mplayer_window) {
		proc->addArgument( "-input" );
		proc->addArgument( "conf=" + Helper::dataPath() +"/input.conf" );
	}
#endif

#ifndef Q_OS_WIN
	if (pref.disable_screensaver) {
		proc->addArgument("-stop-xscreensaver");
	}
#endif

	if (!pref.use_mplayer_window) {
		proc->addArgument("-wid");
		proc->addArgument( QString::number( (int) mplayerwindow->mplayerLayer()->winId() ) );
	
		proc->addArgument("-colorkey");
		proc->addArgument( QString::number(COLORKEY) );

		// Set monitoraspect to desktop aspect
		proc->addArgument("-monitoraspect");
		proc->addArgument( QString::number( desktop_aspectRatio() ) );
	} else {
		// no -wid
		if (!pref.monitor_aspect.isEmpty()) {
			proc->addArgument("-monitoraspect");
			proc->addArgument( pref.monitor_aspect );
		}
	}

	if (pref.use_ass_subtitles) {
		proc->addArgument("-ass");
		proc->addArgument("-embeddedfonts");
		proc->addArgument("-ass-color");
		proc->addArgument( Helper::colorToRGBA( pref.ass_color ) );
		proc->addArgument("-ass-border-color");
		proc->addArgument( Helper::colorToRGBA( pref.ass_border_color ) );
	}

	// Subtitles font
	if ( (pref.use_fontconfig) && (!pref.font_name.isEmpty()) ) {
		proc->addArgument("-fontconfig");
		proc->addArgument("-font");
		proc->addArgument( pref.font_name );
	}

	if ( (!pref.use_fontconfig) && (!pref.font_file.isEmpty()) ) {
		proc->addArgument("-font");
		proc->addArgument( pref.font_file );
	}

	proc->addArgument( "-subfont-autoscale");
	proc->addArgument( QString::number( pref.font_autoscale ) );
	proc->addArgument( "-subfont-text-scale");
	proc->addArgument( QString::number( pref.font_textscale ) );

	if (!pref.subcp.isEmpty()) {
		proc->addArgument("-subcp");
		proc->addArgument( pref.subcp );
	}

	if (mset.current_audio_id != NONE_SELECTED) {
		proc->addArgument("-aid");
		proc->addArgument( QString::number( mset.current_audio_id ) );
	}

	if (!mset.external_subtitles.isEmpty()) {
		if (QFileInfo(mset.external_subtitles).extension(FALSE).lower()=="idx") {
			// sub/idx subtitles
			QFileInfo fi(mset.external_subtitles);
			QString s = fi.dirPath() +"/"+ fi.baseName(FALSE);
			qDebug(" * subtitle file without extension: '%s'", s.utf8().data());
			proc->addArgument("-vobsub");
			proc->addArgument(s);
		} else {
			proc->addArgument("-sub");
			proc->addArgument( mset.external_subtitles );
		}
	}

	proc->addArgument("-subpos");
	proc->addArgument( QString::number(mset.sub_pos) );

	if (mset.audio_delay!=0) {
		proc->addArgument("-delay");
		proc->addArgument( QString::number( (double) mset.audio_delay/1000 ) );
	}

	if (mset.sub_delay!=0) {
		proc->addArgument("-subdelay");
		proc->addArgument( QString::number( (double) mset.sub_delay/1000 ) );
	}

	// Contrast, brightness...
	//if (mset.contrast !=0) {
		proc->addArgument("-contrast");
		proc->addArgument( QString::number( mset.contrast ) );
	//}
	
	#ifdef Q_OS_WIN
	if (mset.brightness !=0) {
	#endif
		proc->addArgument("-brightness");
		proc->addArgument( QString::number( mset.brightness ) );
	#ifdef Q_OS_WIN
	}
	#endif

	//if (mset.hue !=0) {
		proc->addArgument("-hue");
		proc->addArgument( QString::number( mset.hue ) );
	//}

	//if (mset.saturation !=0) {
		proc->addArgument("-saturation");
		proc->addArgument( QString::number( mset.saturation ) );
	//}


	/*
	if (mdat.type==TYPE_DVD) {
		if ( (pref.play_dvd_from_hd) && (!pref.dvd_directory.isEmpty()) ) {
			proc->addArgument("-dvd-device");
			proc->addArgument( pref.dvd_directory );
		} else {
			if (!pref.dvd_device.isEmpty()) {
				proc->addArgument("-dvd-device");
				proc->addArgument( pref.dvd_device );
			}
		}
	}
	*/

	if (mdat.type==TYPE_DVD) {
		if (!dvd_folder.isEmpty()) {
			proc->addArgument("-dvd-device");
			proc->addArgument( dvd_folder );
		} else {
			qWarning("Core::startMplayer: dvd device is empty!");
		}
	}

	if (mset.current_chapter_id > 0) {
		proc->addArgument("-chapter");
		proc->addArgument( QString::number( mset.current_chapter_id ) );
	}

	if (mset.current_angle_id > 0) {
		proc->addArgument("-dvdangle");
		proc->addArgument( QString::number( mset.current_angle_id ) );
	}


	if ( (pref.cache > 0) && ((!mdat.type==TYPE_DVD) || (!pref.fast_chapter_change)) ) {
		proc->addArgument("-cache");
		proc->addArgument( QString::number( pref.cache ) );
	}

	if (mset.speed != 1.0) {
		proc->addArgument("-speed");
		proc->addArgument( QString::number( mset.speed ) );
	}

	// If seek < 5 it's best to allow the video to start from the beginning
	if (seek >= 5) {
		proc->addArgument("-ss");
		proc->addArgument( QString::number( seek ) );
	}

	proc->addArgument("-osdlevel");
	proc->addArgument( QString::number( pref.osd ) );

	// Video filters:
	// Phase
	if (mset.phase_filter) {
		proc->addArgument("-vf-add");
		proc->addArgument( "phase=A" );
	}

	// Deinterlace
	if (mset.current_deinterlacer != DEINTERLACE_NONE) {
		proc->addArgument("-vf-add");
		switch (mset.current_deinterlacer) {
			case DEINTERLACE_L5: 		proc->addArgument("pp=l5"); break;
			case DEINTERLACE_YADIF: 	proc->addArgument("yadif"); break;
			case DEINTERLACE_LB:		proc->addArgument("pp=lb"); break;
		}
	}

	// Panscan (crop)
	if (!mset.panscan_filter.isEmpty()) {
		proc->addArgument( "-vf-add" );
		proc->addArgument( mset.panscan_filter );
	}

	// Crop 4:3 to 16:9
	if (!mset.crop_43to169_filter.isEmpty()) {
		proc->addArgument( "-vf-add" );
		proc->addArgument( mset.crop_43to169_filter );
	}

	// Denoise
	if (mset.current_denoiser != DENOISE_NONE) {
		proc->addArgument("-vf-add");
		if (mset.current_denoiser==DENOISE_SOFT) {
			proc->addArgument( "hqdn3d=2:1:2" );
		} else {
			proc->addArgument( "hqdn3d" );
		}
	}

	// Deblock
	if (mset.deblock_filter) {
		proc->addArgument("-vf-add");
		proc->addArgument( "pp=vb/hb" );
	}

	// Dering
	if (mset.dering_filter) {
		proc->addArgument("-vf-add");
		proc->addArgument( "pp=dr" );
	}

	// Addnoise
	if (mset.noise_filter) {
		proc->addArgument("-vf-add");
		proc->addArgument( "noise=9ah:5ah" );
	}

	// Postprocessing
	if (mset.postprocessing_filter) {
		proc->addArgument("-vf-add");
		proc->addArgument("pp");
		proc->addArgument("-autoq");
		proc->addArgument( QString::number(pref.autoq) );
	}


	// Letterbox (expand)
	if (mset.letterbox == MediaSettings::Letterbox_43) {		
		proc->addArgument("-vf-add");
		proc->addArgument("expand=:::::4/3");
	}
	else
	if (mset.letterbox == MediaSettings::Letterbox_169) {
		proc->addArgument("-vf-add");
		proc->addArgument("expand=:::::16/9");
	}

	// Additional video filters, supplied by user
	// File
	if ( !mset.mplayer_additional_video_filters.isEmpty() ) {
		proc->addArgument("-vf-add");
		proc->addArgument( mset.mplayer_additional_video_filters );
	}
	// Global
	if ( !pref.mplayer_additional_video_filters.isEmpty() ) {
		proc->addArgument("-vf-add");
		proc->addArgument( pref.mplayer_additional_video_filters );
	}

	// Screenshot
	if (!pref.screenshot_directory.isEmpty()) {
		// Subtitles on screenshots
		if (pref.subtitles_on_screenshots) {
			if (pref.use_ass_subtitles) {
				proc->addArgument("-vf-add");
				proc->addArgument("ass");
			} else {
				proc->addArgument("-vf-add");
				proc->addArgument("expand=osd=1");
				proc->addArgument("-noslices");
			}
		}
		proc->addArgument("-vf-add");
		proc->addArgument("screenshot");
	}

	if ( (pref.use_soft_video_eq) && (pref.vo!="gl") && (pref.vo!="gl2") ) {
		proc->addArgument("-vf-add");
		proc->addArgument("eq2");
	}

	// Audio channels
	if (mset.audio_use_channels != 0) {
		proc->addArgument("-channels");
		proc->addArgument( QString::number( mset.audio_use_channels ) );
	}

	// Stereo mode
	if (mset.stereo_mode != 0) {
		proc->addArgument("-stereo");
		proc->addArgument( QString::number( mset.stereo_mode ) );
	}

	// Audio filters
	QString af="";
	if (mset.karaoke_filter) {
		af="karaoke";
	}

	if (mset.extrastereo_filter) {
		if (!af.isEmpty()) af += ",";
		af += "extrastereo";
	}

	// Additional audio filters, supplied by user
	// File
	if ( !pref.mplayer_additional_audio_filters.isEmpty() ) {
		if (!af.isEmpty()) af += ",";
		af += pref.mplayer_additional_audio_filters;
	}
	// Global
	if ( !mset.mplayer_additional_audio_filters.isEmpty() ) {
		if (!af.isEmpty()) af += ",";
		af += mset.mplayer_additional_audio_filters;
	}

	if (!af.isEmpty()) {
		proc->addArgument("-af");
		proc->addArgument( af );
	}

	if (pref.use_soft_vol) {
		proc->addArgument("-softvol");
	}


	proc->addArgument( file );

	// Additional options supplied by the user
	// File
	if (!mset.mplayer_additional_options.isEmpty()) {
		QStringList args = QStringList::split(" ", mset.mplayer_additional_options);
        QStringList::Iterator it = args.begin();
        while( it != args.end() ) {
 			proc->addArgument( (*it) );
			++it;
		}
	}
	// Global
	if (!pref.mplayer_additional_options.isEmpty()) {
		QStringList args = QStringList::split(" ", pref.mplayer_additional_options);
        QStringList::Iterator it = args.begin();
        while( it != args.end() ) {
 			proc->addArgument( (*it) );
			++it;
		}
	}

	//Log command
	//mplayer_log = "Command: \n";
	QString commandline;
    QStringList list = proc->arguments();
    QStringList::Iterator it = list.begin();
    while( it != list.end() ) {
        commandline += ( *it );
		commandline += " ";
        ++it;
    }
	mplayer_log += commandline + "\n\n";
	qDebug("Core::startMplayer: command: '%s'", commandline.utf8().data());

	
	if ( !proc->start() ) {
	    // error handling
		qWarning("Core::startMplayer: mplayer process didn't start");
	}

	//stopped_by_user = FALSE;

	// Try to set the volume as soon as possible
	tellmp("volume " + QString::number(mset.volume) + " 1");
}

void Core::stopMplayer() {
	qDebug("Core::stopMplayer");

	if (!proc->isRunning()) {
		qWarning("Core::stopMplayer: mplayer in not running!");
		return;
	}

    tellmp("quit");
    
	qDebug("Core::stopMplayer: Waiting mplayer to finish...");
	//Helper::finishProcess( proc );
	proc->waitForFinished();

	qDebug("Core::stopMplayer: Finished. (I hope)");
}


/*
void Core::goToSec( double sec )
{
	qDebug("Core::goToSec: %f", sec);

    if (sec < 0) sec = 0;
    if (sec > mdat.duration ) sec = mdat.duration - 20;
    tellmp("seek " + QString::number(sec) + " 2");
}
*/

void Core::seek(int secs) {
	qDebug("seek: %d", secs);
	if ( (proc->isRunning()) && (secs!=0) ) {
		tellmp("seek " + QString::number(secs) + " 0");
	}
}

void Core::sforward() {
	qDebug("Core::sforward");
	seek(+10);
}

void Core::srewind() {
	qDebug("Core::srewind");
	seek(-10);
}


void Core::forward() {
	qDebug("Core::forward");
	seek(+60);
}


void Core::rewind() {
	qDebug("Core::rewind");
	seek(-60);
}


void Core::fastforward() {
	qDebug("Core::fastforward");
	seek(+600);
}


void Core::fastrewind() {
	qDebug("Core::fastrewind");
	seek(-600);
}

void Core::forward(int secs) {
	qDebug("forward: %d", secs);
	seek(secs);
}

void Core::rewind(int secs) {
	qDebug("rewind: %d", secs);
	seek(-secs);
}

void Core::forwardWheel() {
	qDebug("forwardWheel");
	forward( pref.wheel_seek );
}

void Core::rewindWheel() {
	qDebug("rewindWheel");
	rewind( pref.wheel_seek );
}


// Audio filters
void Core::toggleKaraoke() {
	qDebug("Core::toggleKaraoke");
	mset.karaoke_filter = !mset.karaoke_filter;
	restartPlay();
}

void Core::toggleExtrastereo() {
	qDebug("Core::toggleExtrastereo");
	mset.extrastereo_filter = !mset.extrastereo_filter;
	restartPlay();
}

void Core::setAudioChannels(int channels) {
	qDebug("Core::setAudioChannels:%d", channels);
	if (channels != mset.audio_use_channels ) {
		mset.audio_use_channels = channels;
		restartPlay();
	}
}

void Core::setStereoMode(int mode) {
	qDebug("Core::setStereoMode:%d", mode);
	if (mode != mset.stereo_mode ) {
		mset.stereo_mode = mode;
		restartPlay();
	}
}


// Video filters
void Core::toggleAutophase() {
	qDebug("Core::toggleAutophase");
	mset.phase_filter = !mset.phase_filter;
	restartPlay();
}

void Core::toggleDenoiseNormal() {
	qDebug("Core::toggleDenoiseNormal");

	if ( (mset.current_denoiser == DENOISE_NONE) || (mset.current_denoiser == DENOISE_SOFT) ) {
		mset.current_denoiser = DENOISE_NORMAL;
	} else {
		mset.current_denoiser = DENOISE_NONE;
	}

	restartPlay();
}

void Core::toggleDenoiseSoft() {
	qDebug("Core::toggleDenoiseSoft");

	if ( (mset.current_denoiser == DENOISE_NONE) || (mset.current_denoiser == DENOISE_NORMAL) ) {
		mset.current_denoiser = DENOISE_SOFT;
	} else {
		mset.current_denoiser = DENOISE_NONE;
	}

	restartPlay();
}

void Core::toggleDeblock() {
	qDebug("Core::toggleDeblock");
	mset.deblock_filter = !mset.deblock_filter;
	restartPlay();
}

void Core::toggleDering() {
	qDebug("Core::toggleDering");
	mset.dering_filter = !mset.dering_filter;
	restartPlay();
}

void Core::toggleNoise() {
	qDebug("Core::toggleNoise");
	mset.noise_filter = !mset.noise_filter;
	restartPlay();
}

void Core::togglePostprocessing() {
	qDebug("Core::togglePostprocessing");
	mset.postprocessing_filter = !mset.postprocessing_filter;
	restartPlay();
}

void Core::setBrightness(int value) {
	qDebug("Core::setBrightness: %d", value);
	tellmp("brightness " + QString::number(value) + " 1");
	mset.brightness = value;
	displayMessage( tr("Brightness: %1").arg(value) );
	emit equalizerNeedsUpdate();
}


void Core::setContrast(int value) {
	qDebug("Core::setContrast: %d", value);
	tellmp("contrast " + QString::number(value) + " 1");
	mset.contrast = value;
	displayMessage( tr("Contrast: %1").arg(value) );
	emit equalizerNeedsUpdate();
}

void Core::setGamma(int value) {
	qDebug("Core::setGamma: %d", value);
	tellmp("gamma " + QString::number(value) + " 1");
	mset.gamma= value;
	displayMessage( tr("Gamma: %1").arg(value) );
	emit equalizerNeedsUpdate();
}

void Core::setHue(int value) {
	qDebug("Core::setHue: %d", value);
	tellmp("hue " + QString::number(value) + " 1");
	mset.hue = value;
	displayMessage( tr("Hue: %1").arg(value) );
	emit equalizerNeedsUpdate();
}

void Core::setSaturation(int value) {
	qDebug("Core::setSaturation: %d", value);
	tellmp("saturation " + QString::number(value) + " 1");
	mset.saturation = value;
	displayMessage( tr("Saturation: %1").arg(value) );
	emit equalizerNeedsUpdate();
}

void Core::incBrightness() {
	int v = mset.brightness + 4;
	if (v > 100) v = 100;
	setBrightness(v);
}

void Core::decBrightness() {
	int v = mset.brightness - 4;
	if (v < -100) v = -100;
	setBrightness(v);
}

void Core::incContrast() {
	int v = mset.contrast + 4;
	if (v > 100) v = 100;
	setContrast(v);
}

void Core::decContrast() {
	int v = mset.contrast - 4;
	if (v < -100) v = -100;
	setContrast(v);
}

void Core::incGamma() {
	int v = mset.gamma + 4;
	if (v > 100) v = 100;
	setGamma(v);
}

void Core::decGamma() {
	int v = mset.gamma - 4;
	if (v < -100) v = -100;
	setGamma(v);
}

void Core::incHue() {
	int v = mset.hue + 4;
	if (v > 100) v = 100;
	setHue(v);
}

void Core::decHue() {
	int v = mset.hue - 4;
	if (v < -100) v = -100;
	setHue(v);
}

void Core::incSaturation() {
	int v = mset.saturation + 4;
	if (v > 100) v = 100;
	setSaturation(v);
}

void Core::decSaturation() {
	int v = mset.saturation - 4;
	if (v < -100) v = -100;
	setSaturation(v);
}

void Core::setSpeed( double value ) {
	qDebug("Core::setSpeed: %f", value);

	if (value < 0.10) value = 0.10;
	if (value > 100) value = 100;

	mset.speed = value;
	tellmp( "speed_set " + QString::number( value ) );
}

void Core::incSpeed() {
	qDebug("Core::incSpeed");
	setSpeed( (double) mset.speed + 0.1 );
}

void Core::decSpeed() {
	qDebug("Core::decSpeed");
	setSpeed( (double) mset.speed - 0.1 );
}

void Core::doubleSpeed() {
	qDebug("Core::doubleSpeed");
	setSpeed( (double) mset.speed * 2 );
}

void Core::halveSpeed() {
	qDebug("Core::halveSpeed");
	setSpeed( (double) mset.speed / 2 );
}

void Core::normalSpeed() {
	setSpeed(1);
}

void Core::setVolume(int volume, bool force) {
	qDebug("Core::setVolume: %d", volume);

	if ((volume==mset.volume) && (!force)) return;

	mset.volume = volume;
	if (mset.volume > 99 ) mset.volume = 99;
	if (mset.volume < 0 ) mset.volume = 0;

    tellmp("volume " + QString::number(volume) + " 1");

	//if (mset.mute) mute(TRUE);
	mset.mute=FALSE;

	updateWidgets();

	displayMessage( tr("Volume: %1").arg(mset.volume) );
}

void Core::switchMute() {
	qDebug("Core::switchMute");

	mset.mute = !mset.mute;
	mute(mset.mute);
}

void Core::mute(bool b) {
	qDebug("Core::mute");

	mset.mute = b;

	int v = 0;
	if (mset.mute) v = 1;
	tellmp("mute " + QString::number(v) );

	updateWidgets();
}

void Core::incVolume() {
	qDebug("Core::incVolume");
	setVolume(mset.volume + 4);
}

void Core::decVolume() {
	qDebug("Core::incVolume");
	setVolume(mset.volume-4);
}

void Core::incSubDelay() {
	qDebug("Core::incSubDelay");

	mset.sub_delay += 100;
	tellmp("sub_delay " + QString::number( (double) mset.sub_delay/1000 ) +" 1");

	/*tellmp("panscan +0.2 0");*/
}

void Core::decSubDelay() {
	qDebug("Core::decSubDelay");

	mset.sub_delay -= 100;
	tellmp("sub_delay " + QString::number( (double) mset.sub_delay/1000 ) +" 1");
}

void Core::incAudioDelay() {
	qDebug("Core::incAudioDelay");

	mset.audio_delay += 100;
	tellmp("audio_delay " + QString::number( (double) mset.audio_delay/1000 ) +" 1");
}

void Core::decAudioDelay() {
	qDebug("Core::decAudioDelay");

	mset.audio_delay -= 100;
	tellmp("audio_delay " + QString::number( (double) mset.audio_delay/1000 ) +" 1");
}

void Core::incSubPos() {
	qDebug("Core::incSubPos");

	mset.sub_pos++;
	if (mset.sub_pos > 100) mset.sub_pos = 100;
	tellmp("sub_pos " + QString::number( mset.sub_pos ) + " 1");
}

void Core::decSubPos() {
	qDebug("Core::decSubPos");

	mset.sub_pos--;
	if (mset.sub_pos < 0) mset.sub_pos = 0;
	tellmp("sub_pos " + QString::number( mset.sub_pos ) + " 1");
}

void Core::changeCurrentSec(double sec) {
    mset.current_sec = sec;
	
	if (state != PLAYING) {
		state = PLAYING;
		qDebug("mplayer reports that now it's playing");
		emit mediaStartPlay();
		emit stateChanged(state);
	}

	emit showTime(sec);
}

void Core::changeCurrentFrame(int frame) {
	emit showFrame(frame);
}

void Core::changePause() {
	qDebug("Core::changePause");
	qDebug("mplayer reports that it's paused");
	state = PAUSED;
	emit stateChanged(state);
}

void Core::changeDeinterlace(int ID) {
	qDebug("Core::changeDeinterlace: %d", ID);

	if (ID!=mset.current_deinterlacer) {
		mset.current_deinterlacer = ID;
		restartPlay();
	}
}



void Core::changeSubtitle(int ID) {
	qDebug("Core::changeSubtitle: %d", ID);

	mset.current_sub_id = ID;
	if (ID==SUB_ID_NONE) {
		ID=-1;
		
		if (!mset.external_subtitles.isEmpty()) {
			mset.external_subtitles="";
			restartPlay();
		}
	}
	
	qDebug("Core::changeSubtitle: ID: %d", ID);
	tellmp( "sub_select " + QString::number(ID) );
	updateWidgets();
}

void Core::nextSubtitle() {
	qDebug("Core::nextSubtitle");

	int item;
	if ( (mset.current_sub_id == SUB_ID_NONE) && (mdat.subtitles.numItems() > 0) ) {
		item = 0;
		int ID = mdat.subtitles.itemAt(item).ID();
		changeSubtitle(ID);
	} else {
		item = mdat.subtitles.find( mset.current_sub_id );
		if (item == -1) {
			qWarning(" subtitle ID %d not found!", mset.current_sub_id);
		} else {
			qDebug( " numItems: %d, item: %d", mdat.subtitles.numItems(), item);
			item++;
			int ID;
			if (item >= mdat.subtitles.numItems()) {
				ID = SUB_ID_NONE;
			} else {
				ID = mdat.subtitles.itemAt(item).ID();
			}
			qDebug( " item: %d, ID: %d", item, ID);
			changeSubtitle( ID );
		}
	}
}

void Core::changeAudio(int ID) {
	qDebug("Core::changeAudio: ID: %d", ID);

	if (ID!=mset.current_audio_id) {
		mset.current_audio_id = ID;
		qDebug("changeAudio: ID: %d", ID);
		
		if (pref.audio_change_requires_restart) {
			restartPlay(); 
		} else {
			tellmp("switch_audio " + QString::number(ID) );
			if (mset.mute) mute(TRUE); // if muted, mute again
			updateWidgets();
		}
	}
}

void Core::nextAudio() {
	qDebug("Core::nextAudio");

	int item = mdat.audios.find( mset.current_audio_id );
	if (item == -1) {
		qWarning(" audio ID %d not found!", mset.current_audio_id);
	} else {
		qDebug( " numItems: %d, item: %d", mdat.audios.numItems(), item);
		item++;
		if (item >= mdat.audios.numItems()) item=0;
		int ID = mdat.audios.itemAt(item).ID();
		qDebug( " item: %d, ID: %d", item, ID);
		changeAudio( ID );
	}
}

void Core::changeTitle(int ID) {
	QString dvd_url = "dvd://" + QString::number(ID);
	QString folder = Helper::dvdSplitFolder(mdat.filename);
	if (!folder.isEmpty()) dvd_url += ":" + folder;

	openDVD(dvd_url);
	//openDVD( ID );
}

void Core::changeChapter(int ID) {
	qDebug("Core::changeChapter: ID: %d", ID);

	if (ID != mset.current_chapter_id) {
		//if (QFileInfo(mdat.filename).extension().lower()=="mkv") {
		if (mdat.mkv_chapters > 0) {
			// mkv doesn't require to restart
			tellmp("seek_chapter " + QString::number(ID) +" 1");
			mset.current_chapter_id = ID;
			updateWidgets();
		} else {
			if (pref.fast_chapter_change) {
				tellmp("seek_chapter " + QString::number(ID-1) +" 1");
				mset.current_chapter_id = ID;
				updateWidgets();
			} else {
				stopMplayer();
				mset.current_chapter_id = ID;
				//goToPos(0);
				mset.current_sec = 0;
				restartPlay();
			}
		}
	}
}

void Core::prevChapter() {
	qDebug("Core::prevChapter");

	int last_chapter = 0;
	bool matroshka = (mdat.mkv_chapters > 0);

	int first_chapter=1;
	if (matroshka) first_chapter = 0;

	// Matroshka chapters
	if (matroshka) last_chapter = mdat.mkv_chapters;
	else
	// DVD chapters
	if (mset.current_title_id > 0) {
		last_chapter = mdat.titles.item(mset.current_title_id).chapters();
	}

	int ID = mset.current_chapter_id - 1;
	if (ID < first_chapter) {
		ID = last_chapter;
	}
	changeChapter(ID);
}

void Core::nextChapter() {
	qDebug("Core::nextChapter");

	int last_chapter = 0;
	bool matroshka = (mdat.mkv_chapters > 0);

	// Matroshka chapters
	if (matroshka) last_chapter = mdat.mkv_chapters;
	else
	// DVD chapters
	if (mset.current_title_id > 0) {
		last_chapter = mdat.titles.item(mset.current_title_id).chapters();
	}

	int ID = mset.current_chapter_id + 1;
	if (ID > last_chapter) {
		if (matroshka) ID=0; else ID=1;
	}
	changeChapter(ID);
}

void Core::changeAngle(int ID) {
	qDebug("Core::changeAngle: ID: %d", ID);

	if (ID != mset.current_angle_id) {
		mset.current_angle_id = ID;
		restartPlay();
	}
}

void Core::changeAspectRatio( int ID ) {
	qDebug("Core::changeAspectRatio: %d", ID);

	int old_id = mset.aspect_ratio_id;
	mset.aspect_ratio_id = ID;
	bool need_restart = FALSE;

    double asp = mdat.video_aspect; // Set a default

    if (ID==ASPECT_43_LETTERBOX) {  
		need_restart = (old_id != ASPECT_43_LETTERBOX);
		asp = (double) 4 / 3;
        mset.letterbox = MediaSettings::Letterbox_43;
		mset.panscan_filter = "";
		mset.crop_43to169_filter = "";
	}
	else
    if (ID==ASPECT_169_LETTERBOX) {  
		need_restart = (old_id != ASPECT_169_LETTERBOX);
		asp = (double) 16 / 9;
        mset.letterbox = MediaSettings::Letterbox_169;
		mset.panscan_filter = "";
		mset.crop_43to169_filter = "";
	}
	else
	if (ID==ASPECT_43_PANSCAN) {
		need_restart = (old_id != ASPECT_43_PANSCAN);
		mset.crop_43to169_filter = "";
		mset.letterbox = MediaSettings::None;

		asp = (double) 4 / 3;
		int real_width = (int) round(mdat.video_height * mdat.video_aspect);
		mset.panscan_filter = QString("scale=%1:%2,").arg(real_width).arg(mdat.video_height);
		mset.panscan_filter += QString("crop=%1:%2").arg(round(mdat.video_height * 4 /3)).arg(mdat.video_height);
		//mset.crop = QSize( mdat.video_height * 4 /3, mdat.video_height );
		qDebug(" panscan_filter = '%s'", mset.panscan_filter.utf8().data() );

	}
    else
	if (ID==ASPECT_43_TO_169) {
		need_restart = (old_id != ASPECT_43_TO_169);
		mset.panscan_filter = "";
		mset.crop_43to169_filter = "";
		mset.letterbox = MediaSettings::None;

		int real_width = (int) round(mdat.video_height * mdat.video_aspect);
		int height = (int) round(real_width * 9 / 16);

		qDebug("video_width: %d, video_height: %d", real_width, mdat.video_height);
		qDebug("crop: %d, %d", real_width, height );

		if (height > mdat.video_height) {
			// Invalid size, source video is not 4:3
			need_restart = FALSE;
		} else {
			asp = (double) 16 / 9;
			mset.crop_43to169_filter = QString("scale=%1:%2,").arg(real_width).arg(mdat.video_height);
			mset.crop_43to169_filter += QString("crop=%1:%2").arg(real_width).arg(height);
			qDebug(" crop_43to169_filter = '%s'", mset.crop_43to169_filter.utf8().data() );
		}
	}
	else
    {
		//need_restart = (mset.force_letterbox == TRUE);
		need_restart = ( (old_id == ASPECT_43_LETTERBOX) || (old_id == ASPECT_169_LETTERBOX) || 
                         (old_id == ASPECT_43_PANSCAN) || (old_id == ASPECT_43_TO_169) );
		mset.letterbox = MediaSettings::None;
		mset.panscan_filter = "";
		mset.crop_43to169_filter = "";
        switch (ID) {
        	case ASPECT_AUTO: asp = mdat.video_aspect; break;
            case ASPECT_43: asp = (double) 4 / 3; break;
            case ASPECT_169: asp = (double) 16 / 9; break;
			case ASPECT_149: asp = (double) 14 / 9; break;
			case ASPECT_1610: asp = (double) 16 / 10; break;
			case ASPECT_54: asp = (double) 5 / 4; break;
            case ASPECT_235: asp = 2.35; break;
		}
	}

	mplayerwindow->setAspect( asp );
    //tellmp("switch_ratio " + QString::number( asp ) );

	updateWidgets();

    if (need_restart) {
		/*mdat.calculateWinResolution(mset.force_letterbox);*/
    	restartPlay();
	}
}

void Core::changeOSD(int v) {
	qDebug("Core::changeOSD: %d", v);

	pref.osd = v;
	tellmp("osd " + QString::number( pref.osd ) );
	updateWidgets();
}

/*
void Core::processKeyEvent( QKeyEvent *e ) {
	qDebug("Core::processKeyEvent: '%s'", e->text().utf8().data());

	QString f = key_list->function(e);
	qDebug("Core::processKeyEvent: key function: '%s'", f.utf8().data() );

	if (f.isEmpty()) {
		e->ignore(); 
		return;
	} else {
		e->accept();
	}
*/

void Core::processFunction(QString f) {
	qDebug("Core::processFunction: '%s'", f.utf8().data());

	if (f=="OSD") {
		pref.osd++;
		if (pref.osd > OSD_SEEK_TIMER_TOTAL) {
			pref.osd = OSD_NONE;	
		}
		changeOSD( pref.osd );
	}
	else
	if (f=="PAUSE") {
		pause();
	}
	else
	if (f=="PLAY_PAUSE") {
		play_or_pause();
	}
	else
	if (f=="STOP") {
		stop();
	}
	else
	if (f=="FRAME_STEP") {
		frameStep();
	}
	else
	if (f=="FORWARD1") {
		sforward(); // + 10 secs
	}
	else
	if (f=="REWIND1") {
		srewind(); // - 10 secs
	}
	else
	if (f=="FORWARD2") {
		forward(); // + 1 minute
	}
	else
	if (f=="REWIND2") {
		rewind(); // - 1 minute
	}
	else
	if (f=="FORWARD3") {
		fastforward(); // + 10 minutes
	}
	else
	if (f=="REWIND3") {
		fastrewind(); // - 10 minutes
	}
	else
	if (f=="MUTE") {
		switchMute();
	}
	else
	if (f=="INC_VOLUME") {
		incVolume();
	}
	else
	if (f=="DEC_VOLUME") {
		decVolume();
	}
	else
	if (f=="INC_AUDIO_DELAY") {
		incAudioDelay();
	}
	else
	if (f=="DEC_AUDIO_DELAY") {
		decAudioDelay();
	}
	else
	if (f=="INC_SUB_DELAY") {
		incSubDelay();
	}
	else
	if (f=="DEC_SUB_DELAY") {
		decSubDelay();
	}
	else

	// Contrast, Brightness and so on
	if (f=="DEC_CONTRAST") {
		decContrast();
	}
	else
	if (f=="INC_CONTRAST") {
		incContrast();
	}
	else
	if (f=="DEC_BRIGHTNESS") {
		decBrightness();
	}
	else
	if (f=="INC_BRIGHTNESS") {
		incBrightness();
	}
	else
	if (f=="DEC_HUE") {
		decHue();
	}
	else
	if (f=="INC_HUE") {
		incHue();
	}
	else
	if (f=="DEC_SATURATION") {
		decSaturation();
	}
	else
	if (f=="INC_SATURATION") {
		incSaturation();
	}
	if (f=="DEC_GAMMA") {
		decGamma();
	}
	else
	if (f=="INC_GAMMA") {
		incGamma();
	}
	else

	// Subtitles position
	if (f=="DEC_SUBPOS") {
		decSubPos();
	}
	else
	if (f=="INC_SUBPOS") {
		incSubPos();
	}
	else

	// Speed
	if (f=="DEC_SPEED") {
		decSpeed();
	}
	else
	if (f=="INC_SPEED") {
		incSpeed();
	}
	else
	if (f=="HALVE_SPEED") {
		halveSpeed();
	}
	else
	if (f=="DOUBLE_SPEED") {
		doubleSpeed();
	}
	else
	if (f=="NORMAL_SPEED") {
		normalSpeed();
	}
	else

	// Screenshot
	if (f=="SCREENSHOT") {
		screenshot();
	}
	else

	// Next (audio, subtitles...)
	if (f=="AUDIO_NEXT") {
		nextAudio();
	}
	else
	if (f=="SUBTITLE_NEXT") {
		nextSubtitle();
	}
	else
	if (f=="CHAPTER_NEXT") {
		nextChapter();
	}
	else
	if (f=="CHAPTER_PREV") {
		prevChapter();
	}
}


void Core::displayMessage(QString text) {
	qDebug("Core::displayMessage");
	emit showMessage(text);
}

void Core::displayScreenshotName(QString filename) {
	qDebug("Core::displayScreenshotName");
	QString text = tr("Screenshot saved as %1").arg(filename);
	tellmp("osd_show_text \"" + text + "\" 3000 1");
	emit showMessage(text);
}


void Core::gotWindowResolution(int w, int h) {
	qDebug("Core::gotWindowResolution: %d, %d", w, h);
	//double aspect = (double) w/h;

	if (pref.use_mplayer_window) {
		emit noVideo();
	} else {
		if ((pref.resize_method==RESIZE_AFTERLOAD) && (we_are_restarting)) {
			// Do nothing
		} else {
			emit needResize(w,h);
		}
	}

	mset.win_width = w;
	mset.win_height = h;

	mplayerwindow->setResolution( w, h );
	mplayerwindow->setAspect( mset.win_aspect() );
}

void Core::gotNoVideo() {
	// File has no video (a sound file)

	// Reduce size of window
	/*
	mset.win_width = mplayerwindow->size().width();
	mset.win_height = 0;
	mplayerwindow->setResolution( mset.win_width, mset.win_height );
	emit needResize( mset.win_width, mset.win_height );
	*/
	//mplayerwindow->showLogo(TRUE);
	emit noVideo();
}

void Core::gotVO(QString vo) {
	qDebug("Core::gotVO: '%s'", vo.utf8().data() );

	if ( pref.vo.isEmpty()) {
		qDebug("saving vo");
		pref.vo = vo;
	}
}

void Core::gotAO(QString ao) {
	qDebug("Core::gotAO: '%s'", ao.utf8().data() );

	if ( pref.ao.isEmpty()) {
		qDebug("saving ao");
		pref.ao = ao;
	}
}
