/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvd.h>
#include "pes.h"

gboolean
dvd_pes_check_start_code	(guint8		**stream)
{
	guint8 *iter;
	guint32 start_code;
	
	iter = *stream;
	
	start_code = (guint32) *iter;
	start_code = start_code << 8;
	iter++;
	start_code += *iter;
	start_code = start_code << 8;
	iter++;
	start_code += *iter;
	iter++;
	if (start_code != 1) {
		return FALSE;
	} else {
		*stream = iter;
		return TRUE;
	}
}

guint8
dvd_pes_find_header		(guint8		**stream,
				 guint8		 *end)
{
	guint8 *iter;
	guint32 start_code;
	
	iter = *stream;
	
	start_code = (guint32) *iter;
	start_code = start_code << 8;
	iter++;
	start_code += *iter;
	iter++;
	
	while (iter < end) {
		start_code = start_code << 8;
		start_code += *iter;
		iter++;
		if (start_code == 1) {
			guint id;
			
			id = *iter;
			*stream = iter - 3;
			return id;
		}
	}
	*stream = end;
	return DVD_PES_START_CODE_NOT_FOUND;
}

gboolean
dvd_pes_read_pack_header	(DvdPackHeader		 *header,
				 guint8			**stream)
{
	guint8 *iter;
	guint8 bit;
	guint8 stream_id;
	guint8 stuffing;
	
	memset (header, '0', sizeof (DvdPackHeader));
	
	iter = *stream;
	if (dvd_pes_check_start_code (&iter) == FALSE) {
		return FALSE;
	}
	
	stream_id = *iter;
	iter++;
	
	if (stream_id != DVD_PES_START_CODE_PACK_HDR) {
		g_warning ("not a pack header");
		return FALSE;
	}
	
	/* read SCR */
	bit = 0;
	if (getbits (&iter, 2, &bit) != 1) {
		g_warning ("invalid pack header");
		return FALSE;
	}
	header->scr = getbits (&iter, 3, &bit);
	header->scr <<= 15;
	getbits (&iter, 1, &bit);	/* 1 */
	header->scr |= getbits (&iter, 15, &bit);
	header->scr <<= 15;
	getbits (&iter, 1, &bit);	/* 1 */
	header->scr |= getbits (&iter, 15, &bit);
	header->scr *= 300;
	getbits (&iter, 1, &bit);	/* 1 */
	header->scr += getbits (&iter, 9, &bit);
	getbits (&iter, 1, &bit);	/* 1 */
	
	/* mux rate */
	header->mux_rate = *iter;
	header->mux_rate <<= 8;
	iter++;
	header->mux_rate += *iter;
	header->mux_rate <<= 6;
	iter++;
	header->mux_rate += ((*iter) >> 2);
	iter++;
	
	/* stuffing length */
	stuffing = (*iter) & 0x7;
	*stream = iter + stuffing + 1;
	return TRUE;
}

gboolean
dvd_pes_read_stream_header 	(DvdPESHeader		 *header,
				 guint8			**stream)
{
	guint8 *iter;
	guint8 bit;
	
	memset (header, '0', sizeof (DvdPESHeader));
	iter = *stream;
	if (dvd_pes_check_start_code (&iter) == FALSE) {
		return FALSE;
	}
	header->stream_id = *iter;
	iter++;
	
	if (header->stream_id == DVD_PES_START_CODE_PACK_HDR) {
		g_warning ("pack header - not pes header");
		return FALSE;
	}
	header->packet_length = *iter;
	header->packet_length = header->packet_length << 8;
	iter++;
	header->packet_length += *iter;
	iter++;
	
	/*g_message ("pes header type 0x%x", header->stream_id);*/
	if ((header->stream_id == DVD_PES_START_CODE_PADDING) ||
	    (header->stream_id == DVD_PES_START_CODE_PRIV_STREAM2)) {
		/* padding and private stream 2 don't have extension */
		*stream = iter;
		return TRUE;
	}
	
	/* extension flags - byte 7 */
	bit = 0;
	header->reserved0	= getbits (&iter, 2, &bit);
	g_assert (header->reserved0 == 2);
	header->scrambled	= getbits (&iter, 2, &bit);
	header->priority	= getbits (&iter, 1, &bit);
	header->alignmentp	= getbits (&iter, 1, &bit);
	header->copyb		= getbits (&iter, 1, &bit);
	header->original	= getbits (&iter, 1, &bit);
	
	/* extension flags - byte 8 */
	header->pts_dts		= getbits (&iter, 2, &bit);
	header->escre		= getbits (&iter, 1, &bit);
	header->es_ratee	= getbits (&iter, 1, &bit);
	header->dsm_trick_mode	= getbits (&iter, 1, &bit);
	header->addicopyb	= getbits (&iter, 1, &bit);
	header->crce		= getbits (&iter, 1, &bit);
	header->exte		= getbits (&iter, 1, &bit);
	
	/* extension  - byte 9 */
	/* length of header data remaining including 0xff stuffing bytes */
	header->hdr_dlength = *iter;
	iter++;
	*stream = iter + header->hdr_dlength;
	
	/* extension data */
	if (header->pts_dts > 0) {
		header->pts = ((*iter) >> 1) & 7;
		iter++;
		header->pts <<= 15;
		header->pts |= (((iter[0] << 8) | iter[1]) >> 1);
		iter += 2;
		header->pts <<= 15;
		header->pts |= (((iter[0] << 8) | iter[1]) >> 1);
		iter += 2;
		/*g_message ("pts = %u", header->pts);*/
		if (header->pts_dts == 3) {
			header->dts = ((*iter) >> 1) & 7;
			iter++;
			header->dts <<= 15;
			header->dts |= (((iter[0] << 8) | iter[1]) >> 1);
			iter += 2;
			header->dts <<= 15;
			header->dts |= (((iter[0] << 8) | iter[1]) >> 1);
			iter += 2;
			/*g_message ("dts = %u", header->dts);*/
		}
	}
	if (header->escre > 0) {
		/* ESCR used if the stream and system levels are not synchronized */
		/* (ESCR differs from SCR in the PACK header). */
		bit = 2;			/* 00 */
		header->escr = getbits (&iter, 3, &bit);
		header->escr <<= 15;
		getbits (&iter, 1, &bit);	/* 1 */
		header->escr |= getbits (&iter, 15, &bit);
		header->escr <<= 15;
		getbits (&iter, 1, &bit);	/* 1 */
		header->escr |= getbits (&iter, 15, &bit);
		getbits (&iter, 1, &bit);	/* 1 */
		header->escr *= 300;
		header->escr += getbits (&iter, 9, &bit);
		getbits (&iter, 1, &bit);	/* 1 */
		/*g_message ("escr=%llu", (guint64) header->escr / 300);*/
	}
	
	if (header->es_ratee > 0) {
		/* The rate at which data is delivered for this stream */
		/* In units of 50 bytes/second */
		header->es_rate = (*iter) & 0x7f;
		header->es_rate <<= 8;
		iter++;
		header->es_rate += *iter;
		iter++;
		header->es_rate <<= 7;
		header->es_rate += ((*iter) >> 1);
		iter++;
	}
	
	/* DSM trick not used by DVD */
	
	if (header->addicopyb > 0) {
		header->addicopy_info = (*iter) & 0x7f;
	}
	
	if (header->crce > 0) {
		/*g_message ("crce");*/
		header->prev_crc = *iter;
		iter++;
		header->prev_crc <<= 8;
		header->prev_crc += *iter;
		iter++;
	}
	
	/* extensions extension */
	if (header->exte > 0) {
		bit = 0;
		/*g_message ("exte");*/
		header->priv_data	= getbits (&iter, 1, &bit);
		header->pack_hdr	= getbits (&iter, 1, &bit);
		header->pps_counter	= getbits (&iter, 1, &bit);
		header->pstd_buf	= getbits (&iter, 1, &bit);
		getbits (&iter, 3, &bit);
		header->pes_ext2	= getbits (&iter, 1, &bit);
		if (header->priv_data > 0) {
			/* 16 bytes of user defined data */
			iter += 16;
		}
		if (header->pack_hdr > 0) {
			header->pack_length = *iter;
			iter++;
		}
		if (header->pps_counter > 0) {
			header->pps = (*iter) & 0x7f;
			iter++;
			bit = 0;
			getbits (&iter, 1, &bit);
			header->mpeg_id		= getbits (&iter, 1, &bit);
			header->orig_stuffing	= getbits (&iter, 6, &bit);
		}
		/* 2 byte pstd buffer if pstd_buf = 1 */
		if (header->pstd_buf > 0) {
			if (getbits (&iter, 2, &bit) != 1) {
				g_warning ("no 01 before pstd_buf_scale");
			}
			header->pstd_buf_scale = getbits (&iter, 1, &bit);	/* if 1 multiply pstd_duf_size by 1024 else by 128 */
			header->pstd_buf_size = getbits (&iter, 13, &bit);	/* size of P-STD buffer - also see above */
			if (header->pstd_buf_scale == 1) {
				/*g_message ("pstd_buf_size %lu", (guint32) header->pstd_buf_size * 1024);*/
			} else {
				/*g_message ("pstd_buf_size = %lu", (guint32) header->pstd_buf_size * 128);*/
			}
			/*iter += 2;*/
		}
		/* pes extensions extensions extension data if pes_ext2 = 1 */
		if (header->pes_ext2 > 0) {
			iter += ((*iter) & 0x7f) + 2;
		}
	} /* end if (exte > 0) */
	/* maybe some 0xff stuffing bytes */
	/*while (*iter == 0xff) {
		iter++;
		g_message ("stuffing");
	}
	if (*stream != iter) {
		g_warning ("pes header length mismatch");
	}*/
	return TRUE;
}

gboolean
dvd_pes_read_ps1_header		(DvdPS1Header		 *header,
				 guint8			**stream)
{
	guint8 *iter;
	guint8 type;
	
	memset (header, '0', sizeof (DvdPS1Header));
	iter = *stream;
	type = (*iter) >> 3;
	switch (type) {
	case 16:
		header->stream_type = DVD_STREAM_AC3_AUDIO;
		header->track = (*iter) & 0x7;
		break;
	case 17:
		header->stream_type = DVD_STREAM_DTS_AUDIO;
		header->track = (*iter) & 0x7;
		break;
	case 20:
		header->stream_type = DVD_STREAM_LPCM_AUDIO;
		header->track = (*iter) & 0x7;
		break;
	case 4:
	case 5:
	case 6:
	case 7:
		type -= 4;
		header->track = *iter - 33;
		header->stream_type = DVD_STREAM_SUB_PICTURE;
		break;
	default:
		g_warning ("unknown PS1 %d", type);
		break;
	}
	iter++;
	header->frames = *iter;
	iter++;
	header->pts_offset = *iter;
	iter++;
	header->pts_offset <<= 8;
	header->pts_offset += *iter - 1;
	*stream = iter + 1;
	return TRUE;
}

void
dvd_pes_read_lpcm_header	(DvdLpcmHeader		 *header,
				 guint8			**stream)
{
	guint8 bit;
	
	memset (header, '0', sizeof (DvdPS1Header));
	bit = 0;
	header->emphasis  = getbits (stream, 1, &bit);
	header->mute      = getbits (stream, 1, &bit);
	header->reserved1 = getbits (stream, 1, &bit);
	header->frame     = getbits (stream, 5, &bit);
	header->word_size = getbits (stream, 2, &bit);
	header->samples   = getbits (stream, 2, &bit);
 	header->reserved2 = getbits (stream, 1, &bit);
 	header->channels  = getbits (stream, 3, &bit);
	header->dr_x      = getbits (stream, 3, &bit);
 	header->dr_y      = getbits (stream, 5, &bit);
 	/* note - getbits () indexes stream to next byte */
}

/* no better place for this yet */
guint32
getbits				(guint8			**stream,
				 guint8			  bits,
				 guint8			 *cur_bit)
{
	guint8 bit = 0;
	guint32 val = 0;
	guint8 *byte;
	
	for (bit = 0;
	     bit < bits;
	     bit++) {
		val = val << 1;
		byte = *stream;
		if (((*byte) & (0x80 >> (*cur_bit))) != 0) {
			val |= 1;
		}
		*cur_bit += 1;
		if (*cur_bit == 8) {
			*cur_bit = 0;
			*stream = byte + 1;
		}
	}
	return val;
}
