/****************************************************************************
 **
 ** Copyright (C) 2001-2005 Tilo Riemer <riemer@lincvs.org> and
 **                         Frank Hemer <frank@hemer.org>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "Diff.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "DiffDialogImpl.h"
#include "DiffADialogImpl.h"

Diff * Diff::diffFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, NULL, FILE);
}

Diff * Diff::diffDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, NULL, DIR);
}

Diff * Diff::diffAFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, QStringList * currentTagList)
{
   return setup(whatsThisIconSet, parent, workBench, false, currentTagList, AFILE);
}

Diff * Diff::diffADir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, QStringList * currentTagList)
{
   return setup(whatsThisIconSet, parent, workBench, true, currentTagList, ADIR);
}

Diff * Diff::diffFileSideBySide(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, NULL, SIDEBYSIDE);
}

Diff * Diff::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir, QStringList * currentTagList, int mode) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   if (dir->getType() == DirBase::Cvs) {

      DiffADialogImpl * dlg = NULL;
      bool cancel = false;
      if (mode == ADIR || mode == AFILE) {
	 dlg = new DiffADialogImpl(whatsThisIconSet,
	       parent, "DiffADialog", currentTagList,
	       dir->topControlledDir()->relativeName(), true);

	 if (!dlg->exec()) {
	    cancel = true;
	    delete dlg;
	    dlg = NULL;
	 }
      }

      if (!cancel) {
	 Diff * p = new Diff(whatsThisIconSet, parent, workBench, dir, isDir, mode);
	 if (dlg) p->m_dlg = dlg;
	 p->acceptCvs();
      }
   } else {
      qDebug("Diff: Method not implemented");
   }
   return NULL;
}

Diff::Diff(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir, int mode)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_mode(mode)
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
}

Diff::~Diff() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Diff::acceptCvs() {

   assert(m_dir);

   int cmd = -1;
   QString files = QString::null;
   QStringList filesList;
   QString command = CvsOptions::cmprStr();
   switch(m_mode) {
      case FILE: {
         command += " -f diff";
         if (bDiffIgnoreWhiteSpace) command += " -b -B";
         filesList = m_dir->getSelectedFiles();
         if (!filesList.isEmpty()) {
            masqQuoteMarks(&filesList);
            files = masqWs(filesList.join("\" \""));
         }
         cmd = CVS_DIFF_CMD;
         break;
      }
      case DIR: {
         command += " -f diff -N";
         if (bDiffIgnoreWhiteSpace) command += " -b -B";
         cmd = CVS_DIFF_CMD;
         break;
      }
      case AFILE:
         //command += " diff"; <-- add "diff" in ADIR already
         //cmd = CVS_DIFF_CMD; <-- set it in ADIR
         filesList = m_dir->getSelectedFiles();
         if (!filesList.isEmpty()) {
            masqQuoteMarks(&filesList);
            files = masqWs(filesList.join("\" \""));
         }
         // NOTE: no break!!!!
      case ADIR: {
         command += " -f diff -N";
         if (bDiffIgnoreWhiteSpace) command += " -b -B";
         cmd = CVS_DIFF_CMD;
         DiffADialogImpl * dlg = static_cast<DiffADialogImpl*>(m_dlg);
         assert(dlg);
         command += " -r " + dlg->getFirstTag();
         QString secondTag = dlg->getSecondTag();
         if (!secondTag.isEmpty()) command += " -r " + secondTag;
         break;
      }
      case SIDEBYSIDE: {
         QString file = m_dir->getFirstSelectedFile();
         if (file.isEmpty()) break;

         if (bUSEEXTERNALDIFFFORSIDEBYSIDE) {
            m_diffA = m_dir->fullName() + "/" + file;
            m_diffB = createTempFile() + "-" + file;
            QString rev = m_dir->getRev (&file);
            command += " -Q update -r ";
            command += rev + " ";
            //if (bKeywordSuppression) command += "-kk ";// commented because diff compares to local file
            // which is not necessarily checked out with -kk
            command += "-p " + masqWs(file.replace("\"", PLACEHOLDER_FOR_QUOTATION_MARKS)) + " > " + m_diffB;
            cmd = CVS_DIFF_EXT_CMD;
         } else {	      
            m_diffA = file;
            command += " -Q diff -N ";
            if (bDiffIgnoreWhiteSpace) command += "-b -B ";
            //if (bKeywordSuppression) command += "-kk ";// commented because diff compares to local file
            // which is not necessarily checked out with -kk
            QString hlp;
            command += "--side-by-side -t -W "+hlp.setNum(SIDEWIDTH);
            cmd = CVS_DIFF_SBS_CMD;
            files = masqWs(file.replace("\"", PLACEHOLDER_FOR_QUOTATION_MARKS));
         }
         break;
      }
      default: {
         assert(false);
      }
   }
   if (!files.isNull() && files.isEmpty()) {
      reject();
      return;
   }

   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, command,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void Diff::reject() {
   emit deleteObject(this);
}

void Diff::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Diff::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Diff::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   if (failed) {
      cvsCallFinished();
      reject();
      return;
   }
   assert(m_dir);
   switch( cmd) {
      case CVS_DIFF_CMD:
      case NOOP: {
	 break;
      }
      case CVS_DIFF_SBS_CMD: {
	 QString rev = m_dir->getRev(&m_diffA);
	 assert(!rev.isEmpty());
	 DiffDialogImpl *l = new DiffDialogImpl(m_whatsThisIconSet,
	       m_parent, "DiffDlg", 
	       LookAndFeel::g_nonModalF | WDestructiveClose);
               
	 l->parseCvsDiff(buf, m_diffA, rev);
	 l->show();
	 break;
      }
      case CVS_DIFF_EXT_CMD: {
	 QString diffPrg = "\"" + ExtApps::g_diffProg.path + "\" ";
	 QStringList nameList;
	 QString tmp = m_diffB;
	 nameList.append(tmp.replace("\"", PLACEHOLDER_FOR_QUOTATION_MARKS));
	 tmp = m_diffA;
	 nameList.append(tmp.replace("\"", PLACEHOLDER_FOR_QUOTATION_MARKS));
	 nameList.append("");  //3way diff from within logdlg --> there we can use 3 parameters
	 QString options = ExtApps::g_diffProg.options;
	 if (masqedFilenamesForPlaceholders(options, nameList)) diffPrg += options;
	 else {// no options, we add filenames assuming the user want it so ;-)
	    diffPrg += masqWs(m_diffB.latin1()) + " ";
	    diffPrg += masqWs(m_diffA.latin1());
	 }
	 runExternal(diffPrg);
	 break;
      }
      default: {
	 qDebug("Diff::afterCall: Unknown cmd");
      }
   }
   cvsCallFinished();
   reject();
}
