// $Header: /home/cvs/jakarta-jmeter/src/core/org/apache/jmeter/gui/tree/JMeterTreeModel.java,v 1.24 2004/02/13 02:21:36 sebb Exp $
/*
 * Copyright 2001-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
*/

package org.apache.jmeter.gui.tree;

import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.tree.DefaultTreeModel;

import org.apache.jmeter.config.gui.AbstractConfigGui;
import org.apache.jmeter.control.gui.TestPlanGui;
import org.apache.jmeter.control.gui.WorkBenchGui;
import org.apache.jmeter.exceptions.IllegalUserActionException;
import org.apache.jmeter.gui.GuiPackage;
import org.apache.jmeter.gui.JMeterGUIComponent;
import org.apache.jmeter.testelement.TestElement;
import org.apache.jmeter.testelement.TestPlan;
import org.apache.jmeter.testelement.WorkBench;
import org.apache.jmeter.testelement.property.NullProperty;
import org.apache.jmeter.util.NameUpdater;
import org.apache.jorphan.collections.HashTree;
import org.apache.jorphan.collections.ListedHashTree;

/**
 *
 * @author    Michael Stover
 * @version   $Revision: 1.24 $
 */
public class JMeterTreeModel extends DefaultTreeModel
{

    public JMeterTreeModel()
    {
        super(new JMeterTreeNode(new WorkBenchGui().createTestElement(), null));
        initTree();
    }

    /**
     * Returns a list of tree nodes that hold objects of the given class type.
     * If none are found, an empty list is returned. 
     */
    public List getNodesOfType(Class type)
    {
        List nodeList = new LinkedList();
        traverseAndFind(type, (JMeterTreeNode) this.getRoot(), nodeList);
        return nodeList;
    }

    /**
     * Get the node for a given TestElement object.
     */
    public JMeterTreeNode getNodeOf(TestElement userObject)
    {
        return traverseAndFind(
            userObject,
            (JMeterTreeNode)getRoot());
    }

    /**
     * Adds the sub tree at the given node.  Returns a boolean indicating
     * whether the added sub tree was a full test plan.
     */
    public HashTree addSubTree(HashTree subTree, JMeterTreeNode current)
        throws IllegalUserActionException
    {
        Iterator iter = subTree.list().iterator();
        while (iter.hasNext())
        {
            TestElement item = (TestElement) iter.next();
            if (item instanceof TestPlan)
            {
                current =
                    (JMeterTreeNode) ((JMeterTreeNode) getRoot()).getChildAt(0);
                ((TestElement) current.getUserObject()).addTestElement(item);
                ((TestPlan) current.getUserObject()).setName(
                    item.getPropertyAsString(TestElement.NAME));
                ((TestPlan) current.getUserObject()).setFunctionalMode(
                    item.getPropertyAsBoolean(TestPlan.FUNCTIONAL_MODE));
                ((TestPlan) current.getUserObject()).setSerialized(
                    item.getPropertyAsBoolean(TestPlan.SERIALIZE_THREADGROUPS));
                addSubTree(subTree.getTree(item), current);
            }
            else if (item instanceof WorkBench)
            {
                current =
                    (JMeterTreeNode) ((JMeterTreeNode) getRoot()).getChildAt(1);
                ((TestElement) current.getUserObject()).addTestElement(item);
                ((WorkBench) current.getUserObject()).setName(
                    item.getPropertyAsString(TestElement.NAME));
                addSubTree(subTree.getTree(item), current);
            }
            else
            {
                addSubTree(subTree.getTree(item), addComponent(item, current));
            }
        }
        return getCurrentSubTree(current);
    }

    public JMeterTreeNode addComponent(
        TestElement component,
        JMeterTreeNode node)
        throws IllegalUserActionException
    {
        if (node.getUserObject() instanceof AbstractConfigGui)
        {
            throw new IllegalUserActionException(
                    "This node cannot hold sub-elements");
        }
        component.setProperty(
            TestElement.GUI_CLASS,
            NameUpdater.getCurrentName(
                component.getPropertyAsString(TestElement.GUI_CLASS)));
        GuiPackage.getInstance().updateCurrentNode();
        JMeterGUIComponent guicomp = GuiPackage.getInstance().getGui(component);
        guicomp.configure(component);
        guicomp.modifyTestElement(component);
        GuiPackage.getInstance().getCurrentGui(); //put the gui object back to the way it was.
        JMeterTreeNode newNode =
            new JMeterTreeNode((TestElement) component, this);

        // This check the state of the TestElement and if returns false it
        // disable the loaded node
        try
        {
            if (component.getProperty(TestElement.ENABLED)
                instanceof NullProperty
                || component.getPropertyAsBoolean(TestElement.ENABLED))
            {
                newNode.setEnabled(true);
            }
            else
            {
                newNode.setEnabled(false);
            }
        }
        catch (Exception e)
        {
            newNode.setEnabled(true);
        }

        this.insertNodeInto(newNode, node, node.getChildCount());
        return newNode;
    }

    public void removeNodeFromParent(JMeterTreeNode node)
    {
        if (!(node.getUserObject() instanceof TestPlan)
            && !(node.getUserObject() instanceof WorkBench))
        {
            super.removeNodeFromParent(node);
        }
    }

    private void traverseAndFind(
        Class type,
        JMeterTreeNode node,
        List nodeList)
    {
        if (type.isInstance(node.getUserObject()))
        {
            nodeList.add(node);
        }
        Enumeration enum = node.children();
        while (enum.hasMoreElements())
        {
            JMeterTreeNode child = (JMeterTreeNode) enum.nextElement();
            traverseAndFind(type, child, nodeList);
        }
    }

    private JMeterTreeNode traverseAndFind(
        TestElement userObject,
        JMeterTreeNode node)
    {
        if (userObject == node.getUserObject())
        {
            return node;
        }
        Enumeration enum = node.children();
        while (enum.hasMoreElements())
        {
            JMeterTreeNode child = (JMeterTreeNode) enum.nextElement();
            JMeterTreeNode result= traverseAndFind(userObject, child);
            if (result != null) return result;
        }
        return null;
    }

    public HashTree getCurrentSubTree(JMeterTreeNode node)
    {
        ListedHashTree hashTree = new ListedHashTree(node);
        Enumeration enum = node.children();
        while (enum.hasMoreElements())
        {
            JMeterTreeNode child = (JMeterTreeNode) enum.nextElement();
            hashTree.add(node, getCurrentSubTree(child));
        }
        return hashTree;
    }

    public HashTree getTestPlan()
    {
        return getCurrentSubTree(
            (JMeterTreeNode) ((JMeterTreeNode) this.getRoot()).getChildAt(0));
    }

    public void clearTestPlan()
    {
        super.removeNodeFromParent((JMeterTreeNode) getChild(getRoot(), 0));
        initTree();
    }

    private void initTree()
    {
        TestElement tp = new TestPlanGui().createTestElement();
        TestElement wb = new WorkBenchGui().createTestElement();
        this.insertNodeInto(
            new JMeterTreeNode(tp, this),
            (JMeterTreeNode) getRoot(),
            0);
        try
        {
            super.removeNodeFromParent((JMeterTreeNode) getChild(getRoot(), 1));
        }
        catch (RuntimeException e)
        {
        }
        this.insertNodeInto(
            new JMeterTreeNode(wb, this),
            (JMeterTreeNode) getRoot(),
            1);
    }
}
