// $Header: /home/cvs/jakarta-jmeter/src/components/org/apache/jmeter/assertions/gui/SizeAssertionGui.java,v 1.14 2004/03/05 01:32:13 sebb Exp $
/*
 * Copyright 2001-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
*/

package org.apache.jmeter.assertions.gui;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;

import org.apache.jmeter.assertions.SizeAssertion;
import org.apache.jmeter.testelement.TestElement;
import org.apache.jmeter.util.JMeterUtils;
import org.apache.jorphan.gui.layout.VerticalLayout;
import org.apache.jorphan.logging.LoggingManager;
import org.apache.log.Logger;

/**
 * @version   $Revision: 1.14 $ Last updated: $Date: 2004/03/05 01:32:13 $
 */
public class SizeAssertionGui
    extends AbstractAssertionGui
    implements FocusListener, ActionListener
{
    transient private static Logger log = LoggingManager.getLoggerForClass();

    private JTextField size;
    private JRadioButton equalButton,
        notequalButton,
        greaterthanButton,
        lessthanButton,
        greaterthanequalButton,
        lessthanequalButton;
    private int execState; //store the operator 

    public SizeAssertionGui()
    {
        init();
    }

    public String getLabelResource()
    {
        return "size_assertion_title";
    }

    public String getSizeAttributesTitle()
    {
        return JMeterUtils.getResString("size_assertion_size_test");
    }

    public TestElement createTestElement()
    {
        SizeAssertion el = new SizeAssertion();
        modifyTestElement(el);
        return el;
    }

    /**
     * Modifies a given TestElement to mirror the data in the gui components.
     * @see org.apache.jmeter.gui.JMeterGUIComponent#modifyTestElement(TestElement)
     */
    public void modifyTestElement(TestElement el)
    {
        configureTestElement(el);
        String sizeString = size.getText();
        long assertionSize = 0;
        try
        {
            assertionSize = Long.parseLong(sizeString);
        }
        catch (NumberFormatException e)
        {
            assertionSize = Long.MAX_VALUE;
        }
        ((SizeAssertion) el).setAllowedSize(assertionSize);
        ((SizeAssertion) el).setCompOper(getState());
    }

    public void configure(TestElement el)
    {
        super.configure(el);
        SizeAssertion assertion = (SizeAssertion) el;
        size.setText(String.valueOf(assertion.getAllowedSize()));
        setState(assertion.getCompOper());
    }

    /**
     * Set the state of the radio Button
     */
    public void setState(int state)
    {
        if (state == 1)
        {
            equalButton.setSelected(true);
            execState = 1;
        }
        else if (state == 2)
        {
            notequalButton.setSelected(true);
            execState = 2;
        }
        else if (state == 3)
        {
            greaterthanButton.setSelected(true);
            execState = 3;
        }
        else if (state == 4)
        {
            lessthanButton.setSelected(true);
            execState = 4;
        }
        else if (state == 5)
        {
            greaterthanequalButton.setSelected(true);
            execState = 5;
        }
        else if (state == 6)
        {
            lessthanequalButton.setSelected(true);
            execState = 6;
        }
    }

    /**
     * Get the state of the radio Button
     */
    public int getState()
    {
        return execState;
    }
    
    private void init()
    {
        setLayout(
            new VerticalLayout(5, VerticalLayout.LEFT, VerticalLayout.TOP));
        setBorder(makeBorder());

        add(makeTitlePanel());

        // USER_INPUT
        JPanel sizePanel = new JPanel();
        sizePanel.setBorder(
            BorderFactory.createTitledBorder(
                BorderFactory.createEtchedBorder(),
                getSizeAttributesTitle()));

        sizePanel.add(
            new JLabel(JMeterUtils.getResString("size_assertion_label")));
        size = new JTextField(5);
        size.addFocusListener(this);
        sizePanel.add(size);

        sizePanel.add(createComparatorButtonPanel());

        add(sizePanel);
    }

    private Box createComparatorButtonPanel()
    {
        ButtonGroup group = new ButtonGroup();

        equalButton = createComparatorButton("=", SizeAssertion.EQUAL, group);
        notequalButton =
            createComparatorButton("!=", SizeAssertion.NOTEQUAL, group);
        greaterthanButton =
            createComparatorButton(">", SizeAssertion.GREATERTHAN, group);
        lessthanButton =
            createComparatorButton("<", SizeAssertion.LESSTHAN, group);
        greaterthanequalButton =
            createComparatorButton(">=", SizeAssertion.GREATERTHANEQUAL, group);
        lessthanequalButton =
            createComparatorButton("<=", SizeAssertion.LESSTHANEQUAL, group);

        equalButton.setSelected(true);
        execState = Integer.parseInt(equalButton.getActionCommand());

        //Put the check boxes in a column in a panel
        Box checkPanel = Box.createVerticalBox();
        JLabel compareLabel =
            new JLabel(
                JMeterUtils.getResString("size_assertion_comparator_label"));
        checkPanel.add(compareLabel);
        checkPanel.add(equalButton);
        checkPanel.add(notequalButton);
        checkPanel.add(greaterthanButton);
        checkPanel.add(lessthanButton);
        checkPanel.add(greaterthanequalButton);
        checkPanel.add(lessthanequalButton);
        return checkPanel;
    }

    private JRadioButton createComparatorButton(
        String name,
        int value,
        ButtonGroup group)
    {
        JRadioButton button = new JRadioButton(name);
        button.setActionCommand(String.valueOf(value));
        button.addActionListener(this);
        group.add(button);
        return button;
    }

    public void focusLost(FocusEvent e)
    {
        boolean isInvalid = false;
        String sizeString = size.getText();
        if (sizeString != null)
        {
            try
            {
                long assertionSize = Long.parseLong(sizeString);
                if (assertionSize < 0)
                {
                    isInvalid = true;
                }
            }
            catch (NumberFormatException ex)
            {
                isInvalid = true;
            }
            if (isInvalid)
            {
                log.warn("SizeAssertionGui: Not a valid number!");
                JOptionPane.showMessageDialog(
                    null,
                    JMeterUtils.getResString("size_assertion_input_error"),
                    "Error",
                    JOptionPane.ERROR_MESSAGE);
            }
        }
    }

    public void focusGained(FocusEvent e)
    {
    }

    public void actionPerformed(ActionEvent e)
    {
        int comparator = Integer.parseInt(e.getActionCommand());
        execState = comparator;
    }
}
