/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / wave audio render plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */


#include "wav_audio.h"

#define WAVCTX()	WAVContext *ctx = (WAVContext *)dr->opaque;

static Bool WAV_FillBuffer(AudioOutput *dr);

static M4Err WAV_SetupHardware(AudioOutput *dr, void *os_handle, u32 num_buffers, u32 num_buffer_per_sec)
{
	WAVCTX();

	ctx->force_config = (num_buffers && num_buffer_per_sec) ? 1 : 0;
	ctx->cfg_num_buffers = num_buffers;
	if (ctx->cfg_num_buffers <= 1) ctx->cfg_num_buffers = 2;
	ctx->cfg_num_buffer_per_sec = num_buffer_per_sec;
	if (!ctx->force_config) ctx->num_audio_buffer = 6;

	return M4OK;
}

static void CALLBACK WaveProc(HWAVEOUT hwo, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
	AudioOutput *dr = (AudioOutput *) dwInstance;
	WAVCTX();

	if (uMsg == WOM_DONE) {
		SEM_Notify(ctx->sem, 1);
	}
} 


static void close_waveform(AudioOutput *dr)
{
	u32 i;
	Bool not_done;
	MMRESULT res;
	WAVCTX();

	if (ctx->pWAV) {
		/*wait for all buffers to complete, otherwise this locks waveOutReset*/
		while (1) {
			not_done = 0;
			for (i=0 ; i< ctx->num_audio_buffer; i++) {
				if (! (ctx->WaveHdr[i].dwFlags & WHDR_DONE)) {
					not_done = 1;
					break;
				}
			}
			if (!not_done) break;
			Sleep(60);
		}
		/*waveOutReset gives unpredictable results on PocketPC, so just close right away*/
		while (1) {
			res = waveOutClose(ctx->pWAV);
			if (res == MMSYSERR_NOERROR) break;
		}
		ctx->pWAV = NULL;
	}
	if (ctx->wav_buf) free(ctx->wav_buf);
	ctx->wav_buf = NULL;
	if (ctx->sem) SEM_Delete(ctx->sem);
	ctx->sem = NULL;
}

static void WAV_Shutdown(AudioOutput *dr)
{
	close_waveform(dr);
}


/*we assume what was asked is what we got*/
static M4Err WAV_ConfigureOutput(AudioOutput *dr, u32 *SampleRate, u32 *NbChannels, u32 *nbBitsPerSample)
{
	u32 i, retry;
	HRESULT	hr;
	WAVCTX();

	if (!ctx) return M4BadParam;

	/*reset*/
	close_waveform(dr);

	memset (&ctx->format, 0, sizeof(ctx->format));
	ctx->format.cbSize = sizeof(WAVEFORMATEX);
	ctx->format.wFormatTag = WAVE_FORMAT_PCM;
	if (*NbChannels>2) *NbChannels=2;
	ctx->format.nChannels = *NbChannels;
	ctx->format.wBitsPerSample = *nbBitsPerSample;
	ctx->format.nSamplesPerSec = *SampleRate;
	ctx->format.nBlockAlign = ctx->format.wBitsPerSample * ctx->format.nChannels / 8;
	ctx->format.nAvgBytesPerSec = *SampleRate * ctx->format.nBlockAlign; 

	/* Open a waveform device for output using window callback. */ 
	retry = 10;
	while (retry) {		
		hr = waveOutOpen((LPHWAVEOUT)&ctx->pWAV, WAVE_MAPPER, &ctx->format, (DWORD) WaveProc, (DWORD) dr, 
			CALLBACK_FUNCTION | WAVE_ALLOWSYNC | WAVE_FORMAT_DIRECT
			);

		if (hr == MMSYSERR_NOERROR) break;
		/*couldn't open audio*/
		if (hr != MMSYSERR_ALLOCATED) return M4IOErr;
		Sleep(10);
		retry--;
	}
	if (hr != MMSYSERR_NOERROR) return M4IOErr;

	if (!ctx->force_config) {
		/*one wave buffer size*/
		ctx->WaveBufferSize = 1024 * ctx->format.nBlockAlign;
	} else {
		ctx->num_audio_buffer = ctx->cfg_num_buffers;
		ctx->WaveBufferSize = ctx->format.nAvgBytesPerSec / ctx->cfg_num_buffer_per_sec;
	}

	ctx->sem = NewSemaphore(ctx->num_audio_buffer, ctx->num_audio_buffer - 1);

	/*make sure we're aligned*/
	while (ctx->WaveBufferSize % ctx->format.nBlockAlign) ctx->WaveBufferSize++;

	ctx->wav_buf = malloc(ctx->WaveBufferSize*ctx->num_audio_buffer*sizeof(char));
	memset(ctx->wav_buf, 0, ctx->WaveBufferSize*ctx->num_audio_buffer*sizeof(char));

	/*setup wave headers*/
	for (i=0 ; i < ctx->num_audio_buffer; i++) {
		memset(& ctx->WaveHdr[i], 0, sizeof(WAVEHDR));
		ctx->WaveHdr[i].dwBufferLength = ctx->WaveBufferSize; 
		ctx->WaveHdr[i].lpData = & ctx->wav_buf[i*ctx->WaveBufferSize];
		ctx->WaveHdr[i].dwFlags = WHDR_DONE;
		waveOutPrepareHeader(ctx->pWAV, &ctx->WaveHdr[i], sizeof(WAVEHDR));
	}
	ctx->CurrentBuffer = 0;
	ctx->const_delay = 1000 * ctx->num_audio_buffer * ctx->WaveBufferSize / ctx->format.nAvgBytesPerSec;
	return M4OK;
}

static Bool WAV_FillBuffer(AudioOutput *dr)
{
	LPWAVEHDR h;
	WAVCTX();

	/*this is because the waveOut can change on the fly*/
	if (!ctx->pWAV) return 0;

	/*get buffer*/
	h = & ctx->WaveHdr[ctx->CurrentBuffer];
	h->dwBufferLength = ctx->WaveBufferSize;

	/*fill it*/
	dr->FillBuffer(dr->audio_renderer, h->lpData, ctx->WaveBufferSize);

	/*write it*/
	waveOutWrite(ctx->pWAV, h, sizeof(WAVEHDR));

	ctx->CurrentBuffer += 1;
	ctx->CurrentBuffer %= ctx->num_audio_buffer;
	return 1;
}

static void WAV_GotoSleep(AudioOutput *dr)
{
	WAVCTX();
	SEM_Wait(ctx->sem);
}

static void WAV_LockAndWrite(AudioOutput *dr)
{
	/*refill buffers*/
	WAV_FillBuffer(dr);
}

static void WAV_Pause(AudioOutput *dr, Bool DoFreeze)
{
	WAVCTX();
	if (DoFreeze) 
		waveOutPause(ctx->pWAV);
	else 
		waveOutRestart(ctx->pWAV);
}

static void set_vol_pan(WAVContext *ctx)
{
	WORD rV, lV;
	/*in wave, volume & pan are specified as a DWORD. LOW word is LEFT channel, HIGH is right - iPaq doesn't support that*/
	lV = (WORD) (ctx->vol * ctx->pan / 100);
	rV = (WORD) (ctx->vol * (100 - ctx->pan) / 100);

#ifdef _WIN32_WCE
	waveOutSetVolume(0, MAKELONG(lV, rV) );
#else
	waveOutSetVolume(ctx->pWAV, MAKELONG(lV, rV) );
#endif

}

static void WAV_SetVolume(AudioOutput *dr, u32 Volume)
{
	WAVCTX();
	if (Volume > 100) Volume = 100;
	ctx->vol = Volume * 0xFFFF / 100;
	set_vol_pan(ctx);
}

static void WAV_SetPan(AudioOutput *dr, u32 Pan)
{
	WAVCTX();
	if (Pan > 100) Pan = 100;
	ctx->pan = Pan;
	set_vol_pan(ctx);
}

static void WAV_SetPriority(AudioOutput *dr, u32 Priority)
{
	TH_SetPriority(NULL, TH_PRIOR_HIGHEST);
}


static u32 WAV_QueryOutputSampleRate(AudioOutput *dr, u32 desired_samplerate, u32 NbChannels, u32 nbBitsPerSample)
{
	/*iPaq's output frequencies available*/
#ifdef _WIN32_WCE
	switch (desired_samplerate) {
	case 11025:
	case 22050:
		return 22050;
	case 8000:
	case 16000:
	case 32000:
		return 44100;
	case 24000:
	case 48000:
		return 44100;
	default:
		return desired_samplerate;
	}
#else
	return desired_samplerate;
#endif
}

static u32 WAV_GetAudioDelay(AudioOutput *dr)
{
	WAVCTX();
	return ctx->const_delay;
}


void *NewWAVRender()
{
	WAVContext *ctx;
	AudioOutput *driv;
	ctx = malloc(sizeof(WAVContext));
	memset(ctx, 0, sizeof(WAVContext));
	ctx->num_audio_buffer = 10;
	ctx->pan = 50;
	ctx->vol = 100;

//	M4_InitClock();

	driv = malloc(sizeof(AudioOutput));
	memset(driv, 0, sizeof(AudioOutput));
	M4_REG_PLUG(driv, M4_AUDIO_OUTPUT_INTERFACE, "Win32 Wave Output", "gpac distribution", 0)

	driv->opaque = ctx;

	driv->SelfThreaded = 0;
	driv->SetupHardware = WAV_SetupHardware;
	driv->Shutdown = WAV_Shutdown;
	driv->ConfigureOutput = WAV_ConfigureOutput;
	driv->GetAudioDelay = WAV_GetAudioDelay;
	driv->SetVolume = WAV_SetVolume;
	driv->SetPan = WAV_SetPan;
	driv->Pause = WAV_Pause;
	driv->SetPriority = WAV_SetPriority;
	driv->QueryOutputSampleRate = WAV_QueryOutputSampleRate;
	driv->GotoSleep = WAV_GotoSleep;
	driv->LockAndWrite = WAV_LockAndWrite;

	return driv;
}

void DeleteWAVRender(void *ifce)
{
	AudioOutput *dr = (AudioOutput *) ifce;
	WAVContext *ctx = (WAVContext *)dr->opaque;

	//M4_StopClock();

	free(ctx);
	free(dr);
}

Bool QueryInterface(u32 InterfaceType)
{
	if (InterfaceType == M4_AUDIO_OUTPUT_INTERFACE) return 1;
	return 0;
}

void *LoadInterface(u32 InterfaceType)
{
	if (InterfaceType == M4_AUDIO_OUTPUT_INTERFACE) return NewWAVRender();
	return NULL;
}

void ShutdownInterface(void *ifce)
{
	AudioOutput *dr = (AudioOutput *) ifce;
	switch (dr->InterfaceType) {
	case M4_AUDIO_OUTPUT_INTERFACE:
		DeleteWAVRender(dr);
		break;
	}
}
