/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / codec pack plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#ifdef M4_BUILD_FAAD

#include "codec_pack.h"

#include "faad.h"

typedef struct
{
	faacDecHandle codec;
	faacDecFrameInfo info;
	u32 sample_rate, out_size, num_samples;
	u8 num_channels;
	/*no support for scalability in FAAD yet*/
	u16 ES_ID;
	u32 cb_size, cb_trig;
} FAADDec;


#define FAADCTX() FAADDec *ctx = (FAADDec *) ((CodecWraper *)ifcg->privateStack)->opaque

static M4Err FAAD_AttachStream(DecoderInterface *ifcg, u16 ES_ID, unsigned char *decSpecInfo, u32 decSpecInfoSize, u16 DependsOnES_ID, u32 objectTypeIndication, Bool UpStream)
{
	FAADCTX();
	
	if (ctx->ES_ID && ctx->ES_ID!=ES_ID) return M4NotSupported;
	if (!decSpecInfoSize || !decSpecInfo) return M4NonCompliantBitStream;

	if (ctx->codec) faacDecClose(ctx->codec);
	ctx->codec = faacDecOpen();
	if (!ctx->codec) return M4IOErr;

	if (faacDecInit2(ctx->codec, (unsigned char *) decSpecInfo, decSpecInfoSize, (unsigned long *) &ctx->sample_rate, (u8 *) &ctx->num_channels) < 0)
		return M4NonCompliantBitStream;

	ctx->num_samples = 1024;
	ctx->out_size = 2 * ctx->num_samples * ctx->num_channels;
	ctx->ES_ID = ES_ID;
	return M4OK;
}

static M4Err FAAD_DetachStream(DecoderInterface *ifcg, u16 ES_ID)
{
	FAADCTX();
	if (ES_ID != ctx->ES_ID) return M4BadParam;
	if (ctx->codec) faacDecClose(ctx->codec);
	ctx->codec = NULL;
	ctx->ES_ID = 0;
	ctx->sample_rate = ctx->out_size = ctx->num_samples = 0;
	ctx->num_channels = 0;
	return M4OK;
}
static M4Err FAAD_GetCapabilities(DecoderInterface *ifcg, CapObject *capability)
{
	FAADCTX();
	switch (capability->CapCode) {
	/*not tested yet*/
	case CAP_HASRESILIENCE:
		capability->cap.valueINT = 1;
		break;
	case CAP_OUTPUTSIZE:
		capability->cap.valueINT = ctx->out_size;
		break;
	case CAP_SAMPLERATE:
		capability->cap.valueINT = ctx->sample_rate;
		break;
	case CAP_NBCHANNELS:
		capability->cap.valueINT = ctx->num_channels;
		break;
	case CAP_BITSPERSAMPLE:
		capability->cap.valueINT = 16;
		break;
	case CAP_BUFFER_MIN:
		capability->cap.valueINT = ctx->cb_trig;
		break;
	case CAP_BUFFER_MAX:
		capability->cap.valueINT = ctx->cb_size;
		break;
	/*by default AAC access unit lasts num_samples (timescale being sampleRate)*/
	case CAP_CU_DURATION:
		capability->cap.valueINT = ctx->num_samples;
		break;
	/*to refine, it seems that 4 bytes padding is not enough on all streams ?*/
	case CAP_PADDING_BYTES:
		capability->cap.valueINT = 8;
		break;
	default:
		capability->cap.valueINT = 0;
		break;
	}
	return M4OK;
}
static M4Err FAAD_SetCapabilities(DecoderInterface *ifcg, CapObject capability)
{
	/*return unsupported to avoid confusion by the player (like SR changing ...) */
	return M4NotSupported;
}
static M4Err FAAD_Process(DecoderInterface *ifcg, 
		unsigned char *inBuffer, u32 inBufferLength,
		u16 ES_ID,
		unsigned char *outBuffer, u32 *outBufferLength,
		Bool isRAP, u8 PaddingBits, u32 mmlevel)
{
	void *buffer;
	FAADCTX();

	/*check not using scalabilty*/
	if (ctx->ES_ID != ES_ID) return M4BadParam;

	/*if late or seeking don't decode*/
	switch (mmlevel) {
	case MM_LEVEL_SEEK:
	case MM_LEVEL_DROP:
		*outBufferLength = 0;
		return M4OK;
	default:
		break;
	}

	if (ctx->out_size > *outBufferLength) {
		*outBufferLength = ctx->out_size;
		return M4BufferTooSmall;
	}

	buffer = faacDecDecode(ctx->codec, &ctx->info, inBuffer, inBufferLength);

	if (ctx->info.error>0) {
		*outBufferLength = 0;
		return M4NonCompliantBitStream;
	}

	if (!ctx->info.samples || !buffer || !ctx->info.bytesconsumed) {
		*outBufferLength = 0;
		return M4OK;
	}

	if (sizeof(short) * ctx->info.samples > *outBufferLength) {
		*outBufferLength = sizeof(short)*ctx->info.samples; 
		return M4BufferTooSmall;
	} 

	memcpy(outBuffer, buffer, sizeof(short)* ctx->info.samples);
	*outBufferLength = sizeof(short)*ctx->info.samples;
	return M4OK;
}

static const char *FAAD_GetCodecName(DecoderInterface *dec)
{
	return "FAAD2 " FAAD2_VERSION;
}

u32 NewFAADDec(DecoderInterface *ifcd)
{
	CodecWraper *wrap = (CodecWraper *) ifcd->privateStack;
	FAADDec *dec = (FAADDec *) malloc(sizeof(FAADDec));
	memset(dec, 0, sizeof(FAADDec));
	wrap->opaque = dec;
	wrap->type = DEC_FAAD;

	dec->cb_size = DEFAULT_AUDIO_CM_SIZE;
	dec->cb_trig = DEFAULT_AUDIO_CM_TRIGGER;

	/*setup our own interface*/	
	ifcd->Codec_AttachStream = FAAD_AttachStream;
	ifcd->Codec_DetachStream = FAAD_DetachStream;
	ifcd->Codec_GetCapabilities = FAAD_GetCapabilities;
	ifcd->Codec_SetCapabilities = FAAD_SetCapabilities;
	ifcd->Codec_Process = FAAD_Process;
	ifcd->GetCodecName = FAAD_GetCodecName;
	return 1;
}

void DeleteFAADDec(DecoderInterface *ifcg)
{
	FAADCTX();
	if (ctx->codec) faacDecClose(ctx->codec);
	free(ctx);
}


#endif

