/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / codec pack plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "codec_pack.h"
#include <gpac/m4_network.h>


static Bool DEC_CanHandleStream(DecoderInterface *dec, u32 StreamType, u32 ObjectType, unsigned char *decSpecInfo, u32 decSpecInfoSize, u32 PL)
{
	CodecWraper *wrap = (CodecWraper *)dec->privateStack;

	/*we handle audio/video */
	if (!ObjectType) {
		if (StreamType==M4ST_VISUAL) {
#if defined(M4_BUILD_PNG) || defined(M4_BUILD_JPEG) || defined(M4_BUILD_XVID)
		return 1;
#else
		return 0;
#endif
		}
		if (StreamType==M4ST_AUDIO) {
#if defined(M4_BUILD_MAD) || defined(M4_BUILD_FAAD)
		return 1;
#else
		return 0;
#endif
		}
		return 0;	}

	/*should never happen*/
	assert(wrap->type == DEC_RESERVED);

	/*video decs*/
	if (StreamType == M4ST_VISUAL) {
		switch (ObjectType) {
#ifdef M4_BUILD_PNG
		case 0x6D:
			return NewPNGDec(dec);
#endif

#ifdef M4_BUILD_JPEG
		case 0x6C:
			return NewJPEGDec(dec);
#endif

#ifdef M4_BUILD_XVID
		case 0x20:
			return NewXVIDDec(dec);
#endif

		default:
			return 0;
		}
	}
	/*audio decs*/	
	if (StreamType == M4ST_AUDIO) {
		switch (ObjectType) {
#ifdef M4_BUILD_FAAD
		/*MPEG2 aac*/
		case 0x66:
		case 0x67:
		case 0x68:
		/*MPEG4 aac*/
		case 0x40:
			return NewFAADDec(dec);
#endif

#ifdef M4_BUILD_MAD
		case 0x69:
		case 0x6B:
			return NewMADDec(dec);
#endif

		default:
			return 0;
		}
	}
	return 0;
}


DecoderInterface *NewDecoderInterface()
{
	DecoderInterface *ifce;
	CodecWraper *wrap;
	ifce = malloc(sizeof(DecoderInterface));
	memset(ifce, 0, sizeof(DecoderInterface));
	wrap = malloc(sizeof(CodecWraper));
	memset(wrap, 0, sizeof(CodecWraper));
	ifce->privateStack = wrap;
	ifce->CanHandleStream = DEC_CanHandleStream;

	M4_REG_PLUG(ifce, M4MEDIADECODERINTERFACE, "GPAC Decoder package", "gpac distribution", 0)

	/*other interfaces will be setup at run time*/
	return ifce;
}

void DeleteDecoderInterface(DecoderInterface *ifcd)
{
	CodecWraper *wrap = (CodecWraper *)ifcd->privateStack;
	switch (wrap->type) {
#ifdef M4_BUILD_PNG
	case DEC_PNG:
		DeletePNGDec(ifcd);
		break;
#endif

#ifdef M4_BUILD_JPEG
	case DEC_JPEG:
		DeleteJPEGDec(ifcd);
		break;
#endif

#ifdef M4_BUILD_MAD
	case DEC_MAD:
		DeleteMADDec(ifcd);
		break;
#endif

#ifdef M4_BUILD_FAAD
	case DEC_FAAD:
		DeleteFAADDec(ifcd);
		break;
#endif

#ifdef M4_BUILD_XVID
	case DEC_XVID:
		DeleteXVIDDec(ifcd);
		break;
#endif

	default:
		break;
	}
	free(wrap);
	free(ifcd);
}

Bool QueryInterface(u32 InterfaceType)
{
	switch (InterfaceType) {
	case M4MEDIADECODERINTERFACE:
		return 1;
	case M4STREAMINGCLIENT:
		return 1;
	default:
		return 0;
	}
}

void *LoadInterface(u32 InterfaceType)
{
	switch (InterfaceType) {
	case M4MEDIADECODERINTERFACE:
		return NewDecoderInterface();
	case M4STREAMINGCLIENT:
		return NewLoaderInterface();
	default:
		return NULL;
	}
}

void ShutdownInterface(void *ifce)
{
	DecoderInterface *ifcd = (DecoderInterface *)ifce;
	switch (ifcd->InterfaceType) {
	case M4MEDIADECODERINTERFACE:
		DeleteDecoderInterface(ifcd);
		break;
	case M4STREAMINGCLIENT:
		DeleteLoaderInterface(ifce);
		break;
	}
}
