##############################################################################
#
# Copyright (c) 2004 Zope Corporation and Contributors.
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################
"""A WSGI Application wrapper for zope

$Id: __init__.py 29956 2005-04-12 20:34:35Z srichter $
"""
from zope.interface import implements
from zope.publisher.publish import publish
from zope.publisher.interfaces.http import IHeaderOutput

from zope.app.publication.httpfactory import HTTPPublicationRequestFactory
from zope.app.wsgi import interfaces

class WSGIOutput(object):
    """This class handles the output generated by the publisher. It is used to
    collect the headers and as an outstream to output the response body.
    """
    implements(interfaces.IWSGIOutput)

    def __init__(self, start_response):
        self._headers = {}
        self._accumulatedHeaders = []
        self._statusString = ""
        self._headersSent = False

        self.wsgi_write = None
        self.start_response = start_response

    def setResponseStatus(self, status, reason):
        """See zope.publisher.interfaces.http.IHeaderOutput"""
        self._statusString = str(status) + ' ' + reason

    def setResponseHeaders(self, mapping):
        """See zope.publisher.interfaces.http.IHeaderOutput"""
        self._headers.update(mapping)

    def appendResponseHeaders(self, lst):
        """See zope.publisher.interfaces.http.IHeaderOutput"""
        self._accumulatedHeaders.extend(lst)

    def wroteResponseHeader(self):
        """See zope.publisher.interfaces.http.IHeaderOutput"""
        return self._headersSent

    def setAuthUserName(self, name):
        """See zope.publisher.interfaces.http.IHeaderOutput"""
        pass

    def getHeaders(self):
        """See zope.app.wsgi.interfaces.IWSGIOutput"""
        response_headers = self._headers.items()

        accum = [tuple(line.split(':', 1))
                 for line in self._accumulatedHeaders]
            
        response_headers.extend(accum)
        return response_headers

    def write(self, data):
        """See zope.app.wsgi.interfaces.IWSGIOutput"""
        if not self._headersSent:
            self.wsgi_write = self.start_response(self._statusString,
                                                  self.getHeaders())
            self._headersSent = True

        self.wsgi_write(data)


class WSGIPublisherApplication(object):
    """A WSGI application implemenation for the zope publisher

    Instances of this class can be used as a WSGI application object.

    The class relies on a properly initialized request factory.
    """
    implements(interfaces.IWSGIApplication)
    
    def __init__(self, db=None, factory=HTTPPublicationRequestFactory):
        self.requestFactory = None

        if db is not None:
            self.requestFactory = factory(db)

    def __call__(self, environ, start_response):
        """See zope.app.wsgi.interfaces.IWSGIApplication"""
        # wsgiOutput has two purposes: (1) it is the response headers output
        # manager and (2) it functions as the output stream for the publisher. 
        wsgiOutput = WSGIOutput(start_response)

        request = self.requestFactory(
            environ['wsgi.input'], wsgiOutput, environ)

        request.response.setHeaderOutput(wsgiOutput)

        publish(request)

        # since the response is written using the WSGI ``write()`` callable
        # return an empty iterable (see PEP 333).
        return ""
