/*
* Copyright (C) 2012 Canonical Ltd
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License version 3 as
* published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*/

#include "gtest/gtest.h"

#include "parser.h"
#include "xpathquerypart.h"

#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/phoenix_core.hpp>
#include <boost/spirit/include/qi_char_class.hpp>
#include <boost/spirit/include/phoenix_operator.hpp>
#include <iostream>
#include <string>
#include <cstdlib>

namespace qi = boost::spirit::qi;
namespace ascii = boost::spirit::ascii;

namespace parser = xpathselect::parser;


// utility function to test parsers:
template <typename P, typename T>
bool test_parser_attr(std::string input, P const& p, T& attr)
{
    using boost::spirit::qi::parse;

    std::string::iterator f = input.begin();
    std::string::iterator l = input.end();
    if (parse(f, l, p, attr) && (f == l))
    {
        return true;
    }
    else
    {
        return false;
    }
}

template <typename P>
bool test_parser_attr(std::string input, P const& p)
{
    using boost::spirit::qi::parse;

    std::string::iterator f = input.begin();
    std::string::iterator l = input.end();
    if (parse(f, l, p) && (f == l))
    {
        return true;
    }
    else
    {
        return false;
    }
}

class TestXPathParserNodeNames : public ::testing::TestWithParam<std::pair<std::string, bool> >
{
};


TEST_P(TestXPathParserNodeNames, test_spec_node_name)
{
    auto p = GetParam();

    std::string input = p.first;
    bool expect_pass = p.second;

    parser::xpath_grammar<std::string::iterator> g;

    std::string result;
    ASSERT_EQ( expect_pass,  test_parser_attr(input, g.spec_node_name, result) );
    if (expect_pass)
        ASSERT_EQ(input, result);
}

INSTANTIATE_TEST_CASE_P(BasicNodeNames,
                        TestXPathParserNodeNames,
                        ::testing::Values(
                            std::pair<std::string, bool>("a b", true),
                            std::pair<std::string, bool>("a ", false),
                            std::pair<std::string, bool>(" ", false),
                            std::pair<std::string, bool>(" b", false),
                            std::pair<std::string, bool>("a    b", true),
                            std::pair<std::string, bool>("a b b a", true),
                            std::pair<std::string, bool>("a*", false),
                            std::pair<std::string, bool>("HelloWorld", true),
                            std::pair<std::string, bool>("H", true),
                            std::pair<std::string, bool>("h", true),
                            std::pair<std::string, bool>("1", true),
                            std::pair<std::string, bool>("node-name", true),
                            std::pair<std::string, bool>("node_name", true),
                            std::pair<std::string, bool>("node\\name", true)
                         ));

class TestXPathParserParamNames : public ::testing::TestWithParam<std::pair<std::string, bool> >
{
};

TEST_P(TestXPathParserParamNames, test_param_name)
{
    auto p = GetParam();

    std::string input = p.first;
    bool expect_pass = p.second;

    parser::xpath_grammar<std::string::iterator> g;

    std::string result;
    ASSERT_EQ( expect_pass,  test_parser_attr(input, g.param_name, result) );
    if (expect_pass)
        ASSERT_EQ(input, result);
}

INSTANTIATE_TEST_CASE_P(BasicNodeNames,
                        TestXPathParserParamNames,
                        ::testing::Values(
                            std::pair<std::string, bool>("a b", false),
                            std::pair<std::string, bool>("a*", false),
                            std::pair<std::string, bool>("HelloWorld", true),
                            std::pair<std::string, bool>("H", true),
                            std::pair<std::string, bool>("h", true),
                            std::pair<std::string, bool>("1", true),
                            std::pair<std::string, bool>("node-name", true),
                            std::pair<std::string, bool>("node_name", true),
                            std::pair<std::string, bool>("node\\name", true),
                            std::pair<std::string, bool>("node.name", false),
                            std::pair<std::string, bool>("node name", false)
                         ));

class TestXPathParserParamValues : public ::testing::TestWithParam<std::pair<std::string, bool> >
{
};

TEST_P(TestXPathParserParamValues, test_param_value)
{
    auto p = GetParam();

    std::string input = p.first;
    bool expect_pass = p.second;

    parser::xpath_grammar<std::string::iterator> g;

    std::string result;
    ASSERT_EQ( expect_pass,  test_parser_attr(input, g.param_value, result) );
    if (expect_pass)
        ASSERT_EQ(input, result);
}

INSTANTIATE_TEST_CASE_P(BasicNodeNames,
                        TestXPathParserParamValues,
                        ::testing::Values(
                            std::pair<std::string, bool>("a b", true),
                            std::pair<std::string, bool>("a*", false),
                            std::pair<std::string, bool>("HelloWorld", true),
                            std::pair<std::string, bool>("H", true),
                            std::pair<std::string, bool>("h", true),
                            std::pair<std::string, bool>("1", true),
                            std::pair<std::string, bool>("node-name", true),
                            std::pair<std::string, bool>("node_name", true),
                            std::pair<std::string, bool>("node\\name", true),
                            std::pair<std::string, bool>("node name", true),
                            std::pair<std::string, bool>("node.name", true)
                         ));


TEST(TestXPathParser, test_wildcard_node_name_accepts_wildcards)
{
    std::string input("*");

    parser::xpath_grammar<std::string::iterator> g;

    std::string result;
    ASSERT_EQ( true,  test_parser_attr(input, g.wildcard_node_name, result) );
    ASSERT_EQ(input, result);
}

class TestXPathParserWildcardNodeName : public ::testing::TestWithParam<std::pair<std::string, bool> >
{
};

TEST_P(TestXPathParserWildcardNodeName, test_wildcard_node_name_rejects_everything_else)
{
    auto p = GetParam();

    std::string input = p.first;
    bool expect_pass = p.second;

    parser::xpath_grammar<std::string::iterator> g;

    std::string result;
    ASSERT_EQ( expect_pass,  test_parser_attr(input, g.wildcard_node_name, result) );
    if (expect_pass)
        ASSERT_EQ(input, result);
}

INSTANTIATE_TEST_CASE_P(BasicNodeNames,
                        TestXPathParserWildcardNodeName,
                        ::testing::Values(
                            std::pair<std::string, bool>("", false),
                            std::pair<std::string, bool>("* ", false),
                            std::pair<std::string, bool>("**", false),
                            std::pair<std::string, bool>(" ", false),
                            std::pair<std::string, bool>("8", false),
                            std::pair<std::string, bool>("\t", false),
                            std::pair<std::string, bool>("node-name", false),
                            std::pair<std::string, bool>("node_name", false),
                            std::pair<std::string, bool>("*", true)
                         ));


TEST(TestXPathParser, test_param_parser_works)
{
    std::string input("name=value");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryParam result;
    ASSERT_EQ( true,  test_parser_attr(input, g.param, result) );
    ASSERT_EQ("name", result.param_name);
    ASSERT_EQ("value", result.param_value);
}

TEST(TestXPathParser, test_param_parser_fails)
{
    std::string input("name=");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryParam result;
    ASSERT_FALSE( test_parser_attr(input, g.param, result) );
}

TEST(TestXPathParser, test_param_list_single_value)
{
    std::string input("[name=123]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::ParamList result;
    ASSERT_EQ( true,  test_parser_attr(input, g.param_list, result) );
    ASSERT_EQ(1, result.size());
    ASSERT_EQ("name", result.at(0).param_name);
    ASSERT_EQ("123", result.at(0).param_value);
}

TEST(TestXPathParser, test_param_list_two_values)
{
    std::string input("[name=value,other=foo]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::ParamList result;
    ASSERT_EQ( true,  test_parser_attr(input, g.param_list, result) );
    ASSERT_EQ(2, result.size());
    ASSERT_EQ("name", result.at(0).param_name);
    ASSERT_EQ("value", result.at(0).param_value);
    ASSERT_EQ("other", result.at(1).param_name);
    ASSERT_EQ("foo", result.at(1).param_value);
}

TEST(TestXPathParser, test_spec_node_with_parameter)
{
    std::string input("node_name[param_name=123]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.spec_node, result) );
    ASSERT_EQ("node_name", result.node_name_);
    ASSERT_FALSE(result.parameter.empty());
    ASSERT_EQ("param_name", result.parameter.at(0).param_name);
    ASSERT_EQ("123", result.parameter.at(0).param_value);
}

TEST(TestXPathParser, test_spec_node_without_parameter)
{
    std::string input("node_name");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.spec_node, result) );
    ASSERT_EQ("node_name", result.node_name_);
    ASSERT_TRUE(result.parameter.empty());
}

TEST(TestXPathParser, test_wildcard_node)
{
    std::string input("*");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.wildcard_node, result) );
    ASSERT_EQ("*", result.node_name_);
    ASSERT_TRUE(result.parameter.empty());
}

TEST(TestXPathParser, test_wildcard_node_rejects_parameters)
{
    std::string input("*[foo=bar]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_FALSE( test_parser_attr(input, g.wildcard_node, result) );
}

TEST(TestXPathParser, test_wildcard_node_with_params)
{
    std::string input("*[param_name=123]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.wildcard_node_with_params, result) );
    ASSERT_EQ("*", result.node_name_);
    ASSERT_FALSE(result.parameter.empty());
    ASSERT_EQ("param_name", result.parameter.at(0).param_name);
    ASSERT_EQ("123", result.parameter.at(0).param_value);
}


TEST(TestXPathParser, test_node_can_be_a_wildcard_node_with_params)
{
    std::string input("*[name=value]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.node, result) );
    ASSERT_EQ( "*", result.node_name_ );
    ASSERT_EQ( xpathselect::XPathQueryPart::QueryPartType::Normal, result.Type() );
    ASSERT_EQ( 1, result.parameter.size() );
    ASSERT_EQ( "name", result.parameter.at(0).param_name );
    ASSERT_EQ( "value", result.parameter.at(0).param_value );
}

TEST(TestXPathParser, test_node_can_be_a_wildcard_node_without_params)
{
    std::string input("*");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.node, result) );
    ASSERT_EQ( "*", result.node_name_ );
    ASSERT_EQ( xpathselect::XPathQueryPart::QueryPartType::Normal, result.Type() );
}

TEST(TestXPathParser, test_node_can_be_a_spec_node_with_params)
{
    std::string input("foo[name=value]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.node, result) );
    ASSERT_EQ( "foo", result.node_name_ );
    ASSERT_EQ( xpathselect::XPathQueryPart::QueryPartType::Normal, result.Type() );
    ASSERT_EQ( 1, result.parameter.size() );
    ASSERT_EQ( "name", result.parameter.at(0).param_name );
    ASSERT_EQ( "value", result.parameter.at(0).param_value );
}

TEST(TestXPathParser, test_node_can_be_a_spec_node_without_params)
{
    std::string input("foo");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_EQ( true,  test_parser_attr(input, g.node, result) );
    ASSERT_EQ( "foo", result.node_name_ );
    ASSERT_EQ( xpathselect::XPathQueryPart::QueryPartType::Normal, result.Type() );
}

TEST(TestXPathParser, test_search_node_followed_by_normal_node)
{
    // the search_node grammar fails if it's at the end of the line, so we need
    // to give it some more data, even though we're not actually matching it.
    std::string input("//node_name");
    parser::xpath_grammar<std::string::iterator> g;
    xpathselect::XPathQueryPart result;

    // however, this means we can't use the test_parser_attr function, since it
    // returns false on a partial match. Use the parse(...) function directly:
    ASSERT_TRUE( parse(input.begin(), input.end(),g.search_node, result) );
    ASSERT_TRUE( result.Type() == xpathselect::XPathQueryPart::QueryPartType::Search );
}

TEST(TestXPathParser, test_search_node_followed_by_wildcard_node_with_parameters)
{
    // the search_node grammar fails if it's at the end of the line, so we need
    // to give it some more data, even though we're not actually matching it.
    std::string input("//*[foo=bar]");
    parser::xpath_grammar<std::string::iterator> g;
    xpathselect::XPathQueryPart result;

    // however, this means we can't use the test_parser_attr function, since it
    // returns false on a partial match. Use the parse(...) function directly:
    ASSERT_TRUE( parse(input.begin(), input.end(),g.search_node, result) );
    ASSERT_TRUE( result.Type() == xpathselect::XPathQueryPart::QueryPartType::Search );
}

TEST(TestXPathParser, test_search_node_cannot_have_parameters)
{
    std::string input("//[param_name=value]");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::XPathQueryPart result;
    ASSERT_FALSE( test_parser_attr(input, g.search_node, result) );
}

TEST(TestXPathParser, test_normal_sep_works)
{
    std::string input("/");

    parser::xpath_grammar<std::string::iterator> g;

    ASSERT_EQ( true,  test_parser_attr(input, g.normal_sep) );
}

TEST(TestXPathParser, test_normal_sep_does_not_match_search_node)
{
    std::string input("//");

    parser::xpath_grammar<std::string::iterator> g;

    ASSERT_FALSE( test_parser_attr(input, g.normal_sep) );
}

TEST(TestXPathParser, test_can_extract_query_list)
{
    std::string input("/node1/node2");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::QueryList result;
    ASSERT_TRUE(test_parser_attr(input, g.node_sequence, result));
    ASSERT_EQ(2, result.size());
    ASSERT_EQ("node1", result.at(0).node_name_);
    ASSERT_TRUE(result.at(0).parameter.empty());
    ASSERT_EQ("node2", result.at(1).node_name_);
    ASSERT_TRUE(result.at(1).parameter.empty());
}

TEST(TestXPathParser, test_can_extract_query_list_with_search)
{
    std::string input("//node1");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::QueryList result;
    ASSERT_TRUE(test_parser_attr(input, g.node_sequence, result));
    ASSERT_EQ(2, result.size());
    ASSERT_TRUE(result.at(0).Type() == xpathselect::XPathQueryPart::QueryPartType::Search );
    ASSERT_TRUE(result.at(1).Type() == xpathselect::XPathQueryPart::QueryPartType::Normal );
    ASSERT_TRUE(result.at(0).parameter.empty());
    ASSERT_EQ("node1", result.at(1).node_name_);
    ASSERT_TRUE(result.at(1).parameter.empty());
}

TEST(TestXPathParser, test_mix_search_and_normal)
{
    std::string input("/node1//node2");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::QueryList result;
    ASSERT_TRUE(test_parser_attr(input, g.node_sequence, result));

    ASSERT_EQ(3, result.size());

    ASSERT_EQ("node1", result.at(0).node_name_);
    ASSERT_TRUE(result.at(0).Type() == xpathselect::XPathQueryPart::QueryPartType::Normal );
    ASSERT_TRUE(result.at(0).parameter.empty());

    ASSERT_TRUE(result.at(1).Type() == xpathselect::XPathQueryPart::QueryPartType::Search );
    ASSERT_TRUE(result.at(1).parameter.empty());

    ASSERT_EQ("node2", result.at(2).node_name_);
    ASSERT_TRUE(result.at(2).Type() == xpathselect::XPathQueryPart::QueryPartType::Normal );
    ASSERT_TRUE(result.at(2).parameter.empty());
}

TEST(TestXPathParser, test_mix_search_and_long_normal)
{
    std::string input("/node1//node2[name=val]/node3");

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::QueryList result;
    ASSERT_TRUE(test_parser_attr(input, g.node_sequence, result));

    ASSERT_EQ(4, result.size());

    ASSERT_EQ("node1", result.at(0).node_name_);
    ASSERT_TRUE(result.at(0).parameter.empty());

    ASSERT_TRUE(result.at(1).Type() == xpathselect::XPathQueryPart::QueryPartType::Search );
    ASSERT_TRUE(result.at(1).parameter.empty());

    ASSERT_EQ("node2", result.at(2).node_name_);
    ASSERT_EQ(1, result.at(2).parameter.size());
    ASSERT_EQ("name", result.at(2).parameter.at(0).param_name);
    ASSERT_EQ("val", result.at(2).parameter.at(0).param_value);

    ASSERT_EQ("node3", result.at(3).node_name_);
    ASSERT_TRUE(result.at(3).parameter.empty());
}


class TestXPathParserQueryStrings : public ::testing::TestWithParam<std::pair<std::string, bool> >
{
};


TEST_P(TestXPathParserQueryStrings, test_query_acceptance)
{
    auto p = GetParam();

    std::string input = p.first;
    bool expect_pass = p.second;

    parser::xpath_grammar<std::string::iterator> g;

    xpathselect::QueryList result;
    ASSERT_EQ( expect_pass,  test_parser_attr(input, g, result) );
}

INSTANTIATE_TEST_CASE_P(BasicNodeNames,
                        TestXPathParserQueryStrings,
                        ::testing::Values(
                            // queries that must all parse correctly:
                            std::pair<std::string, bool>("//root", true),
                            std::pair<std::string, bool>("/root", true),
                            std::pair<std::string, bool>("/root/node1", true),
                            std::pair<std::string, bool>("/root//node1", true),
                            std::pair<std::string, bool>("//root", true),
                            std::pair<std::string, bool>("/root//node1/node2", true),
                            std::pair<std::string, bool>("/root[p=1]//node1[p=2]/node3", true),
                            std::pair<std::string, bool>("/root[p=1,n=2,d=e3]", true),
                            std::pair<std::string, bool>("//root[p=1,n=2,d=e3]", true),
                            std::pair<std::string, bool>("/Root//*[p=1]", true),
                            std::pair<std::string, bool>("/Root//*[p=1,v=sj,c=true]", true),
                            // queries that must not parse correctly:
                            std::pair<std::string, bool>("//", false),
                            std::pair<std::string, bool>("/root//", false),
                            std::pair<std::string, bool>("/root///", false),
                            std::pair<std::string, bool>("/ /", false),
                            std::pair<std::string, bool>("", false),
                            std::pair<std::string, bool>(" ", false),
                            std::pair<std::string, bool>("//*", false),
                            std::pair<std::string, bool>("/Root///Leaf", false),
                            std::pair<std::string, bool>("/Root////", false)
                         ));
