/*
 *  Xarchiver
 *
 *  Copyright (C) 2005 Giuseppe Torelli - Colossus
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "main.h"

extern GList *ArchiveType;
extern GList *ArchiveSuffix;
extern gboolean file_to_open;
extern gchar *tmp;

gchar *CurrentFolder = NULL;
GList *Suffix , *Name;

void WaitExitStatus ( GPid child_pid , gchar *temp_file)
{
    int status;
    int waiting = TRUE;
    int ps;

    OffDeleteandViewButtons();
    while (waiting)
    {
        ps = waitpid ( (pid_t)child_pid, &status, WNOHANG);
        if (ps < 0) waiting = FALSE;
        else 
        {
            while (gtk_events_pending())
                gtk_main_iteration();
        }
    }
    OffTooltipPadlock();

    if ( WIFEXITED (status) )
	{
		if ( WEXITSTATUS (status) )
		{
            Update_StatusBar ( _("Operation failed."));
            gtk_tooltips_disable ( pad_tooltip );
            gtk_widget_hide ( pad_image );
			archive_error = TRUE;
			SetButtonState (1,1,0,0,0);
			gtk_window_set_title ( GTK_WINDOW (MainWindow) , "Xarchiver " VERSION );
			response = ShowGtkMessageDialog (GTK_WINDOW
			(MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_QUESTION,GTK_BUTTONS_YES_NO,_("An error occurred while accessing the archive.\nDo you want to view the shell output ?") );
			if (response == GTK_RESPONSE_YES) ShowShellOutput (NULL,FALSE);
			return;
		}
	}
    archive_error = FALSE;
    //temp_file contains the filename created in /tmp to add/delete on tar.bzip2 / tar.gzip archives (look in bzip2.c:313 )
    if ( temp_file != NULL)
    {
        unlink ( temp_file );
        g_free (temp_file);
    }
	//This to automatically reload the content of the archive after adding or deleting
	if (action == add || action == delete)
	{
        Update_StatusBar ( _("Please wait while the content of the archive is being updated..."));
        RemoveColumnsListStore();
        action = inactive;
		switch ( CurrentArchiveType )
		{
			//case 0:
			//Bzip2: not needed
			
			//case 1:
			//Gzip: not needed
			
			case 2:
			OpenRar ( FALSE , path );
			break;

			case 3:
			OpenTar ( FALSE , path );
			break;

            case 4:
            OpenBzip2 ( FALSE , path );
            break;

            case 5:
            OpenGzip ( FALSE , path );
            break;

            case 6:
			case 7:
			OpenZip ( FALSE , path );
			break;

            //case 8:
            //RPM: only open and extract

            case 9:
            Open7Zip ( FALSE, path );
            break;

            case 10:
            //This to avoid generating an error when reloading an unexistent archive if the user
            //deletes the whole content since arj deletes it instead of remaining it empty
            if (g_file_test ( path , G_FILE_TEST_EXISTS) == TRUE) OpenArj ( FALSE, path );
                else Update_StatusBar ( _("Operation successfully completed."));
            break;
		}
        return;
	}
    //This to disable the Add and Delete buttons in case a bzip2 / gzip file has been decompressed
    if ( ! bz_gz ) SetButtonState ( 1,1,1,1,1 );
        else SetButtonState (1,1,0,0,0);
    //Let's restore the liststore
    if ( CurrentArchiveType != 0 && CurrentArchiveType != 1)
    {
        gtk_tree_view_set_model (GTK_TREE_VIEW(treeview1), model);
        g_object_unref(model);
    }
    if ( password != NULL || PasswordProtectedArchive )
    {
        gtk_widget_show ( pad_image );
        gtk_tooltips_enable ( pad_tooltip );
    }
    else
    {
        gtk_tooltips_disable ( pad_tooltip );
        gtk_widget_hide ( pad_image );
    }
    //This to enable only the Extract button in case an RPM was opened
    if (CurrentArchiveType == 8) SetButtonState (1,1,0,0,1);
    
    if ( CurrentArchiveType == 2 || CurrentArchiveType == 6 || CurrentArchiveType == 7 || CurrentArchiveType == 9 || CurrentArchiveType == 10) gtk_widget_set_sensitive ( add_pwd , TRUE );
     else gtk_widget_set_sensitive ( add_pwd , FALSE );
    gtk_window_set_title ( GTK_WINDOW (MainWindow) , path );
    Update_StatusBar ( _("Operation successfully completed."));
}

void on_new1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	path = Show_File_Dialog ( 1 , "new" );
	if (path == NULL) return;
	if ( g_file_test ( path , G_FILE_TEST_EXISTS ) )
	{
		response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_QUESTION,
						GTK_BUTTONS_YES_NO,
						_("This archive already exists. Do you want to overwrite it ?")
						);
		if (response != GTK_RESPONSE_YES)
		{
			g_free (path);
			return;
		}
        //The following to avoid the archivers updates the archive instead of adding to it since the filename exists
        unlink ( path );
	}
    EmptyTextBuffer();
    PasswordProtectedArchive = FALSE;
    //Let's off the delete and view buttons and the menu entries to avoid misterious behaviour
    OffDeleteandViewButtons ();
    //This to delete a possible RPM file from /tmp since we don't need it anymore
    if ( tmp != NULL )
    {
        unlink (tmp);
        tmp = NULL;
    }
	if ( liststore != NULL ) RemoveColumnsListStore();
	gtk_window_set_title ( GTK_WINDOW ( MainWindow) , path );
	if ( CurrentArchiveType == 0 || CurrentArchiveType == 1 )
    {
        bz_gz = TRUE;
        SetButtonState (1,1,1,0,0 );
    }
    else
    {
        SetButtonState (1,1,1,1,0 );
        bz_gz = FALSE;
    }
    if ( CurrentArchiveType == 2 || CurrentArchiveType == 6 || CurrentArchiveType == 7 || CurrentArchiveType == 9 || CurrentArchiveType == 10 )
    {
        gtk_widget_set_sensitive ( add_pwd , TRUE );
        Update_StatusBar ( _("Choose Action->Set Password to create a password protected archive."));
    }
    else
    {
        gtk_widget_set_sensitive ( add_pwd , FALSE );
        Update_StatusBar ( _("Choose Add File or Add Folder to create the archive."));
    }
    gtk_tooltips_disable ( pad_tooltip );
    gtk_widget_hide ( pad_image );
    if (password != NULL) g_free (password);
    password = NULL;
}

int ShowGtkMessageDialog ( GtkWindow *window, int mode,int type,int button, gchar *message)
{
	dialog = gtk_message_dialog_new (window, mode, type, button,message);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_YES);
	response = gtk_dialog_run (GTK_DIALOG (dialog) );
	gtk_widget_destroy (GTK_WIDGET (dialog) );
	return response;
}

void on_open1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	if ( ! file_to_open )
	{
		path = Show_File_Dialog ( 1 , "open" );
		if (path == NULL) return;
	}
	file_to_open = FALSE;
	if ( liststore != NULL )
	{
		RemoveColumnsListStore();
		EmptyTextBuffer ();
	}
    //Did the user previously open an RPM file ? Delete it from /tmp since we don't need it anymore
    if ( CurrentArchiveType == 8 ) unlink (tmp);
    EmptyTextBuffer();
    if ( GTK_WIDGET_VISIBLE (OutputWindow )) gtk_widget_hide (OutputWindow);
    PasswordProtectedArchive = FALSE;

    OffDeleteandViewButtons();
    CurrentArchiveType = DetectArchiveType ( path );
    if ( CurrentArchiveType == -1 ) return;
    path = EscapeBadChars ( path );
	bz_gz = FALSE;
    gtk_widget_show ( viewport2 );
    Update_StatusBar ( _("Please wait while the content of the archive is being read..."));
	switch ( CurrentArchiveType )
	{
		case 0:
		OpenBzip2 ( TRUE , path );
		break;

		case 1:
		OpenGzip ( TRUE , path );
		break;

		case 2:
		OpenRar ( TRUE , path );
		break;

		case 3:
		OpenTar ( TRUE , path );
		break;

		case 7:
		OpenZip ( TRUE , path );
		break;

        case 8:
        OpenRPM ( TRUE , path );
        break;

        case 9:
        Open7Zip ( TRUE , path );
        break;

		case 10:
		OpenArj ( TRUE , path );
		break;
	}
    if (password != NULL) g_free (password);
    password = NULL;
}

void on_quit1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    if ( GTK_WIDGET_VISIBLE (viewport2) ) return;
    g_list_free ( Suffix );
	g_list_free ( Name );
	if ( path != NULL ) g_free (path);
    if ( extract_path != NULL )
        if ( strcmp (extract_path,"/tmp/") != 0) g_free (extract_path);
    //This to delete a CPIO temp file in /tmp if existing
    if ( tmp != NULL )
    {
        unlink (tmp);
        tmp = NULL;
    }
    gtk_main_quit();
}

void on_delete1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	gchar *command = NULL;
	GtkTreeSelection *selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview1) );
	names = g_string_new ( " " );
	gtk_tree_selection_selected_foreach (selection, (GtkTreeSelectionForeachFunc) ConcatenateFileNames, names );
	action = delete;
	Update_StatusBar ( _("Deleting files from the archive, please wait..."));
	switch (CurrentArchiveType)
	{
		case 2:
		command = g_strconcat ( "rar d " , path , names->str , NULL );
		break;
		
        case 3:
		command = g_strconcat ( "tar --delete -vf " , path , names->str , NULL );
		break;
		
        case 4:
        DecompressBzipGzip ( names , path , 0  , 0 );
        break;
		
        case 5:
        DecompressBzipGzip ( names , path , 1 , 0 );
		break;	

        case 6:
        case 7:
		command = g_strconcat ( "zip -d " , path , names->str , NULL );
		break;

        case 9:
        command = g_strconcat ( "7za d " , path , names->str , NULL );
        break;

        case 10:
        command = g_strconcat ( "arj d " , path , names->str, NULL);
        break;
	}
	if (command != NULL)
    {
        ExtractAddDelete ( command );
        g_free (command);
    }
	g_string_free (names , FALSE );
}

void on_add_files_activate ( GtkMenuItem *menuitem, gpointer user_data )
{
    gchar *name = NULL;
	gchar *command = NULL;
	GSList *Files_to_Add  = Add_File_Dialog ( user_data );
	if ( Files_to_Add == NULL) return;
	names = g_string_new ( " " );
    Files_to_Add = g_slist_reverse (Files_to_Add);
    //Set the current dir so to avoid archiving the leading directory inside the archive
    name = g_path_get_dirname ( Files_to_Add->data );
	chdir ( name );
    g_free ( name );
    while ( Files_to_Add != NULL )
	{
        //Strip the path from the filename
        name = g_path_get_basename ( Files_to_Add->data );
        ConcatenateFileNames2 ( name , names );
        Files_to_Add = g_slist_next ( Files_to_Add );
	}	
		action = add;	
		switch (CurrentArchiveType)
		{
			case 0:
			Bzip2Add ( names->str , 0 );
			break;
            
			case 1:
			Bzip2Add ( names->str , 1 );
			break;
			
			case 2:
            if (password != NULL) command = g_strconcat ( "rar a -p" , password, " -o+ -ep1 -idp " , path , names->str , NULL );
                else command = g_strconcat ( "rar a -o+ -ep1 -idp " , path , names->str , NULL );
			break;
			
			case 3:
			if ( g_file_test ( path , G_FILE_TEST_EXISTS ) ) command = g_strconcat ( "tar rvvf " , path , names->str , NULL );
                else command = g_strconcat ( "tar cvvf " , path , names->str , NULL );
    		break;
			
            case 4:
            if ( g_file_test ( path , G_FILE_TEST_EXISTS ) ) DecompressBzipGzip ( names , path , 0 , 1 );
                else command = g_strconcat ("tar cvvfj " , path , names->str , NULL );
            break;

            case 5:
            if ( g_file_test ( path , G_FILE_TEST_EXISTS ) ) DecompressBzipGzip ( names , path , 1 , 1 );
                else command = g_strconcat ("tar cvvfz " , path , names->str , NULL );
            break;

			case 6:
			case 7:
            if (password != NULL) command = g_strconcat ( "zip -P " , password , " -r " , path , names->str , NULL );
                else command = g_strconcat ( "zip -r " , path , names->str , NULL );
			break;

            case 9:
            if (password != NULL) command = g_strconcat ( "7za a -ms=off -p" , password , " " , path , names->str , NULL );
                else command = g_strconcat ( "7za a -ms=off " , path , names->str , NULL );
            break;

            case 10:
            if (password != NULL) command = g_strconcat ( "arj a -i -r -g" , password , " " , path , names->str , NULL );
                else command = g_strconcat ( "arj a -i -r " , path , names->str , NULL );
            break;
		}
    Update_StatusBar ( _("Adding files to the archive, please wait..."));
    if (command != NULL)
    {
        ExtractAddDelete ( command );
        g_free (command);
    }
	g_string_free (names , FALSE );
	g_slist_free ( Files_to_Add );
}

void on_extract1_activate ( GtkMenuItem *menuitem , gpointer user_data )
{
	gchar *command = NULL;
	GtkTreeSelection *selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview1) );
	gint selected = gtk_tree_selection_count_selected_rows ( selection );
	extract_window = prefs (selected);
	gtk_dialog_set_default_response (GTK_DIALOG (extract_window), GTK_RESPONSE_OK);
    if ( PasswordProtectedArchive )
    {
            if (password == NULL) Show_pwd_Window ( NULL , NULL );
            if ( password == NULL) return;
    }
    done = FALSE;
    while ( ! done )
	{
		switch (gtk_dialog_run ( GTK_DIALOG (extract_window ) ) )
		{
			case GTK_RESPONSE_CANCEL:
			case GTK_RESPONSE_DELETE_EVENT:
			done = TRUE;
            gtk_widget_destroy ( extract_window );
			break;
			
			case GTK_RESPONSE_OK:
            action = extract;
			extract_path = g_strdup (gtk_entry_get_text ( GTK_ENTRY (entry1) ));
			if ( strlen ( extract_path ) > 0 )
			{
				done = TRUE;
                gchar *text = g_strconcat (_("Extracting files to "),extract_path,NULL );
                Update_StatusBar ( text );
                g_free (text);
                gtk_widget_destroy ( extract_window );
				if ( selected < 1 )
				{
					switch ( CurrentArchiveType )
					{
						case 0:
						//Bzip2 extraction is handled when the the file is opened
						//code execution never reaches here
						break;
						
						case 1:
						//Gzip extraction is handled when the the file is opened
						//code execution never reaches here
						break;
						
						case 2:
                        if (password !=NULL) command = g_strconcat ( "rar x -p",password," -o+ -idp " , path , " " , extract_path , NULL );
                        else command = g_strconcat ( "rar x -o+ -idp " , path , " " , extract_path , NULL );
						break;
						
						case 3:
						command = g_strconcat ( "tar xfv " , path , " -C " , extract_path , NULL );
						break;

						case 4:
						command = g_strconcat ( "tar xfjv " , path , " -C " , extract_path , NULL );
						break;
						
						case 5:
						command = g_strconcat ( "tar xfzv " , path , " -C " , extract_path , NULL );
						break;

                        case 6:
						case 7:
                        if ( password != NULL ) command = g_strconcat ( "unzip -o -P " , password , " " , path , " -d " , extract_path , NULL );
                            else command = g_strconcat ( "unzip -o " , path , " -d " , extract_path , NULL );
						break;

                        case 8:
                        chdir ( extract_path );
                        SpawnCPIO ( "cpio -ivd" , tmp , 0 , 1 );
                        break;

                        case 9:
                        if (password != NULL) command = g_strconcat ( "7za x -aoa -bd -p",password," ", path , " -o" , extract_path , NULL );
                        else command = g_strconcat ( "7za x -aoa -bd " , path , " -o" , extract_path , NULL );
                        break;

						case 10:
						if (password !=NULL) command = g_strconcat ( "arj x -g",password," -i -y " , path , " " , extract_path , NULL );
                        else command = g_strconcat ( "arj x -i -y " , path , " " , extract_path , NULL );
						break;
					}
                    if ( command != NULL )
                    {
                        ExtractAddDelete ( command );
                        g_free (command); 
                    }
				}
				else
				{
					names = g_string_new ( " " );
					gtk_tree_selection_selected_foreach (selection, (GtkTreeSelectionForeachFunc) ConcatenateFileNames, names );
					command = ChooseCommandtoExecute ( 1 );
                    if ( command != NULL )
                    {
                        ExtractAddDelete ( command );
                        g_free (command); 
                    }
                    g_string_free (names , FALSE );
				}
			}
			else response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK, "Please select where to extract files !" );
			break;
		}
	}
}

gchar *ChooseCommandtoExecute ( gboolean full_path )
{
    gchar *command;
    int levels;
    char digit[2];
    gchar *strip = NULL;
    if ( full_path == 0 )
    {
        levels = CountCharacter ( names->str , '/');
        sprintf ( digit , "%d" , levels );
        strip = g_strconcat ( "--strip-components=" , digit , " " , NULL );
    }
    switch (CurrentArchiveType)
	{
	    case 0:
	    //Bzip2 extraction is handled when the the file is opened
		//code execution never reaches here
		break;
						
		case 1:
        //idem
		break;
						
		case 2:
		if (password != NULL) command = g_strconcat ( "rar " , full_path ? "x" : "e" , " -p",password, " -o+ -idp " , path , " " , names->str , " " , extract_path , NULL );
        else command = g_strconcat ( "rar ", full_path ? "x" : "e" , " -o+ -idp " , path , " " , names->str , " " ,extract_path ,NULL);
		break;
						
		case 3:
	    command = g_strconcat ( "tar " , full_path ? "" : strip , "-xvf " , path , " -C " , extract_path , names->str , NULL );
		break;
						
		case 4:
		command = g_strconcat ( "tar " , full_path ? "" : strip , "-xjvf " , path , " -C " , extract_path ,  names->str , NULL );
		break;

		case 5:
        command = g_strconcat ( "tar " , full_path ? "" : strip , "-xvzf " , path , " -C " , extract_path , names->str , NULL );
		break;

        case 6:
		case 7:
        if ( password != NULL ) command = g_strconcat ( "unzip -o -P " , password , full_path ? " " : " -j " , path , names->str , " -d " , extract_path , NULL );
        else command = g_strconcat ( "unzip -o " , full_path ? "" : "-j " , path , names->str , " -d " , extract_path , NULL );
		break;
                        
        case 8:
        chdir ( extract_path );
        command = g_strconcat ( "cpio -ivd" , names->str , NULL );
        SpawnCPIO ( command , tmp , 0 , 1 );
        g_free (command);
        command = NULL;
        break;

        case 9:
        if ( password != NULL) command = g_strconcat ("7za " , full_path ? "x" : "e" , " -p",password," -aoa -bd " , path , names->str , " -o" , extract_path , NULL );
        else command = g_strconcat ( "7za " , full_path ? "x" : "e" ," -aoa -bd " , path , names->str , " -o" , extract_path , NULL );
        break;

		case 10:
		if (password !=NULL) command = g_strconcat ( "arj x -g",password," -i -y " , path , " " , extract_path , names->str , NULL );
        else command = g_strconcat ( "arj ",full_path ? "x" : "e"," -i -y " , path , " " , extract_path , names->str, NULL );
		break;
    }
    if ( strip != NULL) g_free ( strip );
    return command;
}

void on_about1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	static GtkWidget *about = NULL;
	const char *authors[] = {"Giuseppe Torelli - Colossus\n<gt67@users.sourceforge.net>\n\nThanks to Uracile for the stunning logo.\n\nThanks to the following translators:\nEugene Ostapets: russian\nEnrico Troeger: german\nSZERVAC Attila: hungarian\n\nThanks to the people of gtk-app-devel-list\nwho kindly answered my questions.", NULL};
	if (about != NULL)
	{
		gtk_window_present (GTK_WINDOW (about));
		return;
	}
	about = gtk_about_dialog_new ();
	g_object_set (about,
		      "name",  "Xarchiver",
		      "version", VERSION,
		      "copyright", "Copyright @2005 Giuseppe Torelli",
		      "comments", "A lightweight GTK2 archive manager",
		      "authors", authors,
		      "documenters", NULL,
		      "translator_credits", NULL,
		      "logo_icon_name", "xarchiver",
		      "website", "http://xarchiver.sourceforge.net",
		      "website_label", "Home Page",
		      "license",    "Copyright @2005 Giuseppe Torelli - Colossus <gt67@users.sourceforge.net>\n\n"
		      			"This library is free software; you can redistribute it and/or\n"
    					"modify it under the terms of the GNU Library General Public License as\n"
    					"published by the Free Software Foundation; either version 2 of the\n"
    					"License, or (at your option) any later version.\n"
    					"\n"
    					"This library is distributed in the hope that it will be useful,\n"
    					"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    					"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n"
    					"Library General Public License for more details.\n"
    					"\n"
    					"You should have received a copy of the GNU Library General Public\n"
    					"License along with the Gnome Library; see the file COPYING.LIB.  If not,\n"
    					"write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,\n"
    					"Boston, MA 02111-1307, USA.\n",
		      NULL);
	gtk_window_set_destroy_with_parent (GTK_WINDOW (about), TRUE);
	g_signal_connect (G_OBJECT (about), "destroy",  G_CALLBACK (gtk_widget_destroyed), &about);
	gtk_window_set_position (GTK_WINDOW (about), GTK_WIN_POS_CENTER);
	gtk_widget_show (about);
}

GSList *Add_File_Dialog ( gchar *mode )
{
	GSList *list = NULL;
	int flag;
	if ( mode == "file" )
	{
		title = _("Select the files to be added to the current archive; use SHIFT to multiple select");
		flag = GTK_FILE_CHOOSER_ACTION_OPEN;
		
	}
	else
	{
		if (CurrentArchiveType == 10) title = _("Select the folder to be added to the current archive");
            else title = _("Select the folders to be added to the current archive; use SHIFT to multiple select");
		flag = GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER;
	}
	File_Selector = gtk_file_chooser_dialog_new ( title,
							GTK_WINDOW (MainWindow),
							flag,
							GTK_STOCK_CANCEL,
							GTK_RESPONSE_CANCEL,
							GTK_STOCK_ADD,
							GTK_RESPONSE_ACCEPT,
							NULL);
    //We set gtk_file_chooser_set_select_multiple to FALSE because of a bug in ARJ prevents adding more of two directories
    if (CurrentArchiveType == 0 || CurrentArchiveType == 1 || CurrentArchiveType == 10) gtk_file_chooser_set_select_multiple ( GTK_FILE_CHOOSER (File_Selector) , FALSE );
        else gtk_file_chooser_set_select_multiple ( GTK_FILE_CHOOSER (File_Selector) , TRUE );
	if (CurrentFolder != NULL) gtk_file_chooser_set_current_folder ( GTK_FILE_CHOOSER (File_Selector) , CurrentFolder );
	response = gtk_dialog_run (GTK_DIALOG (File_Selector) );
	if (response == GTK_RESPONSE_ACCEPT) list = gtk_file_chooser_get_filenames ( GTK_FILE_CHOOSER (File_Selector) );
	gtk_widget_destroy (File_Selector);
	return list;
}

gchar *Show_File_Dialog ( int dummy , gpointer mode )
{
	GtkWidget *hbox;
	GtkWidget *combo_box;
	GtkWidget *archive_types;
	GtkWidget *check_button;
	GtkFileFilter *filter;
	GtkTooltips *FilterToolTip ;
	gchar *path;
	const gchar *flag2;
	int i , flag;
	
	if ( mode == "new" )
	{
		title = _("Create a new archive");
		flag = GTK_FILE_CHOOSER_ACTION_SAVE;
		flag2 = "gtk-new";
	}
	else if ( mode == "open" )	
	{
		title = _("Open an archive");
		flag = GTK_FILE_CHOOSER_ACTION_OPEN;
		flag2 = "gtk-open";
	}
	
	else if (mode == "extract" )		
	{
		title = _("Choose the destination folder where to extract the current archive");
		File_Selector = gtk_file_chooser_dialog_new ( title, 
		GTK_WINDOW (MainWindow),
		GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
		GTK_STOCK_CANCEL,
		GTK_RESPONSE_CANCEL,
		GTK_STOCK_OPEN,
		GTK_RESPONSE_ACCEPT,
		NULL );
		response = gtk_dialog_run (GTK_DIALOG (File_Selector));
		if (response == GTK_RESPONSE_ACCEPT) gtk_entry_set_text (GTK_ENTRY(entry1),gtk_file_chooser_get_filename ( GTK_FILE_CHOOSER (File_Selector) ) );
		gtk_widget_destroy (File_Selector);
		return NULL;						
	}

	File_Selector = gtk_file_chooser_dialog_new ( title, 
							GTK_WINDOW (MainWindow),
							flag,
							GTK_STOCK_CANCEL,
							GTK_RESPONSE_CANCEL,
							flag2,
							GTK_RESPONSE_ACCEPT,
							NULL);
	if (CurrentFolder != NULL) gtk_file_chooser_set_current_folder ( GTK_FILE_CHOOSER (File_Selector) , CurrentFolder );
	gtk_dialog_set_default_response (GTK_DIALOG (File_Selector), GTK_RESPONSE_ACCEPT);
	
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name ( filter , _("All files") );
	gtk_file_filter_add_pattern ( filter, "*" );
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (File_Selector), filter);
	
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name ( filter , _("Only archives") );
	
	Suffix = g_list_first ( ArchiveSuffix );
	
	while ( Suffix != NULL )
	{
		gtk_file_filter_add_pattern (filter, Suffix->data);
		Suffix = g_list_next ( Suffix );
	}
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (File_Selector), filter);	
	
	Suffix = g_list_first ( ArchiveSuffix );
		
	while ( Suffix != NULL )
	{
		if ( Suffix->data != "" )	//To avoid double filtering when opening the archive
		{
			filter = gtk_file_filter_new ();
			gtk_file_filter_set_name (filter, Suffix->data );
			gtk_file_filter_add_pattern (filter, Suffix->data );
			gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (File_Selector), filter);	
		}
		
		Suffix = g_list_next ( Suffix );
	}
	
	if ( mode == "new" )
	{
		hbox = gtk_hbox_new (FALSE, 12);
		gtk_box_pack_start (GTK_BOX (hbox),gtk_label_new (_("Archive type:")),FALSE, FALSE, 0);
		combo_box = gtk_combo_box_new_text ();
		FilterToolTip = gtk_tooltips_new();
		gtk_tooltips_set_tip (FilterToolTip,combo_box, _("Choose the archive type to create") , NULL);
		Name = g_list_first ( ArchiveType );
		while ( Name != NULL )
		{
			if (Name->data != ".tgz" && Name->data != ".rpm" ) gtk_combo_box_append_text (GTK_COMBO_BOX (combo_box), Name->data );
			Name = g_list_next ( Name );
		}
		gtk_combo_box_set_active (GTK_COMBO_BOX (combo_box), 0);
		gtk_box_pack_start (GTK_BOX (hbox), combo_box, TRUE, TRUE, 0);
		check_button = gtk_check_button_new_with_label (_("Add the archive extension to the filename"));
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(check_button),TRUE);
		gtk_box_pack_start (GTK_BOX (hbox), check_button, TRUE, TRUE, 0);
		gtk_widget_show_all (hbox);
		gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (File_Selector), hbox);
	}
	
	gtk_window_set_modal (GTK_WINDOW (File_Selector),TRUE);
	response = gtk_dialog_run (GTK_DIALOG (File_Selector));
	CurrentFolder = gtk_file_chooser_get_current_folder ( GTK_FILE_CHOOSER (File_Selector) ); 
	if (response == GTK_RESPONSE_ACCEPT)
	{
		path = gtk_file_chooser_get_filename ( GTK_FILE_CHOOSER (File_Selector) );
		if ( mode == "new")
		{
			ComboArchiveType = gtk_combo_box_get_active_text (GTK_COMBO_BOX (combo_box));
            
            if (strcmp ( ComboArchiveType,".arj") == 0) CurrentArchiveType = 10;
                else if (strcmp ( ComboArchiveType,".bz2") == 0) CurrentArchiveType = 0;
                else if (strcmp ( ComboArchiveType,".gz") == 0) CurrentArchiveType = 1;
                else if (strcmp ( ComboArchiveType,".rar") == 0) CurrentArchiveType = 2;
                else if (strcmp ( ComboArchiveType,".tar") == 0) CurrentArchiveType = 3;
                else if (strcmp ( ComboArchiveType,".tar.bz2") == 0) CurrentArchiveType = 4;
                else if (strcmp ( ComboArchiveType,".tar.gz") == 0) CurrentArchiveType = 5;
                else if (strcmp ( ComboArchiveType,".jar") == 0) CurrentArchiveType = 6;
                else if (strcmp ( ComboArchiveType,".zip") == 0) CurrentArchiveType = 7;
                else if (strcmp ( ComboArchiveType,".rpm") == 0) CurrentArchiveType = 8;
                else if (strcmp ( ComboArchiveType,".7z") == 0) CurrentArchiveType = 9;
			if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check_button) ) )
			{
				if ( ! g_str_has_suffix ( path , ComboArchiveType ) )
				{
					gtk_widget_destroy (File_Selector);
					return g_strconcat ( path, ComboArchiveType , NULL);
				}
				
			}
		}
		gtk_widget_destroy (File_Selector);
		return ( path );
	}
	else if ( (response == GTK_RESPONSE_CANCEL) || (response == GTK_RESPONSE_DELETE_EVENT) )
	{
		gtk_widget_destroy (File_Selector);
		return NULL;
	}
}

gboolean isTar ( FILE *ptr )
{
	unsigned char magic[7];
	fseek ( ptr, 0 , SEEK_SET );
    if ( fseek ( ptr , 257 , SEEK_CUR) < 0 ) return FALSE;
	if ( fread ( magic, 1, 7, ptr ) == 0 ) return FALSE;
	if ( memcmp ( magic,"\x75\x73\x74\x61\x72\x00\x30",7 ) == 0 || memcmp (magic,"\x75\x73\x74\x61\x72\x20\x20",7 ) == 0) return TRUE;
		else return FALSE;
}

int DetectArchiveType ( gchar *path )
{
	FILE *dummy_ptr = NULL;
    int xx = -1;
	unsigned char magic[6];
	dummy_ptr = fopen ( path , "r" );
	if (dummy_ptr == NULL)
	{
		gchar *msg = g_strdup_printf (_("Can't open archive %s:\n%s") , path , strerror (errno) ); 
		response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow) , GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK,
		msg);
		g_free (msg);
		return xx;
	 }
	if ( fread ( magic, 1, 6, dummy_ptr ) == 0 )
	{
		fclose ( dummy_ptr);
		response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK,strerror(errno));
		return xx;
	}
    if ( memcmp ( magic,"\x50\x4b\x03\x04",4 ) == 0 || memcmp ( magic,"\x50\x4b\x05\x06",4 ) == 0 )
    {
        PasswordProtectedArchive = DetectPasswordProtectedArchive ( 7 , dummy_ptr , magic );
        xx = 7; // zip
    }
	else if ( memcmp ( magic,"\x60\xea",2 ) == 0 )
    {
        PasswordProtectedArchive = DetectPasswordProtectedArchive ( 10 , dummy_ptr , magic );
        xx = 10;    // arj
    }
	else if ( memcmp ( magic,"\x52\x61\x72\x21",4 ) == 0 ) xx = 2;	// rar
    else if ( memcmp ( magic,"\x42\x5a\x68",3 ) == 0 ) xx = 0;	// bzip2
	else if ( memcmp ( magic,"\x1f\x8b\x08",3 ) == 0) xx = 1;		// gz
    else if ( memcmp ( magic,"\xed\xab\xee\xdb",4 ) == 0) xx = 8;	// rpm
    else if ( memcmp ( magic,"\x37\x7a\xbc\xaf\x27\x1c",6 ) == 0 ) xx = 9;	// 7zip
    else if ( isTar ( dummy_ptr ) ) xx = 3;
	else
	{
		gtk_window_set_title ( GTK_WINDOW (MainWindow) , "Xarchiver " VERSION );
		response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK,
		_("The format of this archive is not recognized !") );
	}
	fclose ( dummy_ptr );
	return xx;
}

gboolean DetectPasswordProtectedArchive ( int type , FILE *stream , unsigned char magic[6] )
{
    //ZIP
    unsigned int fseek_offset;
    unsigned short int password_flag;
    unsigned int compressed_size;
    unsigned int uncompressed_size;
    unsigned short int file_length;
    unsigned short int extra_length;
    
    //ARJ
    unsigned char sig[2];
    unsigned short int basic_header_flag;
    unsigned short int basic_header_size;
    unsigned short int extended_header_size;
    unsigned int basic_header_CRC;
    unsigned int extended_header_CRC;
    unsigned char arj_flag;

    if ( type == 7 )
    {
        while ( memcmp ( magic,"\x50\x4b\x03\x04",4 ) == 0  || memcmp ( magic,"\x50\x4b\x05\x06",4 ) == 0 )
        {
            fread ( &password_flag, 1, 2, stream );
            if (( password_flag & ( 1<<0) ) > 0) return TRUE;
            fseek (stream,10,SEEK_CUR);
            fread (&compressed_size,1,4,stream);
            fread (&uncompressed_size,1,4,stream);
            fread (&file_length,1,2,stream);
            fread (&extra_length,1,2,stream);
            fseek_offset = compressed_size + file_length + extra_length;
            fseek (stream , fseek_offset , SEEK_CUR);
            fread (magic , 1 , 4 , stream);
            fseek ( stream , 2 , SEEK_CUR);
        }
    }
    else if ( type == 10)
    {   
        fseek (stream , magic[2]+magic[3] , SEEK_CUR);
        fseek (stream , 2 , SEEK_CUR);
        fread (&extended_header_size,1,2,stream);
        if (extended_header_size != 0) fread (&extended_header_CRC,1,4,stream);
        fread (&sig,1,2,stream);
        while ( memcmp (sig,"\x60\xea",2) == 0)
        {
            fread ( &basic_header_size , 1 , 2 , stream );
            if ( basic_header_size == 0 ) break;
            fseek ( stream , 4 , SEEK_CUR);
            fread (&arj_flag,1,1,stream);
            if ((arj_flag & ( 1<<0) ) > 0) return TRUE;
            fseek ( stream , 7 , SEEK_CUR);
            fread (&compressed_size,1,4,stream);
            fseek ( stream , basic_header_size - 16 , SEEK_CUR);
            fread (&basic_header_CRC,1,4,stream);
            fread (&extended_header_size,1,2,stream);
            if (extended_header_size != 0) fread (&extended_header_CRC,1,4,stream);
            fseek ( stream , compressed_size , SEEK_CUR);
            fread (&sig,1,2,stream);
        }
    }
    return FALSE;
}

gulong SpawnAsyncProcess (const gchar *command , gboolean ExitStatusFlag , gboolean input)
{
	GError *error = NULL;
	gchar **argv;
	int argcp;
	g_shell_parse_argv ( command , &argcp , &argv , NULL);
	if ( ! g_spawn_async_with_pipes (
		NULL,
		argv,
		NULL,
		G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD,
		NULL,
		NULL,
		&child_pid,
		input ? &input_fd : NULL,
		&output_fd,
		&error_fd,
		&error) )
	{
		response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK, error->message);
		g_error_free (error);
		g_strfreev ( argv );
        return FALSE;
	}
	g_strfreev ( argv );
    return (gulong) child_pid;
}

GIOChannel *SetIOChannel (gint fd, GIOCondition cond, GIOFunc func, gpointer data)
{
	GIOChannel *ioc;
	ioc = g_io_channel_unix_new ( fd );
	g_io_add_watch (ioc, cond, func, data);
	g_io_channel_set_encoding (ioc, "ISO8859-1" , NULL);
	g_io_channel_set_flags ( ioc , G_IO_FLAG_NONBLOCK , NULL );
	return ioc;
}

GIOChannel *SetIOChannelEncondingNULL (gint fd, GIOCondition cond, GIOFunc func, gpointer data)
{
	GIOChannel *ioc;
	ioc = g_io_channel_unix_new ( fd );
	g_io_add_watch (ioc, cond, func, data);
	g_io_channel_set_encoding (ioc, NULL , NULL);
	return ioc;
}

/* These three functions are from File-Roller code */
char *eat_spaces (char *line)
{
	if (line == NULL)
		return NULL;
	while ((*line == ' ') && (*line != 0))
		line++;
	return line;
}

char *get_last_field (char *line,int last_field)
{
	char *field;
	int i;

	if (line == NULL)
		return NULL;

	last_field--;
	field = eat_spaces (line);
	for (i = 0; i < last_field; i++) {
		if (field == NULL)
			return NULL;
		field = strchr (field, ' ');
		field = eat_spaces (field);
	}
	//The following line is mine, I replace the \n with the null terminated
    if (field != NULL) field [ strlen(field) -1 ] = '\000';
	return field;
}

char **split_line (char *line,int n_fields)
{
	char **fields;
	char *scan, *field_end;
	int i;

	fields = g_new0 (char *, n_fields + 1);
	fields[n_fields] = NULL;

	scan = eat_spaces (line);
	for (i = 0; i < n_fields; i++)
	{
		if (scan == NULL)
		{
			fields[i] = NULL;
			continue;
		}
		field_end = strchr (scan, ' ');
		//The following line is mine, I added the case when the last field ends with a newline
		if (field_end == NULL) field_end = strchr (scan, '\n');
		if (field_end != NULL)
		{
			fields[i] = g_strndup (scan, field_end - scan);
			scan = eat_spaces (field_end);
		}
	}
	return fields;
}
/* End code from File-Roller */

void RemoveColumnsListStore()
{
    SetButtonState (1,1,0,0,0);
	gtk_window_set_title ( GTK_WINDOW (MainWindow) , "Xarchiver " VERSION );
	GList *columns = gtk_tree_view_get_columns ( GTK_TREE_VIEW (treeview1) );
	while (columns != NULL)
	{
		gtk_tree_view_remove_column (GTK_TREE_VIEW (treeview1) , columns->data);
		columns = columns->next;
	}
	g_list_free (columns);
}

void EmptyTextBuffer ()
{
	if (textbuf != NULL)
	{
		gtk_text_buffer_get_start_iter (textbuf,&start);
		gtk_text_buffer_get_end_iter (textbuf,&end);
		gtk_text_buffer_delete (textbuf,&start,&end);
		gtk_text_buffer_get_start_iter(textbuf, &enditer);
	}
}

void CreateListStore ( int nc, gchar *columns_names[] , GType columns_types[])
{
	liststore = gtk_list_store_newv ( nc , (GType *)columns_types);
	gtk_tree_view_set_model ( GTK_TREE_VIEW (treeview1), GTK_TREE_MODEL (liststore) );
	gtk_tree_view_set_rules_hint ( GTK_TREE_VIEW (treeview1) , TRUE );
	GtkTreeSelection *sel = gtk_tree_view_get_selection( GTK_TREE_VIEW (treeview1) );
	gtk_tree_selection_set_mode(sel, GTK_SELECTION_MULTIPLE);
	g_signal_connect ((gpointer) sel, "changed", G_CALLBACK (Activate_buttons), NULL);
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview1));
    g_object_ref(model);
    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview1), NULL);
	for (x = 0; x <= nc-1; x++)
	{
		renderer = gtk_cell_renderer_text_new ();
		column = gtk_tree_view_column_new_with_attributes ( columns_names[x],renderer,"text",x,NULL);
		gtk_tree_view_column_set_resizable (column, TRUE);
		gtk_tree_view_column_set_sort_column_id (column, x);
		gtk_tree_view_append_column (GTK_TREE_VIEW (treeview1), column);
	}
}

void ShowShellOutput ()
{
    if (OutputWindow != NULL)
    {
        gtk_window_present ( GTK_WINDOW (OutputWindow) ); 
        return;
    }
	OutputWindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_name (OutputWindow, "OutputWindow");
    gtk_window_set_title (GTK_WINDOW (OutputWindow), _("Shell output") );
	gtk_window_set_position (GTK_WINDOW (OutputWindow), GTK_WIN_POS_CENTER);
	gtk_window_set_default_size(GTK_WINDOW(OutputWindow), 450, 300);
	gtk_window_set_destroy_with_parent (GTK_WINDOW (OutputWindow), TRUE);
	g_signal_connect (G_OBJECT (OutputWindow), "delete-event",  G_CALLBACK (gtk_widget_hide), &OutputWindow);

	vbox = gtk_vbox_new ( FALSE, 2 );
	scrollwin = gtk_scrolled_window_new ( NULL,NULL );
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW( scrollwin ), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	textview = gtk_text_view_new();
	gtk_text_view_set_editable (GTK_TEXT_VIEW(textview), FALSE);
	gtk_container_add (GTK_CONTAINER(scrollwin), textview);
	gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER(OutputWindow), vbox);
	textbuf = gtk_text_view_get_buffer ( GTK_TEXT_VIEW(textview) );
	gtk_text_buffer_get_start_iter (textbuf, &enditer);
	gtk_text_buffer_create_tag (textbuf, "red_foreground","foreground", "red", NULL);
	
	gtk_widget_show (vbox);
	gtk_widget_show (scrollwin);
	gtk_widget_show (textview);
}

void Cancel_Operation ( GtkMenuItem *menuitem , gpointer user_data )
{
    if ( kill ( child_pid , SIGABRT ) < 0 )
    {
        response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK, strerror(errno));
	    return;
    }
    //This in case the user cancel the opening a password protected archive
    if (PasswordProtectedArchive) PasswordProtectedArchive = FALSE;
}

void View_File_Window ( GtkMenuItem *menuitem , gpointer user_data )
{
    gchar *command = NULL;

    if ( PasswordProtectedArchive )
    {
            Show_pwd_Window ( NULL , NULL );
            if ( password == NULL) return;
    }
    extract_path = "/tmp/";
    names = g_string_new ( "" );
    GtkTreeSelection *selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview1) );
    gtk_tree_selection_selected_foreach (selection, (GtkTreeSelectionForeachFunc) ConcatenateFileNames, names );
    command = ChooseCommandtoExecute ( 0 );
   	compressor_pid = SpawnAsyncProcess ( command , 0 , 0);
    g_free ( command );
	if ( compressor_pid == 0 )
    {
        g_string_free ( names , FALSE );
        return;
    }
    SetIOChannel (error_fd, G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL,GenError, NULL );
    g_child_watch_add ( compressor_pid , (GChildWatchFunc) ViewFileFromArchive , names );
}

GChildWatchFunc *ViewFileFromArchive (GPid pid , gint status , GString *data)
{
    GIOChannel *ioc_view = NULL;
    gchar *line = NULL;
    gchar *filename = NULL;
    GError *error = NULL;
    gchar *string = NULL;
    gboolean tofree = FALSE;

    if ( WIFEXITED( status ) )
    {   
	    if ( WEXITSTATUS ( status ) )
    	{
	    	archive_error = TRUE;
    		SetButtonState (1,1,0,0,0);
	    	gtk_window_set_title ( GTK_WINDOW (MainWindow) , "Xarchiver " VERSION );
		    response = ShowGtkMessageDialog (GTK_WINDOW 		(MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_QUESTION,GTK_BUTTONS_YES_NO,_("An error occurred while extracting the file to be viewed.\nDo you want to view the shell output ?") );
            if (response == GTK_RESPONSE_YES) ShowShellOutput (NULL , FALSE);
            unlink ( (char*)data );
			return;
        }
    }
    string = StripPathFromFilename ( data->str );
    if (  string == NULL )
    {
        data->str++;
        filename = g_strconcat ( "/tmp/" , data->str , NULL );
    }
    else
    {
        if ( strchr ( string , ' ' ) )
        {
            string = RemoveBackSlashes ( string );
            tofree = TRUE;
        }
        filename = g_strconcat ( "/tmp" , string , NULL );
        if ( tofree ) g_free ( string );
    }
    if ( g_file_test ( filename , G_FILE_TEST_IS_DIR ) )
    {
       response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,	GTK_BUTTONS_OK,_("Please select a file, not a directory !") );
        rmdir ( filename );
        g_string_free ( data , FALSE );
        g_free ( filename );
        return;
    }
    view_window = view_win();
	ioc_view = g_io_channel_new_file ( filename , "r" , &error );
    if (error == NULL) 
    {
        g_io_channel_set_encoding (ioc_view, "ISO8859-1" , NULL);
        g_io_channel_set_flags ( ioc_view , G_IO_FLAG_NONBLOCK , NULL );
        g_io_channel_read_to_end ( ioc_view , &line , NULL, NULL );
        gtk_text_buffer_get_end_iter ( viewtextbuf, &viewenditer );
        gtk_text_buffer_insert (viewtextbuf, &viewenditer, line, strlen ( line ) );
        g_free ( line );
        g_io_channel_shutdown ( ioc_view , TRUE , NULL );
        g_io_channel_unref (ioc_view);
    }
    unlink ( filename );
    gtk_widget_show (view_window);
    g_free (filename);
    g_string_free ( data , FALSE );
}

void Show_pwd_Window ( GtkMenuItem *menuitem , gpointer user_data )
{
    pwd_window = passwd_win();
    password_entry = lookup_widget ( pwd_window , "pwd_entry" );
    repeat_password = lookup_widget ( pwd_window , "entry2");
        
    gtk_dialog_set_default_response (GTK_DIALOG (pwd_window), GTK_RESPONSE_OK);
    done = FALSE;
	while ( ! done )
	{
		switch ( gtk_dialog_run ( GTK_DIALOG ( pwd_window ) ) )
		{
			case GTK_RESPONSE_CANCEL:
			case GTK_RESPONSE_DELETE_EVENT:
			done = TRUE;
            password = NULL;
            break;
			
			case GTK_RESPONSE_OK:
			password  = g_strdup ( gtk_entry_get_text( GTK_ENTRY ( password_entry ) ) );
             if ( strlen ( password ) == 0 || strlen(gtk_entry_get_text( GTK_ENTRY ( repeat_password )) ) == 0 )
            {
                response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK,_("Please type a password !") );
                break;
            }
            if ( strcmp (password , gtk_entry_get_text( GTK_ENTRY ( repeat_password ) ) ) )
            {
                response = ShowGtkMessageDialog (GTK_WINDOW (MainWindow),GTK_DIALOG_MODAL,GTK_MESSAGE_ERROR,GTK_BUTTONS_OK,_("The passwords don't match !!") );
                gtk_entry_set_text ( GTK_ENTRY ( password_entry ) , "" );
                gtk_entry_set_text ( GTK_ENTRY ( repeat_password ) , "" );
            }
            else done = TRUE;
            break;
        }
    }
    gtk_widget_destroy ( pwd_window );
}

//Taken from xarchive - http://xarchive.sourceforge.net
int is_escaped_char(char c)
{
    switch ( c )
    {
        case ' ':	
        case '\'':	
        case '"':	
        case '(':	
        case ')':	
        case '$':	
        case '\\':	
        case ';':	
        case '<':	
        case '>':	
        case '&':	
        case '#':	
        case '*':	
        case '|':	
        case '`':	
        case '!':	
            return 1;
        default:	
            return 0;
    }
}

gchar *EscapeBadChars ( gchar *string )
{
	char *q;
	char *escaped;
	int escapechars = 0;
	char *p = string;

	while (*p != '\000')
	{
        	if (is_escaped_char(*p)) escapechars++;
	        p++;
    }

	if (!escapechars) return g_strdup(string);
	escaped = (char *) malloc (strlen(string) + escapechars + 1);

	p = string;
	q = escaped;

	while (*p != '\000')
	{
        	if (is_escaped_char(*p)) *q++ = '\\';
		*q++ = *p++;
	}
	*q = '\000';
	return escaped;
}
//End code from xarchive

void Activate_buttons ()
{
	if ( archive_error ) return;
	GtkTreeSelection *selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview1) );
	gint selected = gtk_tree_selection_count_selected_rows ( selection );
	if (selected == 0 ) OffDeleteandViewButtons();
    else
	{
        if ( CurrentArchiveType != 8 ) 
        {
            gtk_widget_set_sensitive ( delete_menu , TRUE );
		    gtk_widget_set_sensitive ( Delete_button , TRUE );
        }
        if (selected > 1 || CurrentArchiveType == 8)
        {
            gtk_widget_set_sensitive ( View_button , FALSE);
		    gtk_widget_set_sensitive ( view_menu, FALSE );
        }
        else
        {
            gtk_widget_set_sensitive ( View_button , TRUE );
		    gtk_widget_set_sensitive ( view_menu, TRUE );
        }
	}
}

void ConcatenateFileNames2 (gchar *filename , GString *data)
{
	gchar *esc_filename = EscapeBadChars ( filename );
	g_string_prepend (data, esc_filename);
	g_string_prepend_c (data, ' ');
	g_free (esc_filename);
	g_free (filename);
}

void ConcatenateFileNames (GtkTreeModel *model, GtkTreePath *treepath, GtkTreeIter *iter, GString *data)
{
	gchar *filename;
	gtk_tree_model_get (model, iter, 0, &filename, -1);
	ConcatenateFileNames2 ( filename , data );
}

void ExtractAddDelete ( gchar *command )
{
	EmptyTextBuffer ();
	compressor_pid = SpawnAsyncProcess ( command , 1 , 0);
	if ( compressor_pid == 0 ) return;
    gtk_widget_show ( viewport2 );
	SetIOChannel (output_fd, G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL , GenOutput, NULL );
	SetIOChannel (error_fd, G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL , GenError, NULL );
    WaitExitStatus ( child_pid , NULL );
}

void Update_StatusBar ( gchar *msg)  
{
    gtk_label_set_text (GTK_LABEL (info_label), msg);
}

gboolean GenError (GIOChannel *ioc, GIOCondition cond, gpointer data)
{
	if (cond & (G_IO_IN | G_IO_PRI) )
	{
		gchar *line = NULL;
		g_io_channel_read_line ( ioc, &line, NULL, NULL, NULL );
        gtk_progress_bar_pulse ( GTK_PROGRESS_BAR (progressbar) );
        while (gtk_events_pending() )
			gtk_main_iteration();
		if (line != NULL && strcmp (line,"\n") )
		{
			gtk_text_buffer_insert_with_tags_by_name (textbuf, &enditer, line , -1, "red_foreground", NULL);
			g_free (line);
		}
		return TRUE;
	}
	else if (cond & (G_IO_ERR | G_IO_HUP | G_IO_NVAL) )
	{
		g_io_channel_shutdown ( ioc,TRUE,NULL );
        g_io_channel_unref (ioc);
		return FALSE;
	}
}

gboolean GenOutput (GIOChannel *ioc, GIOCondition cond, gpointer data)
{
	gchar *line = NULL;
	if (cond & (G_IO_IN | G_IO_PRI) )
	{
		g_io_channel_read_line ( ioc, &line, NULL, NULL, NULL );
        gtk_progress_bar_pulse ( GTK_PROGRESS_BAR (progressbar) );
        while (gtk_events_pending() )
			gtk_main_iteration();
		if (line != NULL )
		{
			gtk_text_buffer_insert (textbuf, &enditer, line, strlen ( line ) );
			g_free (line);
		}
	}
	else if (cond & (G_IO_ERR | G_IO_HUP | G_IO_NVAL) )
	{
	    g_io_channel_shutdown ( ioc,TRUE,NULL );
        g_io_channel_unref (ioc);
		return FALSE;
	}
	return TRUE;
}

gchar *StripPathFromFilename ( gchar *name )
{
    return g_strrstr ( name , "/" );
}

gchar *JoinPathArchiveName ( const gchar *extract_path , gchar *path )
{
	gchar *full_path = NULL;
	full_path = g_strconcat (extract_path,path,NULL);
	return full_path;
}

void OffDeleteandViewButtons()
{
    gtk_widget_set_sensitive ( Delete_button, FALSE);
    gtk_widget_set_sensitive ( delete_menu, FALSE);
    gtk_widget_set_sensitive ( View_button, FALSE);
    gtk_widget_set_sensitive ( view_menu, FALSE);
}

void OffTooltipPadlock()
{
    gtk_widget_hide (viewport2);
    if ( ! PasswordProtectedArchive )
    {
        gtk_tooltips_disable ( pad_tooltip );
        gtk_widget_hide ( pad_image );
    }
}

int CountCharacter ( gchar *string , int chr )
{
    int n = 0;
    while ( *string )
    {
        if ( *string == chr ) n++;
        string++;
    }
    return n;    
}

gchar *RemoveBackSlashes ( gchar *name)
{
    gchar *nome, *q;
    int x = CountCharacter ( name , '\\' );
    nome = (char *) malloc (strlen(name) - x + 1);
    q = nome;
    while ( *name )
    {
        if ( *name == '\\' ) name++;
        *q++ = *name++;
    }
    *q = '\000';
    return nome;
}

