/* vi:set ts=8 sts=4 sw=4 noet:
 *
 * VIM - Vi IMproved	by Bram Moolenaar
 *
 * Do ":help uganda"  in Vim to read copying and usage conditions.
 * Do ":help credits" in Vim to see a list of people who contributed.
 */

/*
 * This file defines the Ex commands.
 * When DO_DECLARE_EXCMD is defined, the table with ex command names and
 * options results.
 * When DO_DECLARE_EXCMD is NOT defined, the enum with all the Ex commands
 * results.
 * This clever trick was invented by Ron Aaron.
 */

/*
 * When adding an Ex command:
 * 1. Add an entry in the table below.  Keep it sorted on the shortest
 *    version of the command name that works.  If it doesn't start with a
 *    lower case letter, add it at the end.
 * 2. Run "make cmdidxs" to re-generate ex_cmdidxs.h.
 * 3. Add a "case: CMD_xxx" in the big switch in ex_docmd.c.
 * 4. Add an entry in the index for Ex commands at ":help ex-cmd-index".
 * 5. Add documentation in ../doc/xxx.txt.  Add a tag for both the short and
 *    long name of the command.
 */

#define EX_RANGE	0x001	// allow a linespecs
#define EX_BANG		0x002	// allow a ! after the command name
#define EX_EXTRA	0x004	// allow extra args after command name
#define EX_XFILE	0x008	// expand wildcards in extra part
#define EX_NOSPC	0x010	// no spaces allowed in the extra part
#define	EX_DFLALL	0x020	// default file range is 1,$
#define EX_WHOLEFOLD	0x040	// extend range to include whole fold also
				// when less than two numbers given
#define EX_NEEDARG	0x080	// argument required
#define EX_TRLBAR	0x100	// check for trailing vertical bar
#define EX_REGSTR	0x200	// allow "x for register designation
#define EX_COUNT	0x400	// allow count in argument, after command
#define EX_NOTRLCOM	0x800	// no trailing comment allowed
#define EX_ZEROR       0x1000	// zero line number allowed
#define EX_CTRLV       0x2000	// do not remove CTRL-V from argument
#define EX_CMDARG      0x4000	// allow "+command" argument
#define EX_BUFNAME     0x8000	// accepts buffer name
#define EX_BUFUNL     0x10000	// accepts unlisted buffer too
#define EX_ARGOPT     0x20000	// allow "++opt=val" argument
#define EX_SBOXOK     0x40000	// allowed in the sandbox
#define EX_CMDWIN     0x80000	// allowed in cmdline window; when missing
				// disallows editing another buffer when
				// curbuf_lock is set
#define EX_MODIFY    0x100000	// forbidden in non-'modifiable' buffer
#define EX_FLAGS     0x200000	// allow flags after count in argument
#define EX_RESTRICT  0x400000	// forbidden in restricted mode
#define EX_EXPAND    0x800000	// expands wildcards later

#define EX_FILES (EX_XFILE | EX_EXTRA)	// multiple extra files allowed
#define EX_FILE1 (EX_FILES | EX_NOSPC)	// 1 file, defaults to current file
#define EX_WORD1 (EX_EXTRA | EX_NOSPC)	// one extra word allowed

#ifndef DO_DECLARE_EXCMD
/*
 * values for cmd_addr_type
 */
typedef enum {
    ADDR_LINES,		 // buffer line numbers
    ADDR_WINDOWS,	 // window number
    ADDR_ARGUMENTS,	 // argument number
    ADDR_LOADED_BUFFERS, // buffer number of loaded buffer
    ADDR_BUFFERS,	 // buffer number
    ADDR_TABS,		 // tab page number
    ADDR_TABS_RELATIVE,	 // Tab page that only relative
    ADDR_QUICKFIX_VALID, // quickfix list valid entry number
    ADDR_QUICKFIX,	 // quickfix list entry number
    ADDR_UNSIGNED,	 // positive count or zero, defaults to 1
    ADDR_OTHER,		 // something else, use line number for '$', '%', etc.
    ADDR_NONE		 // no range used
} cmd_addr_T;
#endif

#ifndef DO_DECLARE_EXCMD
typedef struct exarg exarg_T;
#endif

/*
 * This array maps ex command names to command codes.
 * The order in which command names are listed below is significant --
 * ambiguous abbreviations are always resolved to be the first possible match
 * (e.g. "r" is taken to mean "read", not "rewind", because "read" comes
 * before "rewind").
 * Not supported commands are included to avoid ambiguities.
 */
#ifdef DO_DECLARE_EXCMD
# define EXCMD(a, b, c, d, e)  {(char_u *)b, c, (long_u)(d), e}

typedef void (*ex_func_T) (exarg_T *eap);

static struct cmdname
{
    char_u	*cmd_name;	// name of the command
    ex_func_T   cmd_func;	// function for this command
    long_u	cmd_argt;	// flags declared above
    cmd_addr_T	cmd_addr_type;	// flag for address type
} cmdnames[] =
#else
# define EXCMD(a, b, c, d, e)  a
enum CMD_index
#endif
{
EXCMD(CMD_append,	"append",	ex_append,
	EX_BANG|EX_RANGE|EX_ZEROR|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_abbreviate,	"abbreviate",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_abclear,	"abclear",	ex_abclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_aboveleft,	"aboveleft",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_all,		"all",		ex_all,
	EX_BANG|EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_amenu,	"amenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_anoremenu,	"anoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_args,		"args",		ex_args,
	EX_BANG|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_argadd,	"argadd",	ex_argadd,
	EX_BANG|EX_RANGE|EX_ZEROR|EX_FILES|EX_TRLBAR,
	ADDR_ARGUMENTS),
EXCMD(CMD_argdelete,	"argdelete",	ex_argdelete,
	EX_BANG|EX_RANGE|EX_FILES|EX_TRLBAR,
	ADDR_ARGUMENTS),
EXCMD(CMD_argdo,	"argdo",	ex_listdo,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_ARGUMENTS),
EXCMD(CMD_argedit,	"argedit",	ex_argedit,
	EX_BANG|EX_NEEDARG|EX_RANGE|EX_ZEROR|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_ARGUMENTS),
EXCMD(CMD_argglobal,	"argglobal",	ex_args,
	EX_BANG|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_arglocal,	"arglocal",	ex_args,
	EX_BANG|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_argument,	"argument",	ex_argument,
	EX_BANG|EX_RANGE|EX_COUNT|EX_EXTRA|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_ARGUMENTS),
EXCMD(CMD_ascii,	"ascii",	do_ascii,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_autocmd,	"autocmd",	ex_autocmd,
	EX_BANG|EX_EXTRA|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_augroup,	"augroup",	ex_autocmd,
	EX_BANG|EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_aunmenu,	"aunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_buffer,	"buffer",	ex_buffer,
	EX_BANG|EX_RANGE|EX_BUFNAME|EX_BUFUNL|EX_COUNT|EX_EXTRA|EX_CMDARG|EX_TRLBAR,
	ADDR_BUFFERS),
EXCMD(CMD_bNext,	"bNext",	ex_bprevious,
	EX_BANG|EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_ball,		"ball",		ex_buffer_all,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_badd,		"badd",		ex_edit,
	EX_NEEDARG|EX_FILE1|EX_CMDARG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_bdelete,	"bdelete",	ex_bunload,
	EX_BANG|EX_RANGE|EX_BUFNAME|EX_COUNT|EX_EXTRA|EX_TRLBAR,
	ADDR_BUFFERS),
EXCMD(CMD_behave,	"behave",	ex_behave,
	EX_BANG|EX_NEEDARG|EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_belowright,	"belowright",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_bfirst,	"bfirst",	ex_brewind,
	EX_BANG|EX_RANGE|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_blast,	"blast",	ex_blast,
	EX_BANG|EX_RANGE|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_bmodified,	"bmodified",	ex_bmodified,
	EX_BANG|EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_bnext,	"bnext",	ex_bnext,
	EX_BANG|EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_botright,	"botright",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_bprevious,	"bprevious",	ex_bprevious,
	EX_BANG|EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_brewind,	"brewind",	ex_brewind,
	EX_BANG|EX_RANGE|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_break,	"break",	ex_break,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_breakadd,	"breakadd",	ex_breakadd,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_breakdel,	"breakdel",	ex_breakdel,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_breaklist,	"breaklist",	ex_breaklist,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_browse,	"browse",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_buffers,	"buffers",	buflist_list,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_bufdo,	"bufdo",	ex_listdo,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_BUFFERS),
EXCMD(CMD_bunload,	"bunload",	ex_bunload,
	EX_BANG|EX_RANGE|EX_BUFNAME|EX_COUNT|EX_EXTRA|EX_TRLBAR,
	ADDR_LOADED_BUFFERS),
EXCMD(CMD_bwipeout,	"bwipeout",	ex_bunload,
	EX_BANG|EX_RANGE|EX_BUFNAME|EX_BUFUNL|EX_COUNT|EX_EXTRA|EX_TRLBAR,
	ADDR_BUFFERS),
EXCMD(CMD_change,	"change",	ex_change,
	EX_BANG|EX_WHOLEFOLD|EX_RANGE|EX_COUNT|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_cNext,	"cNext",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_cNfile,	"cNfile",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_cabbrev,	"cabbrev",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cabclear,	"cabclear",	ex_abclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cabove,	"cabove",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_caddbuffer,	"caddbuffer",	ex_cbuffer,
	EX_RANGE|EX_WORD1|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_caddexpr,	"caddexpr",	ex_cexpr,
	EX_NEEDARG|EX_WORD1|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_caddfile,	"caddfile",	ex_cfile,
	EX_TRLBAR|EX_FILE1,
	ADDR_NONE),
EXCMD(CMD_cafter,	"cafter",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_call,		"call",		ex_call,
	EX_RANGE|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_catch,	"catch",	ex_catch,
	EX_EXTRA|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cbuffer,	"cbuffer",	ex_cbuffer,
	EX_BANG|EX_RANGE|EX_WORD1|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_cbefore,	"cbefore",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_cbelow,	"cbelow",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_cbottom,	"cbottom",	ex_cbottom,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_cc,		"cc",		ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_QUICKFIX),
EXCMD(CMD_cclose,	"cclose",	ex_cclose,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_cd,		"cd",		ex_cd,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cdo,		"cdo",		ex_listdo,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_QUICKFIX_VALID),
EXCMD(CMD_center,	"center",	ex_align,
	EX_TRLBAR|EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_cexpr,	"cexpr",	ex_cexpr,
	EX_NEEDARG|EX_WORD1|EX_NOTRLCOM|EX_BANG,
	ADDR_NONE),
EXCMD(CMD_cfile,	"cfile",	ex_cfile,
	EX_TRLBAR|EX_FILE1|EX_BANG,
	ADDR_NONE),
EXCMD(CMD_cfdo,		"cfdo",		ex_listdo,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_QUICKFIX_VALID),
EXCMD(CMD_cfirst,	"cfirst",	ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_cgetfile,	"cgetfile",	ex_cfile,
	EX_TRLBAR|EX_FILE1,
	ADDR_NONE),
EXCMD(CMD_cgetbuffer,	"cgetbuffer",	ex_cbuffer,
	EX_RANGE|EX_WORD1|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_cgetexpr,	"cgetexpr",	ex_cexpr,
	EX_NEEDARG|EX_WORD1|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_chdir,	"chdir",	ex_cd,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_changes,	"changes",	ex_changes,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_checkpath,	"checkpath",	ex_checkpath,
	EX_TRLBAR|EX_BANG|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_checktime,	"checktime",	ex_checktime,
	EX_RANGE|EX_BUFNAME|EX_COUNT|EX_EXTRA|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_chistory,	"chistory",	qf_history,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_clist,	"clist",	qf_list,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_clast,	"clast",	ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_close,	"close",	ex_close,
	EX_BANG|EX_RANGE|EX_COUNT|EX_TRLBAR|EX_CMDWIN,
	ADDR_WINDOWS),
EXCMD(CMD_clearjumps,	"clearjumps",	ex_clearjumps,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cmap,		"cmap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cmapclear,	"cmapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cmenu,	"cmenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_cnext,	"cnext",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_cnewer,	"cnewer",	qf_age,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_cnfile,	"cnfile",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_cnoremap,	"cnoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cnoreabbrev,	"cnoreabbrev",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cnoremenu,	"cnoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_copy,		"copy",		ex_copymove,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_colder,	"colder",	qf_age,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_colorscheme,	"colorscheme",	ex_colorscheme,
	EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_command,	"command",	ex_command,
	EX_EXTRA|EX_BANG|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_comclear,	"comclear",	ex_comclear,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_compiler,	"compiler",	ex_compiler,
	EX_BANG|EX_TRLBAR|EX_WORD1|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_continue,	"continue",	ex_continue,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_confirm,	"confirm",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_const,	"const",	ex_const,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_copen,	"copen",	ex_copen,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_cprevious,	"cprevious",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_cpfile,	"cpfile",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_OTHER),
EXCMD(CMD_cquit,	"cquit",	ex_cquit,
	EX_RANGE|EX_COUNT|EX_ZEROR|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_crewind,	"crewind",	ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_cscope,	"cscope",	ex_cscope,
	EX_EXTRA|EX_NOTRLCOM|EX_XFILE,
	ADDR_NONE),
EXCMD(CMD_cstag,	"cstag",	ex_cstag,
	EX_BANG|EX_TRLBAR|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_cunmap,	"cunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cunabbrev,	"cunabbrev",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cunmenu,	"cunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_cwindow,	"cwindow",	ex_cwindow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_delete,	"delete",	ex_operators,
	EX_RANGE|EX_WHOLEFOLD|EX_REGSTR|EX_COUNT|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_delmarks,	"delmarks",	ex_delmarks,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_debug,	"debug",	ex_debug,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_debuggreedy,	"debuggreedy",	ex_debuggreedy,
	EX_RANGE|EX_ZEROR|EX_TRLBAR|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_def,		"def",		ex_function,
	EX_EXTRA|EX_BANG|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_delcommand,	"delcommand",	ex_delcommand,
	EX_NEEDARG|EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_delfunction,	"delfunction",	ex_delfunction,
	EX_BANG|EX_NEEDARG|EX_WORD1|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_display,	"display",	ex_display,
	EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_diffupdate,	"diffupdate",	ex_diffupdate,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_diffget,	"diffget",	ex_diffgetput,
	EX_RANGE|EX_EXTRA|EX_TRLBAR|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_diffoff,	"diffoff",	ex_diffoff,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_diffpatch,	"diffpatch",	ex_diffpatch,
	EX_EXTRA|EX_FILE1|EX_TRLBAR|EX_MODIFY,
	ADDR_NONE),
EXCMD(CMD_diffput,	"diffput",	ex_diffgetput,
	EX_RANGE|EX_EXTRA|EX_TRLBAR,
	ADDR_LINES),
EXCMD(CMD_diffsplit,	"diffsplit",	ex_diffsplit,
	EX_EXTRA|EX_FILE1|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_diffthis,	"diffthis",	ex_diffthis,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_digraphs,	"digraphs",	ex_digraphs,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_disassemble,	"disassemble",	ex_disassemble,
	EX_EXTRA|EX_NEEDARG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_djump,	"djump",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA,
	ADDR_LINES),
EXCMD(CMD_dlist,	"dlist",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_doautocmd,	"doautocmd",	ex_doautocmd,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_doautoall,	"doautoall",	ex_doautoall,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_drop,		"drop",		ex_drop,
	EX_FILES|EX_CMDARG|EX_NEEDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_dsearch,	"dsearch",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_dsplit,	"dsplit",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA,
	ADDR_LINES),
EXCMD(CMD_edit,		"edit",		ex_edit,
	EX_BANG|EX_FILE1|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_earlier,	"earlier",	ex_later,
	EX_TRLBAR|EX_EXTRA|EX_NOSPC|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_echo,		"echo",		ex_echo,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_echoerr,	"echoerr",	ex_execute,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_echohl,	"echohl",	ex_echohl,
	EX_EXTRA|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_echomsg,	"echomsg",	ex_execute,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_echon,	"echon",	ex_echo,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_else,		"else",		ex_else,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_elseif,	"elseif",	ex_else,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_emenu,	"emenu",	ex_emenu,
	EX_NEEDARG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_RANGE|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_endif,	"endif",	ex_endif,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_enddef,	"enddef",	ex_endfunction,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_endfunction,	"endfunction",	ex_endfunction,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_endfor,	"endfor",	ex_endwhile,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_endtry,	"endtry",	ex_endtry,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_endwhile,	"endwhile",	ex_endwhile,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_enew,		"enew",		ex_edit,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_eval,		"eval",		ex_eval,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_ex,		"ex",		ex_edit,
	EX_BANG|EX_FILE1|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_execute,	"execute",	ex_execute,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_exit,		"exit",		ex_exit,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_FILE1|EX_ARGOPT|EX_DFLALL|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_export,	"export",	ex_export,
	EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_exusage,	"exusage",	ex_exusage,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_file,		"file",		ex_file,
	EX_RANGE|EX_ZEROR|EX_BANG|EX_FILE1|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_files,	"files",	buflist_list,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_filetype,	"filetype",	ex_filetype,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_filter,	"filter",	ex_wrongmodifier,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_find,		"find",		ex_find,
	EX_RANGE|EX_BANG|EX_FILE1|EX_CMDARG|EX_ARGOPT|EX_TRLBAR|EX_NEEDARG,
	ADDR_OTHER),
EXCMD(CMD_finally,	"finally",	ex_finally,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_finish,	"finish",	ex_finish,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_first,	"first",	ex_rewind,
	EX_EXTRA|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_fixdel,	"fixdel",	do_fixdel,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_fold,		"fold",		ex_fold,
	EX_RANGE|EX_WHOLEFOLD|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_foldclose,	"foldclose",	ex_foldopen,
	EX_RANGE|EX_BANG|EX_WHOLEFOLD|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_folddoopen,	"folddoopen",	ex_folddo,
	EX_RANGE|EX_DFLALL|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_LINES),
EXCMD(CMD_folddoclosed,	"folddoclosed",	ex_folddo,
	EX_RANGE|EX_DFLALL|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_LINES),
EXCMD(CMD_foldopen,	"foldopen",	ex_foldopen,
	EX_RANGE|EX_BANG|EX_WHOLEFOLD|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_for,		"for",		ex_while,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_function,	"function",	ex_function,
	EX_EXTRA|EX_BANG|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_global,	"global",	ex_global,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_EXTRA|EX_DFLALL|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_goto,		"goto",		ex_goto,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_grep,		"grep",		ex_make,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_grepadd,	"grepadd",	ex_make,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_gui,		"gui",		ex_gui,
	EX_BANG|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_gvim,		"gvim",		ex_gui,
	EX_BANG|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_help,		"help",		ex_help,
	EX_BANG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_helpclose,	"helpclose",	ex_helpclose,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_helpfind,	"helpfind",	ex_helpfind,
	EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_helpgrep,	"helpgrep",	ex_helpgrep,
	EX_EXTRA|EX_NOTRLCOM|EX_NEEDARG,
	ADDR_NONE),
EXCMD(CMD_helptags,	"helptags",	ex_helptags,
	EX_NEEDARG|EX_FILES|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_hardcopy,	"hardcopy",	ex_hardcopy,
	EX_RANGE|EX_COUNT|EX_EXTRA|EX_EXPAND|EX_TRLBAR|EX_DFLALL|EX_BANG,
	ADDR_LINES),
EXCMD(CMD_highlight,	"highlight",	ex_highlight,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_hide,		"hide",		ex_hide,
	EX_BANG|EX_RANGE|EX_COUNT|EX_EXTRA|EX_TRLBAR,
	ADDR_WINDOWS),
EXCMD(CMD_history,	"history",	ex_history,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_insert,	"insert",	ex_append,
	EX_BANG|EX_RANGE|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_iabbrev,	"iabbrev",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_iabclear,	"iabclear",	ex_abclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_if,		"if",		ex_if,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_ijump,	"ijump",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA,
	ADDR_LINES),
EXCMD(CMD_ilist,	"ilist",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_imap,		"imap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_imapclear,	"imapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_imenu,	"imenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_import,	"import",	ex_import,
	EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_inoremap,	"inoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_inoreabbrev,	"inoreabbrev",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_inoremenu,	"inoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_intro,	"intro",	ex_intro,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_isearch,	"isearch",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_isplit,	"isplit",	ex_findpat,
	EX_BANG|EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_EXTRA,
	ADDR_LINES),
EXCMD(CMD_iunmap,	"iunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_iunabbrev,	"iunabbrev",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_iunmenu,	"iunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_join,		"join",		ex_join,
	EX_BANG|EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_jumps,	"jumps",	ex_jumps,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_k,		"k",		ex_mark,
	EX_RANGE|EX_WORD1|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_keepmarks,	"keepmarks",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_keepjumps,	"keepjumps",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_keeppatterns,	"keeppatterns",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_keepalt,	"keepalt",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_list,		"list",		ex_print,
	EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_lNext,	"lNext",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_lNfile,	"lNfile",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_last,		"last",		ex_last,
	EX_EXTRA|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_labove,	"labove",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_language,	"language",	ex_language,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_laddexpr,	"laddexpr",	ex_cexpr,
	EX_NEEDARG|EX_WORD1|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_laddbuffer,	"laddbuffer",	ex_cbuffer,
	EX_RANGE|EX_WORD1|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_laddfile,	"laddfile",	ex_cfile,
	EX_TRLBAR|EX_FILE1,
	ADDR_NONE),
EXCMD(CMD_lafter,	"lafter",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_later,	"later",	ex_later,
	EX_TRLBAR|EX_EXTRA|EX_NOSPC|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lbuffer,	"lbuffer",	ex_cbuffer,
	EX_BANG|EX_RANGE|EX_WORD1|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_lbefore,	"lbefore",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_lbelow,	"lbelow",	ex_cbelow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_lbottom,	"lbottom",	ex_cbottom,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_lcd,		"lcd",		ex_cd,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lchdir,	"lchdir",	ex_cd,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lclose,	"lclose",	ex_cclose,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_lcscope,	"lcscope",	ex_cscope,
	EX_EXTRA|EX_NOTRLCOM|EX_XFILE,
	ADDR_NONE),
EXCMD(CMD_ldo,		"ldo",		ex_listdo,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_QUICKFIX_VALID),
EXCMD(CMD_left,		"left",		ex_align,
	EX_TRLBAR|EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_leftabove,	"leftabove",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_let,		"let",		ex_let,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lexpr,	"lexpr",	ex_cexpr,
	EX_NEEDARG|EX_WORD1|EX_NOTRLCOM|EX_BANG,
	ADDR_NONE),
EXCMD(CMD_lfile,	"lfile",	ex_cfile,
	EX_TRLBAR|EX_FILE1|EX_BANG,
	ADDR_NONE),
EXCMD(CMD_lfdo,		"lfdo",		ex_listdo,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_QUICKFIX_VALID),
EXCMD(CMD_lfirst,	"lfirst",	ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_lgetfile,	"lgetfile",	ex_cfile,
	EX_TRLBAR|EX_FILE1,
	ADDR_NONE),
EXCMD(CMD_lgetbuffer,	"lgetbuffer",	ex_cbuffer,
	EX_RANGE|EX_WORD1|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_lgetexpr,	"lgetexpr",	ex_cexpr,
	EX_NEEDARG|EX_WORD1|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_lgrep,	"lgrep",	ex_make,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_lgrepadd,	"lgrepadd",	ex_make,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_lhelpgrep,	"lhelpgrep",	ex_helpgrep,
	EX_EXTRA|EX_NOTRLCOM|EX_NEEDARG,
	ADDR_NONE),
EXCMD(CMD_lhistory,	"lhistory",	qf_history,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_ll,		"ll",		ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_QUICKFIX),
EXCMD(CMD_llast,	"llast",	ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_llist,	"llist",	qf_list,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lmap,		"lmap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lmapclear,	"lmapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lmake,	"lmake",	ex_make,
	EX_BANG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_NONE),
EXCMD(CMD_lnoremap,	"lnoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lnext,	"lnext",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_lnewer,	"lnewer",	qf_age,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_lnfile,	"lnfile",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_loadview,	"loadview",	ex_loadview,
	EX_FILE1|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_loadkeymap,	"loadkeymap",	ex_loadkeymap,
	EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lockmarks,	"lockmarks",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_lockvar,	"lockvar",	ex_lockvar,
	EX_BANG|EX_EXTRA|EX_NEEDARG|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lolder,	"lolder",	qf_age,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_UNSIGNED),
EXCMD(CMD_lopen,	"lopen",	ex_copen,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_lprevious,	"lprevious",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_lpfile,	"lpfile",	ex_cnext,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_OTHER),
EXCMD(CMD_lrewind,	"lrewind",	ex_cc,
	EX_RANGE|EX_COUNT|EX_TRLBAR|EX_BANG,
	ADDR_UNSIGNED),
EXCMD(CMD_ltag,		"ltag",	ex_tag,
	EX_TRLBAR|EX_BANG|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_lunmap,	"lunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_lua,		"lua",		ex_lua,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_luado,	"luado",	ex_luado,
	EX_RANGE|EX_DFLALL|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_luafile,	"luafile",	ex_luafile,
	EX_RANGE|EX_FILE1|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_lvimgrep,	"lvimgrep",	ex_vimgrep,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_lvimgrepadd,	"lvimgrepadd",	ex_vimgrep,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_lwindow,	"lwindow",	ex_cwindow,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_ls,		"ls",		buflist_list,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_move,		"move",		ex_copymove,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_mark,		"mark",		ex_mark,
	EX_RANGE|EX_WORD1|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_make,		"make",		ex_make,
	EX_BANG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_NONE),
EXCMD(CMD_map,		"map",		ex_map,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_mapclear,	"mapclear",	ex_mapclear,
	EX_EXTRA|EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_marks,	"marks",	ex_marks,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_match,	"match",	ex_match,
	EX_RANGE|EX_EXTRA|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_menu,		"menu",		ex_menu,
	EX_RANGE|EX_ZEROR|EX_BANG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_menutranslate, "menutranslate", ex_menutranslate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_messages,	"messages",	ex_messages,
	EX_EXTRA|EX_TRLBAR|EX_RANGE|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_mkexrc,	"mkexrc",	ex_mkrc,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_mksession,	"mksession",	ex_mkrc,
	EX_BANG|EX_FILE1|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_mkspell,	"mkspell",	ex_mkspell,
	EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_NONE),
EXCMD(CMD_mkvimrc,	"mkvimrc",	ex_mkrc,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_mkview,	"mkview",	ex_mkrc,
	EX_BANG|EX_FILE1|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_mode,		"mode",		ex_mode,
	EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_mzscheme,	"mzscheme",	ex_mzscheme,
	EX_RANGE|EX_EXTRA|EX_DFLALL|EX_NEEDARG|EX_CMDWIN|EX_SBOXOK|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_mzfile,	"mzfile",	ex_mzfile,
	EX_RANGE|EX_FILE1|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_next,		"next",		ex_next,
	EX_RANGE|EX_BANG|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_nbkey,	"nbkey",	ex_nbkey,
	EX_EXTRA|EX_NEEDARG,
	ADDR_NONE),
EXCMD(CMD_nbclose,	"nbclose",	ex_nbclose,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_nbstart,	"nbstart",	ex_nbstart,
	EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_new,		"new",		ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_nmap,		"nmap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_nmapclear,	"nmapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_nmenu,	"nmenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_nnoremap,	"nnoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_nnoremenu,	"nnoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_noremap,	"noremap",	ex_map,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_noautocmd,	"noautocmd",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_nohlsearch,	"nohlsearch",	ex_nohlsearch,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_noreabbrev,	"noreabbrev",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_noremenu,	"noremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_BANG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_noswapfile,	"noswapfile",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_normal,	"normal",	ex_normal,
	EX_RANGE|EX_BANG|EX_EXTRA|EX_NEEDARG|EX_NOTRLCOM|EX_CTRLV|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_number,	"number",	ex_print,
	EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_nunmap,	"nunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_nunmenu,	"nunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_open,		"open",		ex_open,
	EX_RANGE|EX_BANG|EX_EXTRA,
	ADDR_LINES),
EXCMD(CMD_oldfiles,	"oldfiles",	ex_oldfiles,
	EX_BANG|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_omap,		"omap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_omapclear,	"omapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_omenu,	"omenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_only,		"only",		ex_only,
	EX_BANG|EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_WINDOWS),
EXCMD(CMD_onoremap,	"onoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_onoremenu,	"onoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_options,	"options",	ex_options,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_ounmap,	"ounmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_ounmenu,	"ounmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_ownsyntax,	"ownsyntax",	ex_ownsyntax,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_print,	"print",	ex_print,
	EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN|EX_SBOXOK,
	ADDR_LINES),
EXCMD(CMD_packadd,	"packadd",	ex_packadd,
	EX_BANG|EX_FILE1|EX_NEEDARG|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_packloadall,	"packloadall",	ex_packloadall,
	EX_BANG|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_pclose,	"pclose",	ex_pclose,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_perl,		"perl",		ex_perl,
	EX_RANGE|EX_EXTRA|EX_DFLALL|EX_NEEDARG|EX_SBOXOK|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_perldo,	"perldo",	ex_perldo,
	EX_RANGE|EX_EXTRA|EX_DFLALL|EX_NEEDARG|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_pedit,	"pedit",	ex_pedit,
	EX_BANG|EX_FILE1|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_pop,		"pop",		ex_tag,
	EX_RANGE|EX_BANG|EX_COUNT|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_popup,	"popup",	ex_popup,
	EX_NEEDARG|EX_EXTRA|EX_BANG|EX_TRLBAR|EX_NOTRLCOM|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_ppop,		"ppop",		ex_ptag,
	EX_RANGE|EX_BANG|EX_COUNT|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_preserve,	"preserve",	ex_preserve,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_previous,	"previous",	ex_previous,
	EX_EXTRA|EX_RANGE|EX_COUNT|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_promptfind,	"promptfind",	gui_mch_find_dialog,
	EX_EXTRA|EX_NOTRLCOM|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_promptrepl,	"promptrepl",	gui_mch_replace_dialog,
	EX_EXTRA|EX_NOTRLCOM|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_profile,	"profile",	ex_profile,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_profdel,	"profdel",	ex_breakdel,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_psearch,	"psearch",	ex_psearch,
	EX_BANG|EX_RANGE|EX_WHOLEFOLD|EX_DFLALL|EX_EXTRA,
	ADDR_LINES),
EXCMD(CMD_ptag,		"ptag",		ex_ptag,
	EX_RANGE|EX_BANG|EX_WORD1|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_ptNext,	"ptNext",	ex_ptag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_ptfirst,	"ptfirst",	ex_ptag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_ptjump,	"ptjump",	ex_ptag,
	EX_BANG|EX_TRLBAR|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_ptlast,	"ptlast",	ex_ptag,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_ptnext,	"ptnext",	ex_ptag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_ptprevious,	"ptprevious",	ex_ptag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_ptrewind,	"ptrewind",	ex_ptag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_ptselect,	"ptselect",	ex_ptag,
	EX_BANG|EX_TRLBAR|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_put,		"put",		ex_put,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_REGSTR|EX_TRLBAR|EX_ZEROR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_pwd,		"pwd",		ex_pwd,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_python,	"python",	ex_python,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_pydo,		"pydo",		ex_pydo,
	EX_RANGE|EX_DFLALL|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_pyfile,	"pyfile",	ex_pyfile,
	EX_RANGE|EX_FILE1|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_py3,		"py3",		ex_py3,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_py3do,	"py3do",	ex_py3do,
	EX_RANGE|EX_DFLALL|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_python3,	"python3",	ex_py3,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_py3file,	"py3file",	ex_py3file,
	EX_RANGE|EX_FILE1|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_pyx,		"pyx",		ex_pyx,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_pyxdo,	"pyxdo",	ex_pyxdo,
	EX_RANGE|EX_DFLALL|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_pythonx,	"pythonx",	ex_pyx,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_pyxfile,	"pyxfile",	ex_pyxfile,
	EX_RANGE|EX_FILE1|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_quit,		"quit",		ex_quit,
	EX_BANG|EX_RANGE|EX_COUNT|EX_TRLBAR|EX_CMDWIN,
	ADDR_WINDOWS),
EXCMD(CMD_quitall,	"quitall",	ex_quit_all,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_qall,		"qall",		ex_quit_all,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_read,		"read",		ex_read,
	EX_BANG|EX_RANGE|EX_WHOLEFOLD|EX_FILE1|EX_ARGOPT|EX_TRLBAR|EX_ZEROR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_recover,	"recover",	ex_recover,
	EX_BANG|EX_FILE1|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_redo,		"redo",		ex_redo,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_redir,	"redir",	ex_redir,
	EX_BANG|EX_FILES|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_redraw,	"redraw",	ex_redraw,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_redrawstatus,	"redrawstatus",	ex_redrawstatus,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_redrawtabline, "redrawtabline", ex_redrawtabline,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_registers,	"registers",	ex_display,
	EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_resize,	"resize",	ex_resize,
	EX_RANGE|EX_TRLBAR|EX_WORD1|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_retab,	"retab",	ex_retab,
	EX_TRLBAR|EX_RANGE|EX_WHOLEFOLD|EX_DFLALL|EX_BANG|EX_WORD1|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_return,	"return",	ex_return,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_rewind,	"rewind",	ex_rewind,
	EX_EXTRA|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_right,	"right",	ex_align,
	EX_TRLBAR|EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_rightbelow,	"rightbelow",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_runtime,	"runtime",	ex_runtime,
	EX_BANG|EX_NEEDARG|EX_FILES|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_ruby,		"ruby",		ex_ruby,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_rubydo,	"rubydo",	ex_rubydo,
	EX_RANGE|EX_DFLALL|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_rubyfile,	"rubyfile",	ex_rubyfile,
	EX_RANGE|EX_FILE1|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_rundo,	"rundo",	ex_rundo,
	EX_NEEDARG|EX_FILE1,
	ADDR_NONE),
EXCMD(CMD_rviminfo,	"rviminfo",	ex_viminfo,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_substitute,	"substitute",	do_sub,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_sNext,	"sNext",	ex_previous,
	EX_EXTRA|EX_RANGE|EX_COUNT|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sargument,	"sargument",	ex_argument,
	EX_BANG|EX_RANGE|EX_COUNT|EX_EXTRA|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_ARGUMENTS),
EXCMD(CMD_sall,		"sall",		ex_all,
	EX_BANG|EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sandbox,	"sandbox",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_saveas,	"saveas",	ex_write,
	EX_BANG|EX_FILE1|EX_ARGOPT|EX_CMDWIN|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_sbuffer,	"sbuffer",	ex_buffer,
	EX_BANG|EX_RANGE|EX_BUFNAME|EX_BUFUNL|EX_COUNT|EX_EXTRA|EX_CMDARG|EX_TRLBAR,
	ADDR_BUFFERS),
EXCMD(CMD_sbNext,	"sbNext",	ex_bprevious,
	EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sball,	"sball",	ex_buffer_all,
	EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sbfirst,	"sbfirst",	ex_brewind,
	EX_CMDARG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_sblast,	"sblast",	ex_blast,
	EX_CMDARG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_sbmodified,	"sbmodified",	ex_bmodified,
	EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sbnext,	"sbnext",	ex_bnext,
	EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sbprevious,	"sbprevious",	ex_bprevious,
	EX_RANGE|EX_COUNT|EX_CMDARG|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sbrewind,	"sbrewind",	ex_brewind,
	EX_CMDARG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_scriptnames,	"scriptnames",	ex_scriptnames,
	EX_BANG|EX_RANGE|EX_COUNT|EX_TRLBAR|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_scriptencoding, "scriptencoding", ex_scriptencoding,
	EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_scriptversion, "scriptversion", ex_scriptversion,
	EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_scscope,	"scscope",	ex_scscope,
	EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_set,		"set",		ex_set,
	EX_BANG|EX_TRLBAR|EX_EXTRA|EX_CMDWIN|EX_SBOXOK,
	ADDR_NONE),
EXCMD(CMD_setfiletype,	"setfiletype",	ex_setfiletype,
	EX_TRLBAR|EX_EXTRA|EX_NEEDARG|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_setglobal,	"setglobal",	ex_set,
	EX_BANG|EX_TRLBAR|EX_EXTRA|EX_CMDWIN|EX_SBOXOK,
	ADDR_NONE),
EXCMD(CMD_setlocal,	"setlocal",	ex_set,
	EX_BANG|EX_TRLBAR|EX_EXTRA|EX_CMDWIN|EX_SBOXOK,
	ADDR_NONE),
EXCMD(CMD_sfind,	"sfind",	ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_CMDARG|EX_ARGOPT|EX_TRLBAR|EX_NEEDARG,
	ADDR_OTHER),
EXCMD(CMD_sfirst,	"sfirst",	ex_rewind,
	EX_EXTRA|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_shell,	"shell",	ex_shell,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_simalt,	"simalt",	ex_simalt,
	EX_NEEDARG|EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_sign,		"sign",		ex_sign,
	EX_NEEDARG|EX_RANGE|EX_EXTRA|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_silent,	"silent",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_BANG|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_sleep,	"sleep",	ex_sleep,
	EX_RANGE|EX_COUNT|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_slast,	"slast",	ex_last,
	EX_EXTRA|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_smagic,	"smagic",	ex_submagic,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_smap,		"smap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_smapclear,	"smapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_smenu,	"smenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_snext,	"snext",	ex_next,
	EX_RANGE|EX_BANG|EX_FILES|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_snomagic,	"snomagic",	ex_submagic,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_snoremap,	"snoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_snoremenu,	"snoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_source,	"source",	ex_source,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_sort,		"sort",		ex_sort,
	EX_RANGE|EX_DFLALL|EX_WHOLEFOLD|EX_BANG|EX_EXTRA|EX_NOTRLCOM|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_split,	"split",	ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_spellgood,	"spellgood",	ex_spell,
	EX_BANG|EX_RANGE|EX_NEEDARG|EX_EXTRA|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_spelldump,	"spelldump",	ex_spelldump,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_spellinfo,	"spellinfo",	ex_spellinfo,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_spellrepall,	"spellrepall",	ex_spellrepall,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_spellrare,	"spellrare",	ex_spell,
	EX_BANG|EX_RANGE|EX_NEEDARG|EX_EXTRA|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_spellundo,	"spellundo",	ex_spell,
	EX_BANG|EX_RANGE|EX_NEEDARG|EX_EXTRA|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_spellwrong,	"spellwrong",	ex_spell,
	EX_BANG|EX_RANGE|EX_NEEDARG|EX_EXTRA|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sprevious,	"sprevious",	ex_previous,
	EX_EXTRA|EX_RANGE|EX_COUNT|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_srewind,	"srewind",	ex_rewind,
	EX_EXTRA|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_stop,		"stop",		ex_stop,
	EX_TRLBAR|EX_BANG|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_stag,		"stag",		ex_stag,
	EX_RANGE|EX_BANG|EX_WORD1|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_startinsert,	"startinsert",	ex_startinsert,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_startgreplace, "startgreplace", ex_startinsert,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_startreplace,	"startreplace",	ex_startinsert,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_stopinsert,	"stopinsert",	ex_stopinsert,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_stjump,	"stjump",	ex_stag,
	EX_BANG|EX_TRLBAR|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_stselect,	"stselect",	ex_stag,
	EX_BANG|EX_TRLBAR|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_sunhide,	"sunhide",	ex_buffer_all,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_sunmap,	"sunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_sunmenu,	"sunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_suspend,	"suspend",	ex_stop,
	EX_TRLBAR|EX_BANG|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_sview,	"sview",	ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_swapname,	"swapname",	ex_swapname,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_syntax,	"syntax",	ex_syntax,
	EX_EXTRA|EX_NOTRLCOM|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_syntime,	"syntime",	ex_syntime,
	EX_NEEDARG|EX_WORD1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_syncbind,	"syncbind",	ex_syncbind,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_smile,	"smile",	ex_smile,
	EX_TRLBAR|EX_CMDWIN|EX_SBOXOK,
	ADDR_NONE),
EXCMD(CMD_t,		"t",		ex_copymove,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_tNext,	"tNext",	ex_tag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_tag,		"tag",		ex_tag,
	EX_RANGE|EX_BANG|EX_WORD1|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_tags,		"tags",		do_tags,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tab,		"tab",		ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_tabclose,	"tabclose",	ex_tabclose,
	EX_BANG|EX_RANGE|EX_ZEROR|EX_EXTRA|EX_NOSPC|EX_TRLBAR|EX_CMDWIN,
	ADDR_TABS),
EXCMD(CMD_tabdo,	"tabdo",	ex_listdo,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_TABS),
EXCMD(CMD_tabedit,	"tabedit",	ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_ZEROR|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_TABS),
EXCMD(CMD_tabfind,	"tabfind",	ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_ZEROR|EX_CMDARG|EX_ARGOPT|EX_NEEDARG|EX_TRLBAR,
	ADDR_TABS),
EXCMD(CMD_tabfirst,	"tabfirst",	ex_tabnext,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_tabmove,	"tabmove",	ex_tabmove,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_NOSPC|EX_TRLBAR,
	ADDR_TABS),
EXCMD(CMD_tablast,	"tablast",	ex_tabnext,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_tabnext,	"tabnext",	ex_tabnext,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_NOSPC|EX_TRLBAR,
	ADDR_TABS),
EXCMD(CMD_tabnew,	"tabnew",	ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_ZEROR|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_TABS),
EXCMD(CMD_tabonly,	"tabonly",	ex_tabonly,
	EX_BANG|EX_RANGE|EX_ZEROR|EX_EXTRA|EX_NOSPC|EX_TRLBAR|EX_CMDWIN,
	ADDR_TABS),
EXCMD(CMD_tabprevious,	"tabprevious",	ex_tabnext,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_NOSPC|EX_TRLBAR,
	ADDR_TABS_RELATIVE),
EXCMD(CMD_tabNext,	"tabNext",	ex_tabnext,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_NOSPC|EX_TRLBAR,
	ADDR_TABS_RELATIVE),
EXCMD(CMD_tabrewind,	"tabrewind",	ex_tabnext,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_tabs,		"tabs",		ex_tabs,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tcd,		"tcd",		ex_cd,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tchdir,	"tchdir",	ex_cd,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tcl,		"tcl",		ex_tcl,
	EX_RANGE|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_tcldo,	"tcldo",	ex_tcldo,
	EX_RANGE|EX_DFLALL|EX_EXTRA|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_tclfile,	"tclfile",	ex_tclfile,
	EX_RANGE|EX_FILE1|EX_NEEDARG|EX_CMDWIN|EX_RESTRICT,
	ADDR_LINES),
EXCMD(CMD_tearoff,	"tearoff",	ex_tearoff,
	EX_NEEDARG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_terminal,	"terminal",	ex_terminal,
	EX_RANGE|EX_BANG|EX_FILES|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_tfirst,	"tfirst",	ex_tag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_throw,	"throw",	ex_throw,
	EX_EXTRA|EX_NEEDARG|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tjump,	"tjump",	ex_tag,
	EX_BANG|EX_TRLBAR|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_tlast,	"tlast",	ex_tag,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_tlmenu,	"tlmenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_tlnoremenu,	"tlnoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_tlunmenu,	"tlunmenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_tmenu,	"tmenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_tmap,		"tmap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tmapclear,	"tmapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tnext,	"tnext",	ex_tag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_tnoremap,	"tnoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_topleft,	"topleft",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_tprevious,	"tprevious",	ex_tag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_trewind,	"trewind",	ex_tag,
	EX_RANGE|EX_BANG|EX_TRLBAR|EX_ZEROR,
	ADDR_OTHER),
EXCMD(CMD_try,		"try",		ex_try,
	EX_TRLBAR|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tselect,	"tselect",	ex_tag,
	EX_BANG|EX_TRLBAR|EX_WORD1,
	ADDR_NONE),
EXCMD(CMD_tunmenu,	"tunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_tunmap,	"tunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_undo,		"undo",		ex_undo,
	EX_RANGE|EX_COUNT|EX_ZEROR|EX_TRLBAR|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_undojoin,	"undojoin",	ex_undojoin,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_undolist,	"undolist",	ex_undolist,
	EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_unabbreviate,	"unabbreviate",	ex_abbreviate,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_unhide,	"unhide",	ex_buffer_all,
	EX_RANGE|EX_COUNT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_unlet,	"unlet",	ex_unlet,
	EX_BANG|EX_EXTRA|EX_NEEDARG|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_unlockvar,	"unlockvar",	ex_lockvar,
	EX_BANG|EX_EXTRA|EX_NEEDARG|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_unmap,	"unmap",	ex_unmap,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_unmenu,	"unmenu",	ex_menu,
	EX_BANG|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_unsilent,	"unsilent",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_update,	"update",	ex_update,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_FILE1|EX_ARGOPT|EX_DFLALL|EX_TRLBAR,
	ADDR_LINES),
EXCMD(CMD_vglobal,	"vglobal",	ex_global,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_DFLALL|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_version,	"version",	ex_version,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_verbose,	"verbose",	ex_wrongmodifier,
	EX_NEEDARG|EX_RANGE|EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_vertical,	"vertical",	ex_wrongmodifier,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM,
	ADDR_NONE),
EXCMD(CMD_visual,	"visual",	ex_edit,
	EX_BANG|EX_FILE1|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_view,		"view",		ex_edit,
	EX_BANG|EX_FILE1|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_vimgrep,	"vimgrep",	ex_vimgrep,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_vimgrepadd,	"vimgrepadd",	ex_vimgrep,
	EX_RANGE|EX_BANG|EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_TRLBAR|EX_XFILE,
	ADDR_OTHER),
EXCMD(CMD_vim9script,	"vim9script",	ex_vim9script,
	0,
	ADDR_NONE),
EXCMD(CMD_viusage,	"viusage",	ex_viusage,
	EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_vmap,		"vmap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_vmapclear,	"vmapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_vmenu,	"vmenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_vnoremap,	"vnoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_vnew,		"vnew",		ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_vnoremenu,	"vnoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_vsplit,	"vsplit",	ex_splitview,
	EX_BANG|EX_FILE1|EX_RANGE|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_vunmap,	"vunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_vunmenu,	"vunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_write,	"write",	ex_write,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_FILE1|EX_ARGOPT|EX_DFLALL|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_wNext,	"wNext",	ex_wnext,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_FILE1|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_wall,		"wall",		do_wqall,
	EX_BANG|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_while,	"while",	ex_while,
	EX_EXTRA|EX_NOTRLCOM|EX_SBOXOK|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_winsize,	"winsize",	ex_winsize,
	EX_EXTRA|EX_NEEDARG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_wincmd,	"wincmd",	ex_wincmd,
	EX_NEEDARG|EX_WORD1|EX_RANGE|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_windo,	"windo",	ex_listdo,
	EX_NEEDARG|EX_EXTRA|EX_NOTRLCOM|EX_RANGE|EX_DFLALL,
	ADDR_WINDOWS),
EXCMD(CMD_winpos,	"winpos",	ex_winpos,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_wnext,	"wnext",	ex_wnext,
	EX_RANGE|EX_BANG|EX_FILE1|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_wprevious,	"wprevious",	ex_wnext,
	EX_RANGE|EX_BANG|EX_FILE1|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_wq,		"wq",		ex_exit,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_FILE1|EX_ARGOPT|EX_DFLALL|EX_TRLBAR,
	ADDR_LINES),
EXCMD(CMD_wqall,	"wqall",	do_wqall,
	EX_BANG|EX_FILE1|EX_ARGOPT|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_wundo,	"wundo",	ex_wundo,
	EX_BANG|EX_NEEDARG|EX_FILE1,
	ADDR_NONE),
EXCMD(CMD_wviminfo,	"wviminfo",	ex_viminfo,
	EX_BANG|EX_FILE1|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_xit,		"xit",		ex_exit,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_FILE1|EX_ARGOPT|EX_DFLALL|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_xall,		"xall",		do_wqall,
	EX_BANG|EX_TRLBAR,
	ADDR_NONE),
EXCMD(CMD_xmap,		"xmap",		ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_xmapclear,	"xmapclear",	ex_mapclear,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_xmenu,	"xmenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_xnoremap,	"xnoremap",	ex_map,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_xnoremenu,	"xnoremenu",	ex_menu,
	EX_RANGE|EX_ZEROR|EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_OTHER),
EXCMD(CMD_xrestore,	"xrestore",	ex_xrestore,
	EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_xunmap,	"xunmap",	ex_unmap,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_xunmenu,	"xunmenu",	ex_menu,
	EX_EXTRA|EX_TRLBAR|EX_NOTRLCOM|EX_CTRLV|EX_CMDWIN,
	ADDR_NONE),
EXCMD(CMD_yank,		"yank",		ex_operators,
	EX_RANGE|EX_WHOLEFOLD|EX_REGSTR|EX_COUNT|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_z,		"z",		ex_z,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_FLAGS|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),

// commands that don't start with a letter
EXCMD(CMD_bang,		"!",		ex_bang,
	EX_RANGE|EX_WHOLEFOLD|EX_BANG|EX_FILES|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_pound,	"#",		ex_print,
	EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_and,		"&",		do_sub,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_star,		"*",		ex_at,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_lshift,	"<",		ex_operators,
	EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_equal,	"=",		ex_equal,
	EX_RANGE|EX_TRLBAR|EX_DFLALL|EX_FLAGS|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_rshift,	">",		ex_operators,
	EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),
EXCMD(CMD_at,		"@",		ex_at,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_tilde,	"~",		do_sub,
	EX_RANGE|EX_WHOLEFOLD|EX_EXTRA|EX_CMDWIN|EX_MODIFY,
	ADDR_LINES),

// commands that start with an uppercase letter
EXCMD(CMD_Next,		"Next",		ex_previous,
	EX_EXTRA|EX_RANGE|EX_COUNT|EX_BANG|EX_CMDARG|EX_ARGOPT|EX_TRLBAR,
	ADDR_OTHER),
EXCMD(CMD_Print,	"Print",	ex_print,
	EX_RANGE|EX_WHOLEFOLD|EX_COUNT|EX_FLAGS|EX_TRLBAR|EX_CMDWIN,
	ADDR_LINES),
EXCMD(CMD_X,		"X",		ex_X,
	EX_TRLBAR,
	ADDR_NONE),

#undef EXCMD

#ifndef DO_DECLARE_EXCMD
    CMD_SIZE,		// MUST be after all real commands!
    CMD_USER = -1,	// User-defined command
    CMD_USER_BUF = -2	// User-defined command local to buffer
#endif
};

#ifndef DO_DECLARE_EXCMD
typedef enum CMD_index cmdidx_T;

/*
 * Arguments used for an Ex command.
 */
struct exarg
{
    char_u	*arg;		// argument of the command
    char_u	*nextcmd;	// next command (NULL if none)
    char_u	*cmd;		// the name of the command (except for :make)
    char_u	**cmdlinep;	// pointer to pointer of allocated cmdline
    cmdidx_T	cmdidx;		// the index for the command
    long	argt;		// flags for the command
    int		skip;		// don't execute the command, only parse it
    int		forceit;	// TRUE if ! present
    int		addr_count;	// the number of addresses given
    linenr_T	line1;		// the first line number
    linenr_T	line2;		// the second line number or count
    cmd_addr_T	addr_type;	// type of the count/range
    int		flags;		// extra flags after count: EXFLAG_
    char_u	*do_ecmd_cmd;	// +command arg to be used in edited file
    linenr_T	do_ecmd_lnum;	// the line number in an edited file
    int		append;		// TRUE with ":w >>file" command
    int		usefilter;	// TRUE with ":w !command" and ":r!command"
    int		amount;		// number of '>' or '<' for shift command
    int		regname;	// register name (NUL if none)
    int		force_bin;	// 0, FORCE_BIN or FORCE_NOBIN
    int		read_edit;	// ++edit argument
    int		force_ff;	// ++ff= argument (first char of argument)
    int		force_enc;	// ++enc= argument (index in cmd[])
    int		bad_char;	// BAD_KEEP, BAD_DROP or replacement byte
    int		useridx;	// user command index
    char	*errmsg;	// returned error message
    char_u	*(*getline)(int, void *, int, int);
    void	*cookie;	// argument for getline()
#ifdef FEAT_EVAL
    cstack_T	*cstack;	// condition stack for ":if" etc.
#endif
    long	verbose_save;	 // saved value of p_verbose
    int		save_msg_silent; // saved value of msg_silent
    int		did_esilent;	 // how many times emsg_silent was incremented
#ifdef HAVE_SANDBOX
    int		did_sandbox;	// when TRUE did ++sandbox
#endif
};

#define FORCE_BIN 1		// ":edit ++bin file"
#define FORCE_NOBIN 2		// ":edit ++nobin file"

// Values for "flags"
#define EXFLAG_LIST	0x01	// 'l': list
#define EXFLAG_NR	0x02	// '#': number
#define EXFLAG_PRINT	0x04	// 'p': print

#endif
