/**
 * @file geis_backend_multiplexor.c
 * @brief internal GEIS backend multiplexor implementation
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_config.h"
#include "geis_backend_multiplexor.h"

#include <errno.h>
#include "geis_logging.h"
#include <stdlib.h>
#include <string.h>
#include <sys/epoll.h>


struct _GeisBackendMultiplexor
{
  int mx_fd;
  int mx_max_events_per_pump;
};

typedef struct _CallbackInfo
{
  int                         fd;
  GeisBackendFdEventCallback  callback;
  void                       *context;
} *CallbackInfo;


/**
 * Creates a new backend multiplexor.
 */
GeisBackendMultiplexor
geis_backend_multiplexor_new()
{
  GeisBackendMultiplexor mx = calloc(1, sizeof(struct _GeisBackendMultiplexor));
  if (!mx)
  {
    geis_error("failed to allocate backend multiplexor");
  }
  else
  {
    mx->mx_fd = epoll_create1(EPOLL_CLOEXEC);
    if (mx->mx_fd < 0)
    {
      geis_error("error %d creating backend multiplexor: %s",
                 errno, strerror(errno));
      free(mx);
    }
    mx->mx_max_events_per_pump = GEIS_BE_MX_DEFAULT_EVENTS_PER_PUMP;
  }

  return mx;
}


/**
 * Destroys an backend multiplexor.
 */
void
geis_backend_multiplexor_delete(GeisBackendMultiplexor mx)
{
  free(mx);
}


/**
 * Adds a file descriptor to an backend multiplexor.
 */
void
geis_backend_multiplexor_add_fd(GeisBackendMultiplexor      mx,
                                int                         fd,
                                GeisBackendFdEventCallback  callback,
                                void                       *context)
{
  int status;
  struct epoll_event ev;

  CallbackInfo callback_info = calloc(1, sizeof(struct _CallbackInfo));
  if (!callback_info)
  {
    geis_error("failed to allocate callback info structure");
    return;
  }

  callback_info->fd       = fd;
  callback_info->callback = callback;
  callback_info->context  = context;

  ev.events = EPOLLIN;
  ev.data.ptr = callback_info;
  status = epoll_ctl(mx->mx_fd, EPOLL_CTL_ADD, fd, &ev);
  if (status < 0)
  {
    geis_error("error %d multiplexing fd %d: %s",
               errno, fd, strerror(errno));
  }
}


/**
 * Removes a file descriptor from a backend multiplexor.
 *
 * @todo free callback_info
 */
void
geis_backend_multiplexor_remove_fd(GeisBackendMultiplexor mx, int fd)
{
  int status = epoll_ctl(mx->mx_fd, EPOLL_CTL_DEL, fd, NULL);
  if (status < 0)
  {
    geis_error("error %d multiplexing fd %d: %s",
               errno, fd, strerror(errno));
  }
}


/**
 * Gets the single file descriptor of the backend multiplexir itself.
 */
int
geis_backend_multiplexor_fd(GeisBackendMultiplexor mx)
{
  return mx->mx_fd;
}


/**
 * gets the maximum number of fd events per pump.
 */
int
geis_backend_multiplexor_max_events_per_pump(GeisBackendMultiplexor mx)
{
  return mx->mx_max_events_per_pump;
}


/**
 * Sets the maximum number of fd events processed per pump.
 */
void
geis_backend_multiplexor_set_max_events_per_pump(GeisBackendMultiplexor mx,
                                                 int max_events_per_pump)
{
  mx->mx_max_events_per_pump = max_events_per_pump;
}


/**
 * Dispatches events on the multiplexed file descriptors.
 */
GeisStatus
geis_backend_multiplexor_pump(GeisBackendMultiplexor mx)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  int i;
  int processed_event_count = 0;
  int available_event_count = 1;
  struct epoll_event events[4];

  while (available_event_count > 0
      && processed_event_count < mx->mx_max_events_per_pump)
  {
    available_event_count = epoll_wait(mx->mx_fd, events, 4, 0);
    if (available_event_count < 0)
    {
      geis_error("error %d in epoll_wait: %s", errno, strerror(errno));
      goto error_exit;
    }

    for (i = 0; i < available_event_count; ++i)
    {
      CallbackInfo callback_info = (CallbackInfo)events[i].data.ptr;
      callback_info->callback(callback_info->fd,
                              GEIS_BE_MX_READ_AVAILABLE,
                              callback_info->context);
      ++processed_event_count;
    }
  }
  if (available_event_count)
    status = GEIS_STATUS_CONTINUE;
  else
    status = GEIS_STATUS_SUCCESS;

error_exit:
  return status;
}

