/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Michal Hruby <michal.hruby@canonical.com>
 *
 */

using GLib;

namespace Unity.ShoppingLens
{
  public class ShoppingScope : Unity.Scope
  {
    private const string OFFERS_BASE_URI = "https://productsearch.ubuntu.com";

    private HashTable<string, string> results_details_map;
    private HashTable<string, string> global_results_details_map;
    private Soup.Session session;
    private PreviewPlayerHandler player;

    private PreferencesManager preferences = PreferencesManager.get_default ();

    public ShoppingScope ()
    {
      Object (dbus_path: "/com/canonical/unity/scope/shopping");
    }

    protected override void constructed ()
    {
      session = new Soup.SessionAsync ();
      session.ssl_use_system_ca_file = true;
      session.ssl_strict = true;
      session.user_agent = "Unity Shopping Lens " + Config.VERSION;
      session.add_feature_by_type (typeof (SoupGNOME.ProxyResolverGNOME));

      /* Listen for filter changes */
      filters_changed.connect (() => {
        queue_search_changed (SearchType.DEFAULT);
      });

      active_sources_changed.connect (() => {
        queue_search_changed (SearchType.DEFAULT);
      });

      /* No need to search if only the whitespace changes */
      generate_search_key.connect ((lens_search) => {
        return lens_search.search_string.chug ();
      });

      /* Listen for changes to the lens search entry */
      search_changed.connect ((search, search_type, cancellable) => {
        update_search_async.begin (search, search_type, cancellable);
      });

      preview_uri.connect ((uri) => {
        // FIXME: async?!
        return generate_preview_for_uri (uri);
      });

      // FIXME: the models will be re-created if this will be a remote scope
      results_details_map = new HashTable<string, string> (str_hash, str_equal);
      results_model.set_data ("details-map", results_details_map);
      global_results_details_map = new HashTable<string, string> (str_hash,
                                                                  str_equal);
      global_results_model.set_data ("details-map", global_results_details_map);

      preferences.notify["remote-content-search"].connect ((obj, pspec) => { queue_search_changed (SearchType.GLOBAL); });
    }
    
    private async void update_search_async (LensSearch search,
                                            SearchType search_type,
                                            Cancellable cancellable)
    {

      /**
       * only perform the request if the user has not disabled
       * online/commercial suggestions. That will hide the category as well.
       */
      if (preferences.remote_content_search != Unity.PreferencesManager.RemoteContent.ALL)
      {
        search.results_model.clear ();
        search.finished ();
        return;
      }

      if (search_type == SearchType.GLOBAL)
        yield perform_global_search (search, cancellable);
      else
        yield perform_search (search, cancellable);

      search.finished ();
    }

    private async void perform_global_search (LensSearch search,
                                              Cancellable cancellable)
    {
      search.results_model.clear ();
      string query_string = search.search_string.chug ();
      if (query_string == "") return;

      var uri = build_search_uri (query_string, SearchType.GLOBAL);
      try
      {
        var parser = yield get_json_reply_async (uri, cancellable);
        process_search_reply_json (parser, Category.TREAT_YOURSELF,
                                   search.results_model);
      }
      catch (IOError.CANCELLED canc_err) { /* ignore */ }
      catch (Error err)
      {
        warning ("Error: %s", err.message);
      }
    }

    private async void perform_search (LensSearch search,
                                       Cancellable cancellable)
    {
      search.results_model.clear ();
      string query_string = search.search_string.chug ();
      if (query_string == "") return;

      // we want prefix matching too
      if (!query_string.has_suffix ("*")) query_string += "*";
      var uri = build_search_uri (query_string, SearchType.DEFAULT);
      try
      {
        var parser = yield get_json_reply_async (uri, cancellable);
        process_search_reply_json (parser, Category.PURCHASE,
                                   search.results_model);
      }
      catch (Error err)
      {
        warning ("Error: %s", err.message);
      }
    }

    private async Preview? process_preview_request (string details_uri)
    {
      Json.Parser parser;
      try
      {
        parser = yield get_json_reply_async (details_uri, null);
      }
      catch (Error err)
      {
        warning ("%s", err.message);
        // FIXME: just return null when libunity handles that gracefully
        return new GenericPreview ("", "", null);
      }

      if (U1MSPreviewFactory.is_u1ms_details (parser))
      {
        var u1mspf = new U1MSPreviewFactory ();
        var preview = u1mspf.create_preview (parser);
        u1mspf.add_download_action (preview); // download will be handled by normal activation
        if (player == null)
            player = new PreviewPlayerHandler ();
        player.music_preview = preview;
        return preview;
      }
      else
      {
        var root = parser.get_root ().get_object ();
        unowned string title = root.get_string_member ("title");
        unowned string description = root.get_string_member ("description_html");

        unowned string price = null;
        if (root.has_member ("formatted_price"))
          price = root.get_string_member ("formatted_price");
        else if (root.has_member ("price"))
          price = root.get_string_member ("price");

        var img_obj = root.get_object_member ("images");
        string image_uri = extract_image_uri (img_obj, int.MAX);

        Icon? image = null;
        if (image_uri != "")
        {
          image = new FileIcon (File.new_for_uri (image_uri));
        }

        var preview = new GenericPreview (title, MarkupCleaner.html_to_pango_markup (description), image);
        var icon_dir = File.new_for_path (ICON_PATH);
        var icon = new FileIcon (icon_dir.get_child ("service-amazon.svg"));
        var buy_action = new PreviewAction ("buy", _("Buy"), icon);
        if (price != null) buy_action.extra_text = price;
        /* Leaving the activation on unity for now */
        // buy_action.activated.connect ((uri) => { });
        preview.add_action (buy_action);
        return preview;
      }
    }

    private Preview? generate_preview_for_uri (string uri)
    {
      string? details_uri =
          global_results_details_map[uri] ?? results_details_map[uri];

      if (details_uri == null)
      {
        return new GenericPreview (Path.get_basename (uri),
                                   "No data available", null);
      }

      var preview = new AsyncPreview ();
      process_preview_request.begin (details_uri, (obj, res) =>
      {
        var real_preview = process_preview_request.end (res);
        preview.preview_ready (real_preview);
      });
      return preview;
    }

    private string build_search_uri (string query, SearchType search_type)
    {
      StringBuilder s = new StringBuilder ();

      unowned string base_uri = Environment.get_variable ("OFFERS_URI");
      if (base_uri == null) base_uri = OFFERS_BASE_URI;

      s.append (base_uri);
      s.append ("/v1/search?q=");
      s.append (Uri.escape_string (query, "", false));

      //if (search_type == SearchType.GLOBAL)
      //  s.append ("&include_only_high_relevance=true");

      return s.str;
    }

    private async Json.Parser get_json_reply_async (string uri,
                                                    Cancellable? cancellable)
      throws Error
    {
      message ("Sending request: %s", uri);

      var msg = new Soup.Message ("GET", uri);
      session.queue_message (msg, (session_, msg_) =>
      {
        msg = msg_;
        get_json_reply_async.callback ();
      });

      var cancelled = false;
      ulong cancel_id = 0;
      if (cancellable != null)
      {
        cancel_id = cancellable.connect (() =>
        {
          cancelled = true;
          session.cancel_message (msg, Soup.KnownStatusCode.CANCELLED);
        });
      }

      yield;

      if (cancelled)
      {
        // we can't disconnect right away, as that would deadlock (cause
        // cancel_message doesn't return before invoking the callback)
        Idle.add (get_json_reply_async.callback);
        yield;
        cancellable.disconnect (cancel_id);
        throw new IOError.CANCELLED ("Cancelled");
      }

      if (cancellable != null)
      {
        // clean up
        cancellable.disconnect (cancel_id);
      }

      if (msg.status_code < 100)
      {
        throw new IOError.FAILED ("Request failed with error %u", msg.status_code);
      }
      else if (msg.status_code != 200)
      {
        warning ("Request returned status code %u", msg.status_code);
      }

      if (msg.response_body.data == null)
        throw new IOError.FAILED ("Request didn't return any content");

      var parser = new Json.Parser ();
      parser.load_from_data ((string) msg.response_body.data);

      return parser;
    }

    private HashTable<string, string> get_image_uri_dict (Json.Object image_obj)
    {
      var dict = new HashTable<string, string> (str_hash, str_equal);

      foreach (unowned string dimensions in image_obj.get_members ())
      {
        int width, height;
        int res = dimensions.scanf ("%dx%d", out width, out height);
        if (res != 2) continue;
        dict[dimensions] =
          image_obj.get_array_member (dimensions).get_string_element (0);
      }

      return dict;
    }

    private List<unowned string> get_sorted_keys_for_dim_dict (HashTable<string, string> dict)
    {
      var list = dict.get_keys ();
      list.sort ((a_str, b_str) =>
      {
        int width1, height1, width2, height2;

        a_str.scanf ("%dx%d", out width1, out height1);
        b_str.scanf ("%dx%d", out width2, out height2);

        return width1 * height1 - width2 * height2;
      });

      return list;
    }

    /**
     * extract_image_uri:
     *
     * Returns image uri with pixel size (width * height) that is more than
     * or equal to the given pixel size.
     * In case only images with smaller pixel size are available, returns
     * the largest of those.
     */
    private string extract_image_uri (Json.Object image_obj,
                                      int pixel_size)
    {
      var dict = get_image_uri_dict (image_obj);
      var keys_list = get_sorted_keys_for_dim_dict (dict);
      if (keys_list == null) return "";

      // short-circuit evaluation
      if (pixel_size == int.MAX) return dict[keys_list.last ().data];

      foreach (unowned string dim_string in keys_list)
      {
        int width, height;
        dim_string.scanf ("%dx%d", out width, out height);
        if (width * height >= pixel_size)
        {
          return dict[dim_string];
        }
      }

      return dict[keys_list.last ().data];
    }

    private void process_search_reply_json (Json.Parser parser,
                                            Category results_category,
                                            Dee.Model results_model)
    {
      HashTable<string, string> details_map =
          results_model.get_data ("details-map");
      details_map.remove_all ();

      var root_object = parser.get_root ().get_object ();
      foreach (var r in root_object.get_array_member ("results").get_elements ())
      {
        var result = r.get_object ();

        unowned string result_uri = result.get_string_member ("web_purchase_url");
        if (result_uri == null || result_uri == "") continue;

        var image_obj = result.get_object_member ("images");
        string image_uri = extract_image_uri (image_obj, 128*128);

        unowned string price = null;
        if (result.has_member ("formatted_price"))
          price = result.get_string_member ("formatted_price");
        else if (result.has_member ("price"))
          price = result.get_string_member ("price");

        if (image_uri != "")
        {
          // TODO: what to do if we have price but no icon?
          var file = File.new_for_uri (image_uri);
          var icon = new AnnotatedIcon (new FileIcon (file));
          // FIXME: dash doesn't like empty string as ribbon
          icon.ribbon = price == null || price == "" ? " " : price; 
          // FIXME: the service doesn't expose categories yet
          // icon.category = CategoryType.BOOK;
          image_uri = icon.to_string ();
        }

        /* keep the details uri to be able to generate the preview */
        details_map[result_uri] = result.get_string_member ("details");

        results_model.append (result_uri,
                              image_uri,
                              results_category,
                              "text/html",
                              result.get_string_member ("title"),
                              "",
                              result_uri);
      }
    }
  }
}
