#!/usr/bin/env python
# setup.py - Build system for Ubuntu One Storage Protocol package
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License version 3,
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""setup.py"""

import glob
import os
import sys

import unittest

sys.path.insert(0, os.path.abspath("."))

from distutils.core import setup
from distutils.spawn import find_executable
from distutils.command import clean, build
try:
    # pylint: disable-msg=F0401,E0611
    from DistUtilsExtra.command import check
except ImportError:
    from contrib import check

from ubuntuone.storageprotocol.context import ssl_cert_location

class StorageProtocolBuild(build.build):
    """Class to build the protobuf files."""

    description = "build the protocol buffers with protobuf-compiler"

    def run(self):
        """Do the build"""
        protoc = find_executable("protoc")
        if protoc is None:
            sys.stderr.write("*** Cannot find protoc; is the protobuf-compiler "
                             "package installed?\n")
            sys.exit(-1)

        for source in glob.glob("ubuntuone/storageprotocol/*.proto"):
            protoc_command = protoc + " --python_out=. " + source
            if os.system(protoc_command) != 0:
                sys.exit(-1)

        build.build.run(self)

class StorageProtocolClean(clean.clean):
    """Class to clean up the built protobuf files."""

    description = "clean up files generated by protobuf-compiler"

    def run(self):
        """Do the clean up"""
        for source in \
                glob.glob("ubuntuone/storageprotocol/*_pb2.py"):
            os.unlink(source)

        # Call the parent class clean command
        clean.clean.run(self)

class StorageProtocolCheck(check.check):
    """Class to run some lint checks and tests."""

    description = "validate code with pylint and unit tests"

    def __load_unittest(self, relpath):
        """Load unit tests from a Python module with the given relative path."""
        assert relpath.endswith(".py"), (
            "%s does not appear to be a Python module" % relpath)
        modpath = relpath.replace(os.path.sep, ".")[:-3]
        module = __import__(modpath, None, None, [""])

        # If the module has a 'suite' or 'test_suite' function, use that
        # to load the tests.
        if hasattr(module, "suite"):
            return module.suite()
        elif hasattr(module, "test_suite"):
            return module.test_suite()
        else:
            return unittest.defaultTestLoader.loadTestsFromModule(module)

    def __collect_tests(self):
        """Return the set of unittests."""
        suite = unittest.TestSuite()

        # We don't use the dirs variable, so ignore the warning
        # pylint: disable-msg=W0612
        for root, dirs, files in os.walk("tests"):
            for file in files:
                path = os.path.join(root, file)

                if file.endswith(".py") and file.startswith("test_"):
                    suite.addTest(self.__load_unittest(path))

        return suite

    def run(self):
        """Do the checks."""
        check.check.run(self)

        print "\nrunning tests"
        from twisted.trial.reporter import TreeReporter
        from twisted.trial.runner import TrialRunner

        runner = TrialRunner(reporterFactory=TreeReporter, realTimeErrors=True)

        result = runner.run(self.__collect_tests())
        if not result.wasSuccessful():
            sys.exit(1)


setup(name='ubuntuone-storage-protocol',
      version='1.2.0',
      packages=['ubuntuone',
                'ubuntuone.storageprotocol'],
     data_files=[(ssl_cert_location,
                ['data/UbuntuOne-Go_Daddy_CA.pem',
                 'data/UbuntuOne-Go_Daddy_Class_2_CA.pem'])],

      cmdclass = {
        'build' : StorageProtocolBuild,
        'clean' : StorageProtocolClean,
        'check' : StorageProtocolCheck
        },
      namespace_packages = ['ubuntuone'],
      )
