# -*- coding: utf-8 -*-

# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Share Links Search."""

import os

from twisted.internet import defer

from ubuntuone.controlpanel.gui.tests import USER_HOME
from ubuntuone.controlpanel.gui.qt import share_links_search as gui
from ubuntuone.controlpanel.gui.qt.tests import BaseTestCase

from mock import patch

# pylint: disable=W0212


class SearchBoxTestCase(BaseTestCase):
    """Test the qt control panel."""

    class_ui = gui.SearchBox

    @defer.inlineCallbacks
    def setUp(self):
        yield super(SearchBoxTestCase, self).setUp()
        self.patch(self.ui.backend, "search_files", self.fake_search_files)
        self.files = []
        self._slot_item = None

    def fake_search_files(self, pattern):
        """Fake search_files from the backend."""
        self.files = [os.path.join(USER_HOME, 'blabla', 'iop'),
            os.path.join(USER_HOME, 'one', 'file3'),
            os.path.join(USER_HOME, 'test', 'asd'),
            os.path.join(USER_HOME, 'ubuntu', 'file1'),
            os.path.join(USER_HOME, 'ubuntu', 'file2'),
            os.path.join('other_path', 'test', 'qwe')]
        return sorted(self.files)

    def fake_slot(self, item):
        """Fake function to be called when itemSelected is emitted."""
        self._slot_item = item

    def test_initialization(self):
        """Check that the widget is build properly"""
        self.assertEqual(gui.HOME_DIR, USER_HOME)
        self.assertEqual(self.ui.current_results, [])
        self.assertEqual(self.ui.page_index, 0)
        self.assertEqual(self.ui.page_size, 20)
        self.assertTrue(self.ui.isEnabled())

    def test_key_down_pressed(self):
        """Check the proper actions are executed on key down pressed."""
        self.ui.popup.setVisible(True)
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui._load_items()
        self.ui.popup.list_widget.setCurrentRow(1)
        current = self.ui.popup.list_widget.currentRow()
        self.ui._key_down_pressed(current)
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 2)

    def test_key_down_pressed_load_more_items(self):
        """Check the proper actions are executed on key down pressed."""
        data = []

        def fake_add_items(filenames):
            """Fake add_items."""
            data.append(True)

        self.patch(self.ui.popup, "add_items", fake_add_items)
        self.ui.popup.setVisible(True)
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui.current_results = self.ui.current_results * 4
        self.ui._load_items()
        self.assertEqual(self.ui.popup.list_widget.count(), 22)
        self.ui.popup.list_widget.setCurrentRow(
            self.ui.popup.list_widget.count() - 5)
        current = self.ui.popup.list_widget.currentRow()
        self.ui._key_down_pressed(current)
        self.assertEqual(data, [True])

    def test_key_up_pressed(self):
        """Check the proper actions are executed on key up pressed."""
        self.ui.popup.setVisible(True)
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui._load_items()
        self.ui.popup.list_widget.setCurrentRow(1)
        current = self.ui.popup.list_widget.currentRow()
        self.ui._key_up_pressed(current)
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 0)

    def test_key_up_pressed_stay_in_0(self):
        """Check the proper actions are executed on key up pressed."""
        self.ui.popup.setVisible(True)
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui._load_items()
        self.ui.popup.list_widget.setCurrentRow(0)
        current = self.ui.popup.list_widget.currentRow()
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 0)
        self.ui._key_up_pressed(current)
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 0)

    def test_key_return_pressed(self):
        """Check the proper actions are executed on key return pressed."""
        self.ui.popup.setVisible(True)
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui._load_items()
        self.ui.popup.list_widget.setCurrentRow(3)
        current = self.ui.popup.list_widget.currentRow()
        self.ui.itemSelected.connect(self.fake_slot)
        self.ui._key_return_pressed(current)
        expected = unicode(self._slot_item).replace('~', USER_HOME)
        self.assertEqual(expected, self.files[1])

    def test_mouse_click_pressed(self):
        """Check the proper actions are executed when click is pressed."""
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui.popup.setVisible(True)
        self.ui._load_items()
        self.ui.popup.list_widget.setCurrentRow(3)
        current = self.ui.popup.list_widget.currentItem()
        self.ui.itemSelected.connect(self.fake_slot)
        self.ui.popup.list_widget.itemPressed.emit(current)
        expected = unicode(self._slot_item).replace('~', USER_HOME)
        self.assertEqual(expected, self.files[1])

    def test_mouse_scroll(self):
        """Check that add_items is called when we reach the end of scroll."""
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui.popup.setVisible(True)
        self.ui._load_items()
        self.patch(self.ui.popup, "add_items", self._set_called)
        self.ui._scroll_fetch_more(
            self.ui.popup.list_widget.verticalScrollBar().maximum())
        expected = (([],), {})
        self.assertEqual(expected, self._called)

    def test_process_volumes_info(self):
        """Check that _process_volumes_info obtain the proper info."""
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui.popup.setVisible(True)
        self.addCleanup(self.ui.popup.hide)
        self.ui._load_items()
        expected = self.files
        expected.sort()
        self.assertEqual(self.ui.current_results, expected)
        self.assertEqual(self.ui.popup.list_widget.count(), 8)

    def test_set_selected_item(self):
        """Check the notification of the selected item."""
        self.ui.itemSelected.connect(self._set_called)
        self.patch(self.ui.popup, "isVisible", lambda: True)
        self.ui.current_results = self.ui.backend.search_files('')
        self.ui._load_items()
        self.ui.popup.list_widget.setCurrentRow(2)
        item = self.ui.popup.list_widget.currentItem()
        self.ui._set_selected_item(item)
        self.assertEqual(self._called,
                ((os.path.join(USER_HOME, 'blabla', 'iop'),), {}))
        self.assertEqual(self.ui.text(), '')


class SearchingTestCase(BaseTestCase):
    """Set up patches used by subclasses."""
    class_ui = gui.SearchBox

    @defer.inlineCallbacks
    def setUp(self):
        yield super(SearchingTestCase, self).setUp()

        self.search_files_started_q = defer.DeferredQueue()
        self.search_files_done_q = defer.DeferredQueue()

        # patch here instead of using class decorators because the
        # decorators send unexpected kwargs to superclass methods.
        @defer.inlineCallbacks
        def fake_search_files(query):
            """A fake that waits for a signal."""
            self.search_files_started_q.put(query)
            rv = yield self.search_files_done_q.get()
            defer.returnValue(rv)

        self.search_files_patch = patch.object(self.ui.backend,
                                               'search_files',
                                               new=fake_search_files)
        self.mock_search_files = self.search_files_patch.start()
        self.addCleanup(self.search_files_patch.stop)

        self.text_patch = patch.object(self.ui, 'text')
        self.mock_text = self.text_patch.start()
        self.addCleanup(self.text_patch.stop)

        self.load_items_patch = patch.object(self.ui, '_load_items')
        self.mock_load_items = self.load_items_patch.start()
        self.addCleanup(self.load_items_patch.stop)


class DoSearchTestCase(SearchingTestCase):
    """A subclass so that MultipleSearchingTestCase doesn't also call these."""
    @defer.inlineCallbacks
    def test_do_search_text_same(self):
        """If searchbox text same after search_files call, call load_items."""
        self.mock_text.return_value = 'query'
        d = self.ui._do_search()
        query = yield self.search_files_started_q.get()
        self.assertEqual(query, 'query')
        self.search_files_done_q.put([])
        yield d
        self.mock_load_items.assert_called_once_with()

    @defer.inlineCallbacks
    def test_do_search_text_changed(self):
        """If searchbox text changed after search_files call, bail."""
        self.mock_text.return_value = 'query'
        d = self.ui._do_search()
        query = yield self.search_files_started_q.get()
        self.assertEqual(query, 'query')
        self.mock_text.return_value = 'new-query'
        self.search_files_done_q.put([])
        yield d
        self.assertFalse(self.mock_load_items.called)


class MultipleSearchingTestCase(SearchingTestCase):
    """Test multiple fast calls to handle_text."""

    @defer.inlineCallbacks
    def setUp(self):
        # do all this before calling super, because ui._do_search is
        # connected to a Qt signal in the __init__ of the ui class,
        # and we need to connect our patched version:
        self.do_search_ended = defer.DeferredQueue()
        # save unbound function because self.ui won't exist yet:
        self.orig_do_search = gui.SearchBox._do_search

        @defer.inlineCallbacks
        def do_search_later():
            # call unbound function with now-existing self.ui:
            yield self.orig_do_search(self.ui)
            self.do_search_ended.put("done")

        self.do_search_patch = patch.object(gui.SearchBox, '_do_search')
        self.do_search_mock = self.do_search_patch.start()
        self.do_search_mock.side_effect = do_search_later
        self.addCleanup(self.do_search_patch.stop)

        yield super(MultipleSearchingTestCase, self).setUp()

    @defer.inlineCallbacks
    def test_multiple_searches_while_waiting(self):
        """Only call load_items once despite multiple quick text changes."""

        # This test checks a case not covered earlier, where
        # text changes once, then the text changes again, after
        # _do_search is called but before search_files has returned.

        # _do_search should be called twice, and search_files should
        # be called twice, but _load_items should only be called once,
        # by the last call to _do_search.

        # call once with original query
        self.mock_text.return_value = 'query'
        self.ui.handle_text_changed('query')

        # wait for the first delayed do_search call to call
        # search_files and check its arg for good measure
        search_files_query = yield self.search_files_started_q.get()
        self.assertEqual('query', search_files_query)

        # first call to search_files is paused waiting for
        # search_files_done_q, simulating a long IPC call.
        # the delayed call to do_search is no longer active.

        # while we wait for search_files, the user changes the
        # query, scheduling a new delayed call to do_search:
        self.mock_text.return_value = 'query2'
        self.ui.handle_text_changed('query2')

        # release both calls to search_files:
        self.search_files_done_q.put(['result1'])
        self.search_files_done_q.put(['result2'])

        # wait for first delayed call to do_search to finish:
        yield self.do_search_ended.get()

        # check that the second call to search_files got the right
        # text:
        search_files_query = yield self.search_files_started_q.get()
        self.assertEqual('query2', search_files_query)
        # wait for second do_search to end
        yield self.do_search_ended.get()

        # check that _load_items is only called once, and that it's
        # using only the results from the last call to search_files:
        self.mock_load_items.assert_called_once_with()
        self.assertEqual(['result2'], self.ui.current_results)


class FileItemTestCase(BaseTestCase):
    """Test the File Item."""

    class_ui = gui.FileItem
    file_path = os.path.join(USER_HOME, 'ubuntu', 'file1.txt')
    kwargs = {'file_path': file_path}

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FileItemTestCase, self).setUp()
        gui.HOME_DIR = '/path/'

    def test_default(self):
        """Check the default style of the item"""
        name = os.path.basename(self.file_path)
        style = self.ui.text_style.format(name, self.file_path,
            '#333333', 'grey')
        self.assertEqual(self.ui.text(), style)

    def test_selected(self):
        """Check the default style of the item"""
        self.ui.set_selected()
        name = os.path.basename(self.file_path)
        style = self.ui.text_style.format(name, self.file_path,
            'white', 'white')
        self.assertEqual(self.ui.text(), style)

    def test_not_selected(self):
        """Check the default style of the item"""
        self.ui.set_not_selected()
        name = os.path.basename(self.file_path)
        style = self.ui.text_style.format(name, self.file_path,
            '#333333', 'grey')
        self.assertEqual(self.ui.text(), style)

    def test_reduce_path(self):
        """Check the default style of the item"""
        gui.HOME_DIR = USER_HOME
        result = self.ui.reduced_path(self.file_path)
        expected = os.path.join('~', 'ubuntu', 'file1.txt')
        self.assertEqual(expected, result)


class FilesPopupTestCase(BaseTestCase):

    """FilesPopup tests."""

    class_ui = gui.FilesPopup
    text_style = (u"<span style='color: {2};'>{0}</span><br>"
            "<span style='font-size: 13px; color: {3};'>({1})</span>")

    def test_load_items(self):
        """Tests that the items are loaded properly."""
        items = [
            '/home/tester/file1',
            '/home/tester/file2',
            '/home/tester/file3',
        ]

        self.assertEqual(self.ui.list_widget.count(), 0)
        self.ui.load_items(items, '')
        self.assertEqual(self.ui.list_widget.count(), 5)
        # Check that we erase the list on reload
        self.ui.load_items(items, '')
        self.assertEqual(self.ui.list_widget.count(), 5)

    def test_add_items(self):
        """Tests that the items are loaded properly."""
        items = [
            '/home/tester/file1',
            '/home/tester/file2',
            '/home/tester/file3',
        ]

        self.assertEqual(self.ui.list_widget.count(), 0)
        self.ui.load_items(items, '')
        self.assertEqual(self.ui.list_widget.count(), 5)
        self.ui.add_items(items)
        self.assertEqual(self.ui.list_widget.count(), 8)

    def test_repaint_items(self):
        """Check the style of the items change acording to the selection."""
        items = [
            '/home/tester/file1',
            '/home/tester/file2',
            '/home/tester/file3',
        ]

        self.ui.load_items(items, '')
        self.ui.list_widget.setCurrentRow(2)
        current = self.ui.list_widget.item(2)
        widget = self.ui.list_widget.itemWidget(current)
        next_ = self.ui.list_widget.item(3)
        widget2 = self.ui.list_widget.itemWidget(next_)
        name = os.path.basename('/home/tester/file1')
        style = self.text_style.format(name, '/home/tester/file1',
            'white', 'white')
        name2 = os.path.basename('/home/tester/file2')
        style2 = self.text_style.format(name2, '/home/tester/file2',
            '#333333', 'grey')
        self.assertEqual(widget.text(), style)
        self.assertEqual(widget2.text(), style2)

        self.ui.list_widget.setCurrentRow(3)
        current = self.ui.list_widget.item(3)
        widget = self.ui.list_widget.itemWidget(current)
        previous = self.ui.list_widget.item(2)
        widget2 = self.ui.list_widget.itemWidget(previous)
        name = os.path.basename('/home/tester/file2')
        style = self.text_style.format(name, '/home/tester/file2',
            'white', 'white')
        name2 = os.path.basename('/home/tester/file1')
        style2 = self.text_style.format(name2, '/home/tester/file1',
            '#333333', 'grey')
        self.assertEqual(widget.text(), style)
        self.assertEqual(widget2.text(), style2)

    def test_first_list_item(self):
        """Check that the first item in the popup has the proper string."""
        self.ui.load_items([], 'file')
        current = self.ui.list_widget.item(0)
        text = current.text()
        expected = gui.SEARCH_FOR % 'file'
        self.assertEqual(expected, text)

    def test_second_list_item_for_empty_list(self):
        """Check that the second item in the popup has the proper string."""
        self.ui.load_items([], 'file')
        current = self.ui.list_widget.item(1)
        text = current.text()
        self.assertEqual(gui.NO_RESULTS, text)

    def test_second_list_item(self):
        """Check that the second item in the popup has the proper string."""
        items = [
            '/home/tester/file1',
            '/home/tester/file2',
            '/home/tester/file3',
        ]

        self.ui.load_items(items, 'file')
        current = self.ui.list_widget.item(1)
        text = current.text()
        self.assertEqual(gui.SHARE_THESE_FILES, text)
