# -*- coding: utf-8 -*-

# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Share Links Search."""

import os

from dirspec import utils
from twisted.internet import defer

from ubuntuone.controlpanel.gui.tests import USER_HOME
from ubuntuone.controlpanel.gui.qt import share_links_search as gui
from ubuntuone.controlpanel.gui.qt.tests import BaseTestCase


# pylint: disable=W0212


class SearchBoxTestCase(BaseTestCase):
    """Test the qt control panel."""

    class_ui = gui.SearchBox

    @defer.inlineCallbacks
    def setUp(self):
        yield super(SearchBoxTestCase, self).setUp()

        self.patch(utils, "user_home", USER_HOME)
        self.patch(self.ui._thread_explore, "get_folder_info",
            self.fake_get_folder_info)
        self.patch(self.ui._thread_explore, "start",
            lambda *args, **kwargs: self.ui._thread_explore.run())
        self.folder_info = {
            'folder1': [
                os.path.join(USER_HOME, 'ubuntu', 'file1'),
                os.path.join(USER_HOME, 'ubuntu', 'file2'),
                os.path.join(USER_HOME, 'one', 'file3'),
            ],
            'folder2': [
                os.path.join(USER_HOME, 'test', 'asd'),
                os.path.join('other_path', 'test', 'qwe'),
                os.path.join(USER_HOME, 'blabla', 'iop'),
            ]
        }
        self._slot_item = None

    def fake_slot(self, item):
        """Fake function to be called when itemSelected is emitted."""
        self._slot_item = item

    def fake_get_folder_info(self, folder):
        """Fake get_folder_info."""
        return self.folder_info.get(folder, [])

    def test_initialization(self):
        """Check that the widget is build properly"""
        self.assertEqual(self.ui.home_dir, USER_HOME)
        self.assertEqual(self.ui._thread_explore.u1_files, [])
        self.assertEqual(self.ui.temp_u1_files, [])
        self.assertEqual(self.ui.items_per_page, 0)
        self.assertEqual(self.ui.items_step, 20)
        self.assertEqual(self.ui.prefix, '')
        self.assertFalse(self.ui.isEnabled())

    def test_folder_content_obtained_enables(self):
        """Check that _folder_content_obtained enables the widget."""
        self.ui._folder_content_obtained()
        self.assertTrue(self.ui.isEnabled())

    def test_key_down_pressed(self):
        """Check the proper actions are executed on key down pressed."""
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.popup.list_widget.setCurrentRow(1)
        current = self.ui.popup.list_widget.currentRow()
        self.ui._key_down_pressed(current)
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 2)

    def test_key_down_pressed_load_more_items(self):
        """Check the proper actions are executed on key down pressed."""
        data = []

        def fake_fetch_more(filenames):
            """Fake fetch_more."""
            data.append(True)

        self.patch(self.ui.popup, "fetch_more", fake_fetch_more)
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2),
            (0, 0, data1), (0, 0, data2), (0, 0, data1), (0, 0, data2),
            (0, 0, data1), (0, 0, data2), (0, 0, data1), (0, 0, data2),
            (0, 0, data1), (0, 0, data2), (0, 0, data1), (0, 0, data2)])
        self.assertEqual(self.ui.popup.list_widget.count(), 20)
        self.ui.popup.list_widget.setCurrentRow(
            self.ui.popup.list_widget.count() - 5)
        current = self.ui.popup.list_widget.currentRow()
        self.ui._key_down_pressed(current)
        self.assertEqual(data, [True])

    def test_key_up_pressed(self):
        """Check the proper actions are executed on key up pressed."""
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.popup.list_widget.setCurrentRow(1)
        current = self.ui.popup.list_widget.currentRow()
        self.ui._key_up_pressed(current)
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 0)

    def test_key_up_pressed_stay_in_0(self):
        """Check the proper actions are executed on key up pressed."""
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.popup.list_widget.setCurrentRow(0)
        current = self.ui.popup.list_widget.currentRow()
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 0)
        self.ui._key_up_pressed(current)
        self.assertEqual(self.ui.popup.list_widget.currentRow(), 0)

    def test_key_return_pressed(self):
        """Check the proper actions are executed on key return pressed."""
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.popup.list_widget.setCurrentRow(1)
        current = self.ui.popup.list_widget.currentRow()
        self.ui.itemSelected.connect(self.fake_slot)
        self.ui._key_return_pressed(current)
        expected = unicode(self._slot_item).replace('~', USER_HOME)
        self.assertEqual(expected, self.folder_info['folder2'][2])

    def test_mouse_click_pressed(self):
        """Check the proper actions are executed when click is pressed."""
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.popup.list_widget.setCurrentRow(1)
        current = self.ui.popup.list_widget.currentItem()
        self.ui.itemSelected.connect(self.fake_slot)
        self.ui.popup.list_widget.itemPressed.emit(current)
        expected = unicode(self._slot_item).replace('~', USER_HOME)
        self.assertEqual(expected, self.folder_info['folder2'][2])

    def test_mouse_scroll(self):
        """Check that fetch_more is called when we reach the end of scroll."""
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.patch(self.ui.popup, "fetch_more", self._set_called)
        self.ui._scroll_fetch_more(
            self.ui.popup.list_widget.verticalScrollBar().maximum())
        expected = (([],), {})
        self.assertEqual(expected, self._called)

    def test_key_space_pressed(self):
        """Check the proper actions are executed on key space pressed."""
        data = []

        def fake_set_text(text):
            """fake setText."""
            data.append(text)

        self.patch(self.ui, "setText", fake_set_text)
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.popup.list_widget.setCurrentRow(1)

        self.ui._key_space_pressed()
        expected = ['iop']
        self.assertEqual(expected, data)

    def test_key_space_pressed_without_selection(self):
        """Check the proper actions are executed on key space pressed."""
        self.ui.popup.list_widget.setCurrentItem(None)
        self.assertFalse(self.ui._key_space_pressed())

    def test_process_volumes_info(self):
        """Check that _process_volumes_info obtain the proper info."""
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        expected = [
            os.path.join('other_path', 'test', 'qwe'),
            os.path.join('~', 'blabla', 'iop'),
            os.path.join('~', 'one', 'file3'),
            os.path.join('~', 'test', 'asd'),
            os.path.join('~', 'ubuntu', 'file1'),
            os.path.join('~', 'ubuntu', 'file2')]
        self.assertEqual(self.ui._thread_explore.u1_files, expected)
        self.assertEqual(self.ui.popup.list_widget.count(), 6)

    def test_filter(self):
        """Check the results of the filter."""
        # This tests the behavior of:
        # filter
        # _show_filter
        # _get_filtered_list
        self.patch(self.ui.popup, "isVisible", lambda: True)
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.filter('p')
        expected = [os.path.join('~', 'blabla', 'iop')]
        self.assertEqual(expected, self.ui.temp_u1_files)

        self.ui.filter('i')
        expected = [
            os.path.join('~', 'blabla', 'iop'),
            os.path.join('~', 'one', 'file3'),
            os.path.join('~', 'ubuntu', 'file1'),
            os.path.join('~', 'ubuntu', 'file2')]
        self.assertEqual(expected, self.ui.temp_u1_files)

    def test_set_selected_item(self):
        """Check the notification of the selected item."""
        self.ui.itemSelected.connect(self._set_called)
        self.patch(self.ui.popup, "isVisible", lambda: True)
        data1 = [{'path': 'folder1'}]
        data2 = [{'realpath': 'folder2'}]
        self.ui._process_volumes_info([(0, 0, data1), (0, 0, data2)])
        self.ui.popup.list_widget.setCurrentRow(0)
        item = self.ui.popup.list_widget.currentItem()
        self.ui._set_selected_item(item)
        self.assertEqual(self._called,
                ((os.path.join(u'other_path', u'test', u'qwe'),), {}))
        self.assertEqual(self.ui.text(), '')


class FileItemTestCase(BaseTestCase):
    """Test the File Item."""

    class_ui = gui.FileItem
    file_path = '/home/tester/ubuntu/file1.txt'
    kwargs = {'file_path': file_path}

    def test_default(self):
        """Check the default style of the item"""
        name = os.path.basename(self.file_path)
        style = self.ui.text_style.format(name, self.file_path,
            '#333333', 'grey')
        self.assertEqual(self.ui.text(), style)

    def test_selected(self):
        """Check the default style of the item"""
        self.ui.set_selected()
        name = os.path.basename(self.file_path)
        style = self.ui.text_style.format(name, self.file_path,
            'white', 'white')
        self.assertEqual(self.ui.text(), style)

    def test_not_selected(self):
        """Check the default style of the item"""
        self.ui.set_not_selected()
        name = os.path.basename(self.file_path)
        style = self.ui.text_style.format(name, self.file_path,
            '#333333', 'grey')
        self.assertEqual(self.ui.text(), style)


class FilesPopupTestCase(BaseTestCase):

    """FilesPopup tests."""

    class_ui = gui.FilesPopup
    text_style = (u"<span style='color: {2};'>{0}</span><br>"
            "<span style='font-size: 13px; color: {3};'>({1})</span>")

    def test_load_items(self):
        """Tests that the items are loaded properly."""
        items = [
            '/home/tester/file1',
            '/home/tester/file2',
            '/home/tester/file3',
        ]

        self.assertEqual(self.ui.list_widget.count(), 0)
        self.ui.load_items(items)
        self.assertEqual(self.ui.list_widget.count(), 3)
        # Check that we erase the list on reload
        self.ui.load_items(items)
        self.assertEqual(self.ui.list_widget.count(), 3)

    def test_fetch_more(self):
        """Tests that the items are loaded properly."""
        items = [
            '/home/tester/file1',
            '/home/tester/file2',
            '/home/tester/file3',
        ]

        self.assertEqual(self.ui.list_widget.count(), 0)
        self.ui.load_items(items)
        self.assertEqual(self.ui.list_widget.count(), 3)
        self.ui.fetch_more(items)
        self.assertEqual(self.ui.list_widget.count(), 6)

    def test_repaint_items(self):
        """Check the style of the items change acording to the selection."""
        items = [
            '/home/tester/file1',
            '/home/tester/file2',
            '/home/tester/file3',
        ]

        self.ui.load_items(items)
        current = self.ui.list_widget.item(0)
        widget = self.ui.list_widget.itemWidget(current)
        next_ = self.ui.list_widget.item(1)
        widget2 = self.ui.list_widget.itemWidget(next_)
        name = os.path.basename('/home/tester/file1')
        style = self.text_style.format(name, '/home/tester/file1',
            'white', 'white')
        name2 = os.path.basename('/home/tester/file2')
        style2 = self.text_style.format(name2, '/home/tester/file2',
            '#333333', 'grey')
        self.assertEqual(widget.text(), style)
        self.assertEqual(widget2.text(), style2)

        self.ui.list_widget.setCurrentRow(1)
        current = self.ui.list_widget.item(1)
        widget = self.ui.list_widget.itemWidget(current)
        previous = self.ui.list_widget.item(0)
        widget2 = self.ui.list_widget.itemWidget(previous)
        name = os.path.basename('/home/tester/file2')
        style = self.text_style.format(name, '/home/tester/file2',
            'white', 'white')
        name2 = os.path.basename('/home/tester/file1')
        style2 = self.text_style.format(name2, '/home/tester/file1',
            '#333333', 'grey')
        self.assertEqual(widget.text(), style)
        self.assertEqual(widget2.text(), style2)
