# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The file sync status widget."""

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel import backend, cache
from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    ERROR_COLOR,
    FILE_SYNC_CONNECT,
    FILE_SYNC_CONNECT_TOOLTIP,
    FILE_SYNC_DISABLED,
    FILE_SYNC_DISCONNECT,
    FILE_SYNC_DISCONNECT_TOOLTIP,
    FILE_SYNC_DISCONNECTED,
    FILE_SYNC_ENABLE,
    FILE_SYNC_ENABLE_TOOLTIP,
    FILE_SYNC_ERROR,
    FILE_SYNC_IDLE,
    FILE_SYNC_RESTART,
    FILE_SYNC_RESTART_TOOLTIP,
    FILE_SYNC_START,
    FILE_SYNC_START_TOOLTIP,
    FILE_SYNC_STARTING,
    FILE_SYNC_STOP,
    FILE_SYNC_STOP_TOOLTIP,
    FILE_SYNC_STOPPED,
    FILE_SYNC_SYNCING,
    LOADING,
    PLEASE_WAIT,
)
from ubuntuone.controlpanel.gui.qt import pixmap_from_name
from ubuntuone.controlpanel.gui.qt.ui import filesyncstatus_ui


logger = setup_logging('qt.filesyncstatus')


WARNING_MARKUP = '<font color="%s"><b>%%s</b></font>' % ERROR_COLOR
DISCONNECTED_ICON = 'sync_status_disconnected'
ERROR_ICON = 'sync_status_alert'
IDLE_ICON = 'sync_status_sync_done'
SYNCING_ICON = 'sync_status_syncing'

FILE_SYNC_STATUS = {
    backend.FILE_SYNC_DISABLED:
        {'msg': FILE_SYNC_DISABLED, 'action': FILE_SYNC_ENABLE,
         'backend_method': 'enable_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_ENABLE_TOOLTIP},
    backend.FILE_SYNC_DISCONNECTED:
        {'msg': FILE_SYNC_DISCONNECTED, 'action': FILE_SYNC_CONNECT,
         'backend_method': 'connect_files',
         'icon': DISCONNECTED_ICON,
         'tooltip': FILE_SYNC_CONNECT_TOOLTIP},
    backend.FILE_SYNC_ERROR:
        {'msg': WARNING_MARKUP % FILE_SYNC_ERROR, 'action': FILE_SYNC_RESTART,
         'backend_method': 'restart_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_RESTART_TOOLTIP},
    backend.FILE_SYNC_IDLE:
        {'msg': FILE_SYNC_IDLE, 'action': FILE_SYNC_DISCONNECT,
         'backend_method': 'disconnect_files',
         'icon': IDLE_ICON,
         'tooltip': FILE_SYNC_DISCONNECT_TOOLTIP},
    backend.FILE_SYNC_STARTING:
        {'msg': FILE_SYNC_STARTING, 'action': FILE_SYNC_STOP,
         'backend_method': 'stop_files',
         'icon': SYNCING_ICON,
         'tooltip': FILE_SYNC_STOP_TOOLTIP},
    backend.FILE_SYNC_STOPPED:
        {'msg': FILE_SYNC_STOPPED, 'action': FILE_SYNC_START,
         'backend_method': 'start_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_START_TOOLTIP},
    backend.FILE_SYNC_SYNCING:
        {'msg': FILE_SYNC_SYNCING, 'action': FILE_SYNC_DISCONNECT,
         'backend_method': 'disconnect_files',
         'icon': SYNCING_ICON,
         'tooltip': FILE_SYNC_DISCONNECT_TOOLTIP},
    backend.FILE_SYNC_UNKNOWN:
        {'msg': WARNING_MARKUP % FILE_SYNC_ERROR, 'action': FILE_SYNC_RESTART,
         'backend_method': 'restart_files',
         'icon': ERROR_ICON,
         'tooltip': FILE_SYNC_RESTART_TOOLTIP},
}


def icon_name_from_status(status_key):
    """Get the icon name for the status."""
    icon_name = FILE_SYNC_STATUS[status_key]['icon']
    return icon_name


class FileSyncStatus(cache.Cache, QtGui.QWidget):
    """The FileSyncStatus widget"""

    def __init__(self, *args, **kwargs):
        """Initialize the UI of the widget."""
        super(FileSyncStatus, self).__init__(*args, **kwargs)
        self.ui = filesyncstatus_ui.Ui_Form()
        self.ui.setupUi(self)

        self._backend_method = None

    # Invalid name "showEvent"
    # pylint: disable=C0103

    def showEvent(self, event):
        """Load info."""
        super(FileSyncStatus, self).showEvent(event)
        self.load()

    # pylint: enable=C0103

    @defer.inlineCallbacks
    def load(self):
        """Load specific tab info."""
        info = yield self.backend.file_sync_status()
        self.process_info(info)
        self.backend.status_changed_handler = self.process_info

    @log_call(logger.debug)
    def process_info(self, status):
        """Match status with signals."""
        try:
            status_key = status[backend.STATUS_KEY]
            data = FILE_SYNC_STATUS[status_key]
        except (KeyError, TypeError):
            logger.exception('process_info: received unknown status dict %r',
                             status)
            return

        msg = data['msg']
        if status_key in (backend.FILE_SYNC_ERROR, backend.FILE_SYNC_UNKNOWN):
            reason = status[backend.MSG_KEY]
            if reason:
                msg += ' (' + reason + ')'
        self.ui.sync_status_label.setText(msg)

        icon = data.get('icon')
        if icon is not None:
            pixmap = pixmap_from_name(icon)
            self.ui.sync_status_icon.setPixmap(pixmap)

        action = data.get('action')
        if action is not None:
            self.ui.sync_status_button.setText(action)
            self.ui.sync_status_button.setEnabled(True)
            self.ui.sync_status_button.show()
            default = (status_key == backend.FILE_SYNC_DISCONNECTED)
            self.ui.sync_status_button.setDefault(default)
        else:
            self.ui.sync_status_button.hide()

        tooltip = data.get('tooltip')
        if tooltip is not None:
            self.ui.sync_status_button.setToolTip(tooltip)

        self._backend_method = getattr(self.backend, data['backend_method'])

    @QtCore.pyqtSlot()
    def on_sync_status_button_clicked(self, *a, **kw):
        """Button was clicked, act accordingly to the label."""
        self.ui.sync_status_button.setEnabled(False)
        if self._backend_method is not None:
            self.ui.sync_status_label.setText(LOADING)
            self.ui.sync_status_button.setText(PLEASE_WAIT)
            pixmap = pixmap_from_name('sync_status_loading')
            self.ui.sync_status_icon.setPixmap(pixmap)
            self.ui.sync_status_button.setDefault(False)
            self._backend_method()
        else:
            logger.error('on_sync_status_button_clicked: backend method is '
                         'None!')
