# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the GoToWebButton widget."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui import qt, UBUNTUONE_LINK
from ubuntuone.controlpanel.gui.qt import gotoweb as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
)


class GoToWebButtonTestCase(BaseTestCase):
    """The test suite for the GoToWebButton."""

    class_ui = gui.GoToWebButton

    @defer.inlineCallbacks
    def setUp(self):
        yield super(GoToWebButtonTestCase, self).setUp()

    def test_uri_can_be_set(self):
        """The uri can be set."""
        uri = 'something'
        self.ui.uri = uri
        self.assertEqual(self.ui.uri, uri)

    def test_layout_direction(self):
        """The layout direction is RightToLeft."""
        self.assertEqual(self.ui.layoutDirection(), gui.QtCore.Qt.RightToLeft)

    def test_cursor_pointer(self):
        """The cursor is PointingHandCursor."""
        self.assertEqual(self.ui.cursor().shape(),
                         gui.QtCore.Qt.PointingHandCursor)

    def test_open_uri_when_clicked(self):
        """When clicking the button, the uri is opened."""
        self.patch(qt, 'uri_hook', self._set_called)
        self.ui.uri = 'yadda-yadda-yoo'
        self.ui.click()

        self.assertEqual(self._called, ((self.ui.uri,), {}))

    def test_do_nothing_on_clicked_if_uri_is_none(self):
        """When clicking the button, if the uri is None, do nothing."""
        self.patch(qt, 'uri_hook', self._set_called)
        self.ui.uri = None
        self.ui.click()

        self.assertEqual(self._called, False)


class SignUrlTestCase(GoToWebButtonTestCase):
    """The test suite for the sign url management."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(SignUrlTestCase, self).setUp()
        self.patch(qt, 'uri_hook', lambda url: None)
        self.patch(gui, 'sign_url', self._set_called)
        self.creds = yield self.ui.backend.get_credentials()
        assert len(self.creds) > 0

    def test_without_ubuntuone_prefix(self):
        """If given url is not an ubuntuone url, don't sign it."""
        self.ui.uri = 'bad_prefix' + UBUNTUONE_LINK
        self.ui.click()

        self.assertFalse(self._called)

    def test_with_ubuntuone_prefix(self):
        """If given url is an ubuntuone url, sign it."""
        self.ui.uri = UBUNTUONE_LINK + 'foo'
        self.ui.click()

        self.assertEqual(self._called, ((self.ui.uri, self.creds,), {}))


class SignUrlNoCredsTestCase(SignUrlTestCase):
    """The test suite for the sign url management when there are no creds."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(SignUrlNoCredsTestCase, self).setUp()
        self.patch(self.ui.backend, 'get_credentials', lambda: {})

    def test_with_ubuntuone_prefix(self):
        """If given url is an ubuntuone url, don't sign it.

        Since we have no credentials, the url should not be signed.

        """
        self.ui.uri = UBUNTUONE_LINK + 'foo'
        self.ui.click()

        self.assertFalse(self._called)
