# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
#          Natalia B. Bidart <nataliabidart@canonical.com>
#          Eric Casteleijn <eric.casteleijn@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the control panel backend DBus service."""

import dbus
import mocker

from dbus.mainloop.glib import DBusGMainLoop

from ubuntuone.controlpanel.gui.gtk import gui
from ubuntuone.devtools.testcase import DBusTestCase
from twisted.trial.unittest import TestCase


class MockWindow(object):
    """A mock backend."""

    exception = None

    def __init__(self, switch_to=None, alert=False):
        self.called = []

    def draw_attention(self):
        """Draw attention to the control panel."""
        self.called.append('draw_attention')

    def switch_to(self, panel):
        """Switch to named panel."""
        self.called.append(('switch_to', panel))


class DBusServiceMockTestCase(TestCase):
    """Tests for the main function."""

    def setUp(self):
        self.mocker = mocker.Mocker()

    def tearDown(self):
        self.mocker.restore()
        self.mocker.verify()

    def test_dbus_service_main(self):
        """The main method starts the loop and hooks up to DBus."""
        self.patch(gui, 'ControlPanelWindow', MockWindow)
        dbus_gmain_loop = self.mocker.replace(
            "dbus.mainloop.glib.DBusGMainLoop")
        register_service = self.mocker.replace(
            "ubuntuone.controlpanel.gui.gtk.gui.register_service")
        publish_service = self.mocker.replace(
            "ubuntuone.controlpanel.gui.gtk.gui.publish_service")
        main = self.mocker.replace("gtk.main")
        dbus_gmain_loop(set_as_default=True)
        loop = self.mocker.mock()
        self.mocker.result(loop)
        register_service(mocker.ANY)
        self.mocker.result(True)
        publish_service(switch_to='', alert=False)
        main()
        self.mocker.replay()
        gui.main()


class DBusServiceTestCase(DBusTestCase):
    """Test for the DBus service."""

    def _set_called(self, *args, **kwargs):
        """Keep track of function calls, useful for monkeypatching."""
        self._called = (args, kwargs)

    def setUp(self):
        """Initialize each test run."""
        super(DBusServiceTestCase, self).setUp()
        DBusGMainLoop(set_as_default=True)
        self._called = False

    def test_register_service(self):
        """The DBus service is successfully registered."""
        bus = dbus.SessionBus()
        ret = gui.register_service(bus)
        self.assertTrue(ret)
