/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010-2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 *
 */

#include "config.h"
#ifdef HAVE_GDBUS
#else
#include <dbus/dbus-glib.h>
#endif
#include "syncdaemon-config-interface.h"

G_DEFINE_TYPE(SyncdaemonConfigInterface, syncdaemon_config_interface, SYNCDAEMON_TYPE_INTERFACE)

struct _SyncdaemonConfigInterfacePrivate {
	GObject *proxy;
};

static void
syncdaemon_config_interface_finalize (GObject *object)
{
	SyncdaemonConfigInterface *interface = SYNCDAEMON_CONFIG_INTERFACE (object);

	if (interface->priv != NULL) {
		g_free (interface->priv);
	}

	G_OBJECT_CLASS (syncdaemon_config_interface_parent_class)->finalize (object);
}

static void
syncdaemon_config_interface_class_init (SyncdaemonConfigInterfaceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_config_interface_finalize;
}

static void
syncdaemon_config_interface_init (SyncdaemonConfigInterface *interface)
{
	interface->priv = g_new0 (SyncdaemonConfigInterfacePrivate, 1);

	/* Setup DBus proxy */
	interface->priv->proxy = syncdaemon_interface_setup_proxy (SYNCDAEMON_INTERFACE (interface),
								   "com.ubuntuone.SyncDaemon",
								   "/config", "com.ubuntuone.SyncDaemon.Config");
}

/**
 * syncdaemon_config_interface_new:
 */
SyncdaemonConfigInterface *
syncdaemon_config_interface_new (SyncdaemonDaemon *daemon)
{
	g_return_val_if_fail (SYNCDAEMON_IS_DAEMON (daemon), NULL);

	return g_object_new (SYNCDAEMON_TYPE_CONFIG_INTERFACE, "daemon", daemon, NULL);
}

static gboolean
get_boolean_method (SyncdaemonConfigInterface *interface, const gchar *method)
{
	gboolean enabled;
	GError *error = NULL;

	if (!dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), method, &error,
				G_TYPE_INVALID,
				G_TYPE_BOOLEAN, &enabled,
				G_TYPE_INVALID)) {
		g_warning ("Failed calling %s: %s", method, error->message);
		g_error_free (error);

		return FALSE;
	}

	return enabled;
}

static void
enable_setting_method (SyncdaemonConfigInterface *interface,
		       const gchar *enable_method,
		       const gchar *disable_method,
		       gboolean enabled)
{
	GError *error = NULL;

	if (enabled) {
		dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), enable_method, &error,
					    G_TYPE_INVALID,
					    G_TYPE_INVALID);
	} else {
		dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), disable_method, &error,
					    G_TYPE_INVALID,
					    G_TYPE_INVALID);
	}

	if (error) {
	    g_warning ("Failed calling %s: %s", enabled ? enable_method : disable_method, error->message);
		g_error_free (error);
	}
}

/**
 * syncdaemon_config_interface_get_bandwidth_throttling:
 */
gboolean
syncdaemon_config_interface_get_bandwidth_throttling (SyncdaemonConfigInterface *interface)
{
	g_return_val_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface), FALSE);

	return get_boolean_method (interface, "bandwidth_throttling_enabled");
}

/**
 * syncdaemon_config_interface_set_bandwidth_throttling:
 */
void
syncdaemon_config_interface_set_bandwidth_throttling (SyncdaemonConfigInterface *interface, gboolean enabled)
{
    	g_return_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface));

	enable_setting_method (interface, "enable_bandwidth_throttling", "disable_bandwidth_throttling", enabled);
}

/**
 * syncdaemon_config_interface_get_files_sync:
 */
gboolean
syncdaemon_config_interface_get_files_sync (SyncdaemonConfigInterface *interface)
{
	gboolean enabled;
	GError *error = NULL;

	g_return_val_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface), FALSE);

	if (!dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), "files_sync_enabled", &error,
				G_TYPE_INVALID,
				G_TYPE_BOOLEAN, &enabled,
				G_TYPE_INVALID)) {
		g_warning ("Failed calling files_sync_enabled: %s", error->message);
		g_error_free (error);

		return FALSE;
	}

	return enabled;
}

/**
 * syncdaemon_config_interface_set_files_sync:
 */
void
syncdaemon_config_interface_set_files_sync (SyncdaemonConfigInterface *interface, gboolean enabled)
{
	GError *error = NULL;

	g_return_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface));

	if (!dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), "set_files_sync_enabled", &error,
				G_TYPE_BOOLEAN, enabled,
				G_TYPE_INVALID,
				G_TYPE_INVALID)) {
		g_warning ("Failed calling set_files_sync_enabled: %s", error->message);
		g_error_free (error);
	}
}

/**
 * syncdaemon_config_interface_get_throttling_limits:
 */
void
syncdaemon_config_interface_get_throttling_limits (SyncdaemonConfigInterface *interface,
						   gint *download,
						   gint *upload)
{
	GHashTable *result;
	GError *error = NULL;

	g_return_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface));

	if (!dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), "get_throttling_limits", &error,
				G_TYPE_INVALID,
				dbus_g_type_get_map ("GHashTable",
						     G_TYPE_STRING,
						     G_TYPE_INT), &result,
				G_TYPE_INVALID)) {
		g_warning ("Failed calling get_throttling_limits: %s", error->message);
		g_error_free (error);
	} else {
		gint *download_got, *upload_got;

		download_got = g_hash_table_lookup (result, "download");
		upload_got = g_hash_table_lookup (result, "upload");

		if (download != NULL)
			*download = *download_got;
		if (upload != NULL)
			*upload = *upload_got;

		g_hash_table_destroy (result);
	}
}

/**
 * syncdaemon_config_interface_set_throttling_limits:
 */
void
syncdaemon_config_interface_set_throttling_limits (SyncdaemonConfigInterface *interface,
						   gint download,
						   gint upload)
{
	GError *error = NULL;

	g_return_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface));

	if (!dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), "set_throttling_limits", &error,
				G_TYPE_INT, download,
				G_TYPE_INT, upload,
				G_TYPE_INVALID,
				G_TYPE_INVALID)) {
		g_warning ("Failed calling set_throttling_limits: %s", error->message);
		g_error_free (error);
	}
}

/**
 * syncdaemon_config_interface_get_udf_autosubscribe:
 */
gboolean
syncdaemon_config_interface_get_udf_autosubscribe (SyncdaemonConfigInterface *interface)
{
	g_return_val_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface), FALSE);

	return get_boolean_method (interface, "udf_autosubscribe_enabled");
}

/**
 * syncdaemon_config_interface_set_bandwidth_throttling_enabled:
 */
void
syncdaemon_config_interface_set_udf_autosubscribe (SyncdaemonConfigInterface *interface, gboolean enabled)
{
      	g_return_if_fail (SYNCDAEMON_IS_CONFIG_INTERFACE (interface));

	enable_setting_method (interface, "enable_udf_autosubscribe", "disable_udf_autosubscribe", enabled);
}
