# tests.platform.linux.test_messaging
#
# Author: Eric Casteleijn <eric.casteleijn@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the messaging on linux. These tests are kind of stupid, but at
least they ensure 100% coverage and hence no silly/syntax errors.

"""


from mocker import Mocker, ANY
from twisted.trial.unittest import TestCase

from ubuntuone.platform.linux.messaging import Messaging, _server_callback

FAKE_TIME = 123
FAKE_TIME2 = 456
FAKE_COUNT = 11
FAKE_SENDER = "Mom"
FAKE_ICON = object()


def callback(indicator, message_time=None):
    """Dummy callback."""
    pass


class MessagingTestCase(TestCase):
    """Test the Messaging API."""

    def setUp(self):
        self.mocker = Mocker()

    def tearDown(self):
        self.mocker.restore()
        self.mocker.verify()

    # pylint: disable=R0913
    def _show_message_setup(self, message_time=None, message_count=None,
                            icon=None, update_count=None, real_callback=False):
        """Set up the mocker expectations for show_method."""
        indicate = self.mocker.replace("indicate")
        mock_server = self.mocker.mock()
        indicate.indicate_server_ref_default()
        self.mocker.result(mock_server)
        mock_server.connect('server-display', _server_callback)
        mock_server.set_type("message.u1")
        mock_server.set_desktop_file(
            "/usr/share/applications/ubuntuone-control-panel-gtk.desktop")
        mock_server.show()
        mock_indicator = self.mocker.mock()
        indicate.Indicator()
        self.mocker.result(mock_indicator)
        mock_indicator.set_property("subtype", "u1")
        mock_indicator.set_property("name", FAKE_SENDER)
        mock_indicator.set_property("sender", FAKE_SENDER)
        mock_indicator.connect("user-display", ANY)
        if icon is not None:
            mock_indicator.set_property_icon("icon", icon)
        if message_count is not None:
            mock_indicator.set_property("count", str(message_count))
        else:
            if message_time is None:
                mock_time = self.mocker.replace("time.time")
                mock_time()
                self.mocker.result(FAKE_TIME2)
                mock_indicator.set_property_time("time", FAKE_TIME2)
            else:
                mock_indicator.set_property_time("time", message_time)
        mock_indicator.set_property("draw-attention", "true")
        if update_count:
            mock_indicator.get_property("count")
            self.mocker.result("1200")
            mock_indicator.set_property("count", '2500')
        mock_indicator.show()
        if real_callback:
            open_volumes = self.mocker.replace(
                "ubuntuone.platform.linux.messaging.open_volumes")
            open_volumes()
            mock_indicator.set_property('draw-attention', 'false')
            mock_indicator.hide()
        self.mocker.replay()
    # pylint: enable=R0913

    def test_show_message(self):
        """On message, libnotify receives the proper calls."""
        self._show_message_setup()
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback)

    def test_show_message_with_time(self):
        """On message with time, libnotify receives the proper calls."""
        self._show_message_setup(FAKE_TIME)
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback, message_time=FAKE_TIME)

    def test_show_message_with_icon(self):
        """On message with icon, libnotify receives the proper calls."""
        self._show_message_setup(icon=FAKE_ICON)
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback, icon=FAKE_ICON)

    def test_show_message_with_count(self):
        """On message with count, libnotify receives the proper calls."""
        self._show_message_setup(message_count='1200')
        messaging = Messaging()
        messaging.show_message(FAKE_SENDER, callback, message_count=1200)

    def test_update_count(self):
        """On message count update, libnotify receives the proper calls."""
        self._show_message_setup(message_count='1200', update_count='1300')
        messaging = Messaging()
        indicator = messaging.show_message(
            FAKE_SENDER, callback, message_count=1200)
        messaging.update_count(indicator, 1300)

    def test_callback(self):
        """When the callback fires, libnotify receives the proper calls."""
        self._show_message_setup(real_callback=True)
        messaging = Messaging()
        mock_indicator = messaging.show_message(FAKE_SENDER)
        self.assertEquals(1, len(messaging.indicators))
        actual_callback = messaging.create_callback()
        actual_callback(mock_indicator)
        self.assertEquals(0, len(messaging.indicators))
